import * as VAPI from "../index.js";
import * as VScript from "vscript";
export class NetworkConfig {
    literal;
    constructor(literal) {
        this.literal = literal;
    }
    as_literal() {
        return this.literal;
    }
    // NOTE: automatically filters out any factory-resets
    filter_map(f) {
        const result = {}; // :(
        for (const k in this.literal) {
            if (this.literal[k] === "factory-reset")
                continue;
            VScript.enforce(typeof this.literal[k] === "object");
            const filtered = f(this.literal[k]);
            if (filtered !== undefined)
                result[k] = filtered;
        }
        return new NetworkConfig(result);
    }
}
export async function do_upload_config(vsocket, config) {
    const cfg_obj = config instanceof NetworkConfig ? config.as_literal() : config;
    const protocol = vsocket.protocol === "ws" ? "http" : "https";
    const maybe_login = vsocket.login;
    const rsp = await fetch(`${protocol}://${vsocket.ip}/network_config`, {
        method: "POST",
        headers: {
            "Content-Type": "application/json",
            ...(maybe_login
                ? {
                    Authorization: `Basic ${btoa(`${maybe_login.user}:${maybe_login.password}`)}`,
                }
                : {}),
        },
        body: JSON.stringify(cfg_obj),
    });
    const code = `Network reconfiguration request sent to ${protocol}://${vsocket.ip} failed with status code`;
    switch (rsp.status) {
        case 200:
        case 201:
            break;
        case 401:
            throw new Error(`${code} 401: apparently the login you are currently using does not have net-admin rights; you may have to specify a different login or contact your system administrator`);
        case 503:
            throw new Error(`${code} 503: a concurrent configuration request may currently be underway. Please try again later`);
        default:
            throw new Error(`${code} ${rsp.status}. The submitted config file was:\n${JSON.stringify(config, null, 2)}`);
    }
    return { restart_required: rsp.status === 201 };
}
async function read_vlan_config(x) {
    return {
        dhcp: await x.dhcp.read(),
        ntpd: await x.ntpd.read(),
        reverse_path_filter: await x.reverse_path_filter.read(),
        routes: await VScript.asyncMap(await x.routes.rows(), async (route) => {
            const result = {
                via: VScript.enforce_nonnull(await route.via.read()),
            };
            const maybe_dst = await route.dst.read();
            const maybe_dst_prefix = await route.dst_prefix.read();
            const maybe_weight = await route.weight.read();
            if (!!maybe_dst &&
                maybe_dst_prefix !== undefined &&
                maybe_dst_prefix !== null) {
                result.dst = { address: maybe_dst, prefix: maybe_dst_prefix };
            }
            if (maybe_weight !== null && maybe_weight !== undefined)
                result.weight = maybe_weight;
            return result;
        }),
        ip_addresses: await VScript.asyncMap(await x.ip_addresses.rows(), async (entry) => {
            const result = {
                address: VScript.enforce_nonnull(await entry.ip_address.read()),
                prefix: VScript.enforce_nonnull(await entry.prefix.read()),
            };
            return result;
        }),
    };
}
async function read_vlans(port) {
    const result = {};
    result[0] = await read_vlan_config(port.current_configuration.base);
    await VScript.asyncIter(await port.current_configuration.vlans.rows(), async (vlan) => {
        const vlan_id = await vlan.vlan_id.read();
        if (!vlan_id)
            return; // non-null but numerically zero is considered an invalid vlan, too
        result[vlan_id] = await read_vlan_config(vlan.settings);
    });
    return result;
}
async function read_syslog(port) {
    const result = [];
    await VScript.asyncIter([...port.current_syslog_configuration.syslog_servers], async (entry) => {
        const address_with_port = await entry.address.read();
        if (!address_with_port || address_with_port.trim().length === 0)
            return;
        result.push({
            address_with_port,
            protocol: await entry.protocol.read(),
            keep_alive: await entry.keep_alive.read(),
            rebind_interval: await entry.rebind_interval.read(),
            format: await entry.format.read(),
        });
    });
    return result;
}
async function download_at1130_config(network_interfaces) {
    const result = {};
    await VScript.asyncIter(await network_interfaces.ports.rows(), async (port) => {
        const ns_name = VScript.enforce_nonnull(await port.network_namespace.read());
        const p = {
            vlans: await read_vlans(port),
            syslog: await read_syslog(port),
        };
        if (await port.supports_rtp.read()) {
            // media port
            const m = {
                phy: {
                    pma_settings: VScript.enforce_nonnull(await port.pma_settings.read()),
                    port_mode: VScript.enforce_nonnull(await port.port_mode.status.read()),
                },
                ...p,
            };
            result[ns_name] = m;
        }
        else {
            result[ns_name] = p;
        }
    });
    return result;
}
async function download_at1101_config(network_interfaces) {
    const result = {};
    await VScript.asyncIter(await network_interfaces.ports.rows(), async (port) => {
        const ns_name = VScript.enforce_nonnull(await port.network_namespace.read());
        const p = {
            vlans: await read_vlans(port),
            syslog: await read_syslog(port),
        };
        result[ns_name] = p;
    });
    return result;
}
// NOTE: not sure if there should be http GET support for network configs?
// For now I'm just going to use our websocket interface
export async function assemble_config_record(network_interfaces) {
    if (network_interfaces instanceof VAPI.AT1101.NetworkInterfaces.All)
        return new NetworkConfig(await download_at1101_config(network_interfaces));
    else {
        VScript.enforce(network_interfaces instanceof VAPI.AT1130.NetworkInterfaces.All);
        return new NetworkConfig(await download_at1130_config(network_interfaces));
    }
}
