import { read_error_message } from "./error_channels.js";
import { NumericIDPool, StringIDPool } from "./id_pool.js";
import { KWLCache } from "./kwl_cache.js";
import { ListenerRegistry } from "./listener_registry.js";
import { download_json } from "./polyfills/polyfill_http.js";
import * as WebSocket from "./polyfills/polyfill_ws.js";
import { binary_concat } from "./polyfills/polyfill_ws.js";
import { get_env } from "./polyfills/polyfill_globals.js";
import {
	component_of_kwl,
	EventHandler,
	MissingData,
	Watcher,
	with_timeout,
} from "./pervasives.js";
import { ResourceCache } from "./resource_cache.js";
import { indices as indices_of_rowmask } from "./rowmask.js";
import { ensure_v2 } from "./schema_v1_to_v2.js";
import * as V2 from "./schema_v2.js";
import { Duration } from "./time.js";
import {
	enforce,
	enforce_eq,
	enforce_nonnull,
	path_hd,
	path_index,
	path_strip_trailing_index,
	path_to_branch,
	pause,
	snake_case,
	unreachable,
} from "./utilities.js";
import { status_is, status_sparsely_matches } from "./write_validators.js";
export class ThumbnailListener {
	id;
	destructor;
	constructor(id, destructor) {
		this.id = id;
		this.destructor = destructor;
	}
	unregister() {
		this.destructor(this);
	}
}
var ConnectionStage;
(function (ConnectionStage) {
	ConnectionStage[(ConnectionStage["Disconnected"] = 0)] = "Disconnected";
	ConnectionStage[(ConnectionStage["ConnectionPending"] = 1)] =
		"ConnectionPending";
	ConnectionStage[(ConnectionStage["ConnectionEstablished"] = 2)] =
		"ConnectionEstablished";
})(ConnectionStage || (ConnectionStage = {}));
var TowelState;
(function (TowelState) {
	TowelState[(TowelState["Blocking"] = 0)] = "Blocking";
	TowelState[(TowelState["Set"] = 1)] = "Set";
	TowelState[(TowelState["Unset"] = 2)] = "Unset";
})(TowelState || (TowelState = {}));
export function stringify_failure(failure, vsocket) {
	switch (failure.type) {
		case "access-denied":
			return (
				`Unable to access ${failure.kwl}${failure.kw !== undefined ? `.${failure.kw}` : ""} at ${vsocket.ip}` +
				`: ${failure.reason}`
			);
		case "create-row":
			return (
				`Unable to create new row within ${failure.kwl} at ${vsocket.ip}` +
				(failure.opts.allow_reuse_row !== undefined ||
				failure.opts.index !== undefined ||
				failure.opts.name !== undefined
					? ` (options were ${JSON.stringify(failure.opts)})`
					: "")
			);
		case "rename-row":
			return `Unable to rename ${failure.kwl} at ${vsocket.ip} to ${failure.name}`;
		case "delete-rows":
			return `Unable to delete ${failure.kwls.join(", ")} at ${vsocket.ip}`;
		case "write":
			return (
				`Unable to change ${failure.kwl}.${failure.kw} at ${vsocket.ip} to ${failure.payload}` +
				(failure.reason ? `: ${failure.reason}` : "")
			);
		case "read":
			return `Unable to read ${failure.kwl}.${failure.kw} from ${vsocket.ip}`;
		case "wait-until":
			return `wait_until(${failure.kwl}.${failure.kw}, ${failure.criterion}) at ${vsocket.ip} failed to resolve within ${failure.timeout.toString("convenient")}`;
	}
}
function make_throwing_failure_handler(vsocket) {
	return (failure) => {
		throw new Error(stringify_failure(failure, vsocket));
	};
}
var ExpectedMessageType;
(function (ExpectedMessageType) {
	ExpectedMessageType[(ExpectedMessageType["JSON"] = 0)] = "JSON";
	ExpectedMessageType[(ExpectedMessageType["ThumbnailHeader"] = 1)] =
		"ThumbnailHeader";
	ExpectedMessageType[(ExpectedMessageType["Thumbnail"] = 2)] = "Thumbnail";
	ExpectedMessageType[(ExpectedMessageType["File"] = 3)] = "File";
})(ExpectedMessageType || (ExpectedMessageType = {}));
function generate_random_towel() {
	let res = "random-towel-";
	for (let i = 0; i < 20; ++i) {
		res += Math.round(36 * Math.random()).toString(36);
	}
	return res;
}
export class VSocket {
	static INTERVAL_SECS = 1;
	static schema_cache = new ResourceCache();
	static s_timeout_scale = 1;
	handle_error(error) {
		this.connection_pars.event_handler({ event_type: "error", error });
	}
	m_pending_subscriptions = {
		burstcount: 0,
		pending_flush: null,
		per_kwl: new Map(),
	};
	m_kwl_cache = new KWLCache();
	state = {
		stage: 0,
		reconnect_interval_ms: null,
	};
	max_messages_per_burst = 1000;
	get check_component_liveness() {
		return !this.connection_pars.permit_degraded_mode;
	}
	static scaledTimeout(suggested) {
		return (suggested ?? new Duration(5, "s")).times(VSocket.s_timeout_scale);
	}
	dump_stats() {
		console.log(`stats for vsocket@${this.ip}: KWLCache: `);
		this.m_kwl_cache.dump_stats();
		console.log(`ListenerRegistry:`);
		this.listener_registry.dump_stats();
	}
	static setTimeoutScale(factor) {
		this.s_timeout_scale = factor;
	}
	connection_pars;
	outbound_snooper = null;
	listener_registry = new ListenerRegistry((error) =>
		this.connection_pars.event_handler({ event_type: "error", error }),
	);
	prune_listeners(keep_if) {
		this.listener_registry.prune_listeners(keep_if);
	}
	constructor(pars) {
		const protocol = pars.protocol ?? "ws";
		const ip = pars.ip ?? get_env()["IP"];
		if (!ip)
			throw new Error(
				`No IP address has been specified, and no default address has been set through an IP=... environment parameter`,
			);
		this.connection_pars = {
			...pars,
			ip,
			login: pars.login ?? null,
			protocol,
			towel: pars.towel ?? generate_random_towel(),
			port: pars.port ?? (protocol === "ws" ? 80 : 443),
			cache_everything: pars.cache_everything ?? false,
			reject_unauthorized: pars.reject_unauthorized ?? false,
			remove_unhandled_subscriptions:
				pars.remove_unhandled_subscriptions ?? true,
			timeout: VSocket.scaledTimeout(pars.timeout),
			max_payload_size: pars.max_payload_size ?? 4 * 1024 * 1024,
			ignored_modules: pars.ignored_modules ?? [],
			check_towel_on_write: pars.check_towel_on_write ?? true,
			event_handler:
				pars.event_handler ??
				((event) => {
					switch (event.event_type) {
						case "error":
							setTimeout(() => {
								throw event.error;
							}, 0);
							break;
						case "info":
							console.log(event.msg);
					}
				}),
			permit_degraded_mode: pars.permit_degraded_mode ?? false,
			failure_handler:
				pars.failure_handler ?? make_throwing_failure_handler(this),
		};
	}
	async do_check_write(full_kwl, opts) {
		enforce(this.is_ready());
		if (
			(opts?.check_component_liveness ?? this.check_component_liveness) &&
			!this.module_registry.is_online(component_of_kwl(full_kwl))
		) {
			this.connection_pars.failure_handler({
				type: "access-denied",
				kwl: full_kwl,
				reason: `component \`${component_of_kwl(full_kwl)}\` appears to be offline.`,
			});
			return;
		}
		if (!(opts?.check_towel ?? this.connection_pars.check_towel_on_write)) {
			return;
		}
		switch (this.current_towel.state) {
			case 0:
				this.connection_pars.failure_handler({
					type: "access-denied",
					kwl: full_kwl,
					reason: `blocked by towel '${this.current_towel.value}'`,
				});
				return;
			case 1:
				return;
			case 2:
				await this.place_towel();
		}
		enforce_eq(this.current_towel.state, 1);
	}
	is_ready() {
		return this.state.stage === 2;
	}
	get_state() {
		if (this.state.stage !== 2) {
			throw new Error(
				`Unable to access connection state, as this VSocket currently isn't connected to ${this.connection_pars.ip}`,
			);
		}
		return this.state;
	}
	get protocol_features() {
		enforce(this.state.stage !== 0);
		return this.state.protocol_features;
	}
	set_snooper(outbound) {
		this.outbound_snooper = outbound;
	}
	set_failure_handler(failure_handler) {
		this.connection_pars.failure_handler =
			failure_handler ?? make_throwing_failure_handler(this);
	}
	identify() {
		return `${this.protocol}://${this.ip}`;
	}
	get current_towel() {
		return this.get_state().current_towel;
	}
	get build_info() {
		return this.get_state().build_info;
	}
	get ip() {
		return this.connection_pars.ip;
	}
	get login() {
		return this.connection_pars.login
			? { ...this.connection_pars.login }
			: null;
	}
	get protocol() {
		return this.connection_pars.protocol;
	}
	get port() {
		return this.connection_pars.port;
	}
	get towel() {
		return this.connection_pars.towel;
	}
	get runtime_constants() {
		return this.get_state().runtime_constants;
	}
	get schema() {
		return this.get_state().schema;
	}
	get module_registry() {
		return this.get_state().module_registry;
	}
	get root() {
		return new Root(this);
	}
	async place_towel(pars) {
		const state = this.get_state();
		if (
			state.current_towel.state === 0 &&
			!(pars?.override_preexisting_towel ?? false)
		) {
			this.connection_pars.failure_handler({
				type: "access-denied",
				kwl: "system.usrinfo",
				kw: "towel",
				reason:
					"currently blocked by preexisting towel. If you wish to clear it anyway, use await place_towel({ override_preexisting_towel: true })",
			});
			return;
		}
		const old_towel = { ...state.current_towel };
		state.current_towel = { value: this.towel, state: 1 };
		try {
			await this.write({ kwl: "system.usrinfo", kw: "towel" }, this.towel, {
				check_towel: false,
			});
		} catch (e) {
			state.current_towel = old_towel;
			throw e;
		}
	}
	async subscription_flush() {
		if (this.m_pending_subscriptions.pending_flush) {
			await this.m_pending_subscriptions.pending_flush;
		}
	}
	async watch_error_channels(handler, opts) {
		const result = [];
		await Promise.all(
			this.schema.error_feedback_channels.map(async (channel) => {
				const owningModule = channel.owner;
				const shouldBeLive =
					this.schema.keywords.find(
						(comp) =>
							comp.owning_module === owningModule && comp.enabled === true,
					) !== undefined;
				if (shouldBeLive) {
					result.push(
						await this.watch(
							{ kwl: channel.kwl, kw: "error_feedback" },
							(raw) => {
								const msg = read_error_message(raw);
								if (!msg) {
									return;
								}
								handler(msg);
							},
							{
								check_component_liveness: false,
								...(opts?.listener_id
									? {
											listener_id: `_err_${channel.kwl}_${opts.listener_id}`,
										}
									: {}),
							},
						),
					);
				}
			}),
		);
		return result;
	}
	recover_subscriptions() {
		for (const [kwl, { complete, partial }] of this.listener_registry
			.kwl_listeners) {
			if (Object.keys(complete).length !== 0) {
				this.subscribe_to({ kwl });
			} else {
				for (const kw in partial) {
					this.subscribe_to({ kwl, kw: kw });
				}
			}
		}
	}
	async marker(pars) {
		const state = this.get_state();
		const id = state.next_marker_id;
		const did_collide = state.pending_markers.has(id);
		const pending_markers = state.pending_markers;
		state.next_marker_id = (id + 1) % (1 << 15);
		if (pars?.flush_subscriptions ?? true) {
			await this.subscription_flush();
		}
		return new Promise((resolve, reject) => {
			if (did_collide) {
				reject(`Marker IDs exhausted`);
			} else {
				this.send([{ op: "marker", marker: id | (1 << 15) }]);
				const expiryDateMS =
					new Date().valueOf() + VSocket.scaledTimeout(pars?.timeout).ms();
				pending_markers.set(id, { expiryDateMS, resolve, reject });
			}
		});
	}
	query_cache(path) {
		return this.m_kwl_cache.query_kw(path);
	}
	query_cache_kwl(kwl) {
		return this.m_kwl_cache.query_kwl(kwl);
	}
	do_get_runtime_constant(constants, constant_string, runtime_constants) {
		const parts = constant_string.split("::");
		enforce(parts.length === 2);
		const [ua_name, constant_name] = parts;
		if (!constants[ua_name][constant_name])
			throw new Error(
				`Unable to read runtime constant ${ua_name}.${constant_name}; perhaps you are trying to connect to an old software build?`,
			);
		const sysname_path = constants[ua_name][constant_name].path;
		const sysname_parts = sysname_path.split(".");
		enforce(sysname_parts.length === 2);
		return runtime_constants.get_constant(sysname_parts[0], sysname_parts[1]);
	}
	get_runtime_constant(constant_string) {
		return this.do_get_runtime_constant(
			this.schema.constants,
			constant_string,
			this.runtime_constants,
		);
	}
	send(req) {
		if (this.state.stage === 0) {
			throw new Error(
				`Unable to dispatch request ${JSON.stringify(req)}, as this socket is currently not connected to ${this.connection_pars.ip}`,
			);
		}
		this.state.websocket.send(JSON.stringify(req));
		this.outbound_snooper?.(req);
	}
	async read_unchecked(path, raw_timeout) {
		await this.subscription_flush();
		const timeout = VSocket.scaledTimeout(raw_timeout);
		return await new Promise((resolve, reject) => {
			let watchdog = null;
			const listener = this.register_kw_listener(
				{
					kwl: path.kwl,
					kw: path.kw,
					listener_type: 1,
					execution_strategy: 0,
				},
				(payload) => {
					clearTimeout(watchdog);
					resolve(payload);
				},
			);
			watchdog = setTimeout(() => {
				listener.unregister();
				reject(
					new Error(`Unable to read ${path.kwl}.${path.kw} from ${this.ip}`),
				);
			}, timeout.ms());
			this.send([{ op: "readAll", kwl: path.kwl, kw: path.kw }]);
		});
	}
	write_unchecked(path, payload) {
		this.send([
			{
				op: "data",
				kwl: path.kwl,
				kw: { [path.kw]: payload },
			},
		]);
	}
	static async open(pars) {
		const result = new VSocket(pars);
		await result.do_open_connection("first-connect");
		return result;
	}
	do_open_connection(mode) {
		const aborted = Symbol("aborted");
		const timeout = VSocket.scaledTimeout(this.connection_pars.timeout);
		const MIN_RECONNECT_INTERVAL_MS = 250;
		const MAX_RECONNECT_INTERVAL_MS = 2000;
		let did_open_at_least_once = mode === "reconnect";
		let timer_pending = false;
		const arm_reconnect_timer = () => {
			if (timer_pending) return;
			if (this.state.stage !== 0) {
				try {
					this.state.websocket.close();
				} catch (_) {
				} finally {
					this.state = {
						stage: 0,
						reconnect_interval_ms: MIN_RECONNECT_INTERVAL_MS,
					};
				}
			}
			if (
				!(this.state.stage === 0 && this.state.reconnect_interval_ms === null)
			) {
				const reconnect_interval_ms =
					this.state.stage !== 0
						? MIN_RECONNECT_INTERVAL_MS
						: Math.min(
								Math.round(
									enforce_nonnull(this.state.reconnect_interval_ms) * 1.5,
								),
								MAX_RECONNECT_INTERVAL_MS,
							);
				this.state = {
					stage: 0,
					reconnect_interval_ms,
				};
				setTimeout(async () => {
					timer_pending = false;
					enforce(this.state.stage === 0);
					if (this.state.reconnect_interval_ms === null) {
						this.connection_pars.event_handler({
							event_type: "info",
							msg: "Aborting reconnect attempt (based on the assumption that this socket has been closed deliberately)",
						});
						return;
					}
					try {
						await this.do_open_connection("reconnect");
						enforce(this.state.stage === 2);
						this.recover_subscriptions();
						enforce(this.state.stage === 2);
					} catch (_) {
						return;
					}
					this.connection_pars.event_handler?.({
						event_type: "connection-reopened",
					});
				}, reconnect_interval_ms);
				timer_pending = true;
			}
		};
		const open_websocket = () => {
			return new Promise((resolve, reject) => {
				const websocket = new WebSocket.Adapter({
					...this.connection_pars,
					timeout,
					on_close: () => {
						try {
							this.m_kwl_cache.clear();
							this.connection_pars.event_handler(
								this.state.stage === 2 &&
									this.state.expecting_close_because !== null
									? {
											event_type: "expected-close",
											reason: this.state.expecting_close_because,
										}
									: { event_type: "unexpected-close" },
							);
						} finally {
							if (did_open_at_least_once) arm_reconnect_timer();
						}
					},
					on_error: () => {
						try {
							this.connection_pars.event_handler({
								event_type: "websocket-error",
							});
						} finally {
							if (did_open_at_least_once) arm_reconnect_timer();
						}
					},
					reject: (err) => reject(err),
					on_first_msg: (msg_data) => {
						try {
							let obj = JSON.parse(msg_data);
							if (!Array.isArray(obj)) {
								reject(
									`Got unexpected first message: should have been an array [{ "webserver_buildinfo": <...> }]), but was '${msg_data}'`,
								);
								return;
							}
							obj = obj[0];
							if (
								Object.prototype.hasOwnProperty.call(obj, "webserver_buildinfo")
							) {
								if (
									Object.prototype.hasOwnProperty.call(obj, "protocol_features")
								) {
									enforce(
										typeof obj.protocol_features === "object" &&
											obj.protocol_features !== null,
									);
									resolve([
										obj.webserver_buildinfo,
										{
											create_row_request_id:
												obj.protocol_features.create_row_request_id ?? false,
											runtime_constants_json:
												obj.protocol_features.runtime_constants_json ?? false,
											readable_command_tracking:
												obj.protocol_features.readable_command_tracking ??
												false,
											vscriptd: obj.protocol_features.vscriptd ?? false,
											http_network_config:
												obj.protocol_features.http_network_config ?? false,
										},
										websocket,
									]);
								} else {
									resolve([
										obj.webserver_buildinfo,
										{
											create_row_request_id: false,
											runtime_constants_json: false,
											readable_command_tracking: false,
											vscriptd: obj.protocol_features.vscriptd ?? false,
											http_network_config:
												obj.protocol_features.http_network_config ?? false,
										},
										websocket,
									]);
								}
							} else {
								throw new Error(
									`Got unexpected first message from ${this.connection_pars.ip}: should have contained a field named 'webserver_buildinfo'`,
								);
							}
						} catch (e) {
							reject(e);
						}
					},
					on_further_msg: (msg) => this.handle_incoming(msg),
				});
			});
		};
		return with_timeout(timeout, async (context) => {
			context.desc = `while trying to open WebSocket connection to ${this.identify()}`;
			let websocket = null;
			try {
				enforce(this.state.stage === 0);
				const [webserver_build_info, protocol_features, ws] =
					await open_websocket();
				websocket = ws;
				this.state = {
					stage: 1,
					expected_message_type: 0,
					websocket,
					webserver_build_info,
					protocol_features,
				};
				const protocol =
					this.connection_pars.protocol === "ws" ? "http" : "https";
				if (context.timed_out) throw aborted;
				context.desc = `while trying to download data/build_info.json from ${this.identify()}`;
				const build_info = await download_json({
					ip: this.connection_pars.ip,
					protocol,
					port: this.connection_pars.port,
					path: "/data/build_info.json",
					reject_unauthorized: this.connection_pars.reject_unauthorized,
					timeout,
				});
				if (context.timed_out) throw aborted;
				const key_suffix = build_info.commit.endsWith("+")
					? "|" + build_info.timestamp
					: "";
				const key =
					build_info.commit +
					key_suffix +
					(build_info.hardware_model ?? "<unknown hardware model>");
				context.desc = `while trying to download data/schema.json from ${this.identify()}`;
				if (context.timed_out) throw aborted;
				const raw_schema = ensure_v2(
					await VSocket.schema_cache.retrieve({
						ip: this.connection_pars.ip,
						protocol,
						port: this.connection_pars.port,
						path: "/data/schema.json",
						key,
						reject_unauthorized: this.connection_pars.reject_unauthorized,
						timeout,
					}),
				);
				if (context.timed_out) throw aborted;
				const ignored_component_ua_names = raw_schema.keywords
					.filter(
						(comp) =>
							this.connection_pars.ignored_modules.findIndex(
								(m) => m === comp.owning_module,
							) !== -1,
					)
					.map((comp) => comp.ua_name);
				if (this.connection_pars.ignored_modules.length !== 0) {
					raw_schema.keywords = raw_schema.keywords.filter(
						(comp) =>
							this.connection_pars.ignored_modules.findIndex(
								(m) => m === comp.owning_module,
							) === -1,
					);
					const is_ignored = (td_name) => {
						const comp_ua_name = td_name.split("::")[0];
						return (
							ignored_component_ua_names.findIndex(
								(ua) => comp_ua_name === ua,
							) !== -1
						);
					};
					const old_typedefs = raw_schema.typedefs;
					const old_typedef_identifiers = raw_schema.typedef_identifiers;
					raw_schema.typedefs = {};
					raw_schema.typedef_identifiers = [];
					for (const td_name in old_typedefs) {
						if (!is_ignored(td_name)) {
							raw_schema.typedefs[td_name] = old_typedefs[td_name];
						}
					}
					for (const td_name of old_typedef_identifiers) {
						if (!is_ignored(td_name)) {
							raw_schema.typedef_identifiers.push(td_name);
						}
					}
				}
				context.desc = `while trying to initialize module registry for ${this.identify()}`;
				if (context.timed_out) throw aborted;
				const module_registry = await ModuleRegistry.initialize(
					raw_schema.keywords,
					this,
					this.connection_pars.permit_degraded_mode,
					context,
				);
				if (context.timed_out) throw aborted;
				const [runtime_constants, disabledComponents] =
					await RuntimeConstants.initialize(raw_schema, this, timeout, context);
				if (context.timed_out) throw aborted;
				const schema = V2.annotate(
					raw_schema,
					disabledComponents,
					(constant_name) => {
						const result = this.do_get_runtime_constant(
							raw_schema.constants,
							constant_name,
							runtime_constants,
						);
						return result;
					},
				);
				this.state = {
					stage: 2,
					build_info,
					current_towel: { state: 2, value: "" },
					expecting_close_because: null,
					expected_message_type: 0,
					module_registry,
					next_marker_id: 0,
					pending_markers: new Map(),
					protocol_features,
					reconnect_on_close: true,
					runtime_constants,
					schema,
					thumbnails: null,
					webserver_build_info: this.state.webserver_build_info,
					websocket: this.state.websocket,
				};
				this.register_kw_listener(
					{
						listener_id: "_towel_watch",
						kwl: "system.usrinfo",
						kw: "towel",
						ensure_initial_read: true,
						execution_strategy: 0,
						cache: true,
						listener_type: 0,
					},
					(new_towel) => {
						if (this.state.stage === 2) {
							this.state.current_towel = {
								value: new_towel,
								state:
									new_towel === this.connection_pars.towel
										? 1
										: new_towel.length === 0
											? 2
											: 0,
							};
						}
					},
				);
				this.send([
					{
						op: "subscribe",
						kwl: "system.usrinfo",
						kw: ["towel"],
					},
				]);
				for (const [kwl, cp] of this.listener_registry.kwl_listeners) {
					if (Object.keys(cp.complete).length > 0) {
						this.subscribe_to({ kwl, ensure_initial_read: true });
					} else {
						for (const kw in cp.partial) {
							this.subscribe_to({
								kwl,
								kw: kw,
								ensure_initial_read: true,
							});
						}
					}
				}
				did_open_at_least_once = true;
			} catch (e) {
				if (did_open_at_least_once) {
					arm_reconnect_timer();
				} else {
					enforce(websocket === null || this.state.stage !== 0);
					await this.close();
				}
				if (typeof e !== typeof aborted) {
					throw e;
				}
			}
			if (this.state.stage === 2)
				this.connection_pars.event_handler({
					event_type: "connection-reopened",
				});
		});
	}
	async close() {
		try {
			if (
				this.state.stage === 2 &&
				this.state.current_towel.value.length > 0 &&
				this.state.current_towel.value === this.connection_pars.towel
			) {
				enforce(this.state.current_towel.state === 1);
				await this.write({ kwl: "system.usrinfo", kw: "towel" }, "");
			}
		} catch (_) {
			this.write_unchecked({ kwl: "system.usrinfo", kw: "towel" }, "");
		} finally {
			if (this.state.stage !== 0) {
				this.state.websocket.close();
			}
			this.state = {
				stage: 0,
				reconnect_interval_ms: null,
			};
			this.listener_registry.close();
		}
	}
	async recovery(pars) {
		const timeout = VSocket.scaledTimeout(
			pars?.timeout ??
				Duration.max(this.connection_pars.timeout, new Duration(3, "min")),
		);
		const interval = new Duration(100, "ms");
		for (
			const t0 = new Date().valueOf();
			new Date().valueOf() < t0 + timeout.ms();
		) {
			if (this.is_ready()) {
				await this.read({
					kwl: "system.usrinfo",
					kw: "towel",
				}).catch(() => {});
				return;
			}
			await pause(interval);
		}
		throw new Error(
			`${this.connection_pars.ip} failed to recover within ${Math.round(timeout.s())} s; maybe the machine had to swap partitions?`,
		);
	}
	register_kw_listener(pars, handler) {
		if (!this.listener_registry.have_permanent_listener(pars)) {
			switch (pars.listener_type) {
				case 1:
					this.send([{ op: "readAll", kwl: pars.kwl, kw: [pars.kw] }]);
					break;
				case 0:
					if (pars.execution_strategy === 0) {
						try {
							this.send([
								pars.kw
									? { kwl: pars.kwl, kw: [pars.kw], op: "subscribe" }
									: { kwl: pars.kwl, op: "subscribe" },
							]);
						} catch (_) {
							this.subscribe_to(pars);
						}
					} else {
						this.subscribe_to(pars);
					}
					break;
			}
		}
		return this.listener_registry.register_kw_listener(pars, handler);
	}
	register_kwl_listener(pars, handler) {
		if (!this.listener_registry.have_permanent_listener(pars)) {
			if (pars.execution_strategy === 0) {
				try {
					this.send([{ op: "subscribe", kwl: pars.kwl }]);
				} catch (_) {
					this.subscribe_to(pars);
				}
			} else {
				this.subscribe_to(pars);
			}
		}
		return this.listener_registry.register_kwl_listener(pars, handler);
	}
	register_global_listener(handler, pars) {
		return this.listener_registry.register_global_listener(handler, pars);
	}
	subscribe_to(pars) {
		if (
			this.state &&
			this.state.stage !== 0 &&
			this.m_pending_subscriptions.burstcount++ < this.max_messages_per_burst
		) {
			this.send([
				pars.kw
					? { kwl: pars.kwl, kw: [pars.kw], op: "subscribe" }
					: { kwl: pars.kwl, op: "subscribe" },
			]);
		} else {
			const entry = this.m_pending_subscriptions.per_kwl.get(pars.kwl);
			if (entry === undefined) {
				this.m_pending_subscriptions.per_kwl.set(pars.kwl, {
					ensure_initial_read: pars.ensure_initial_read ?? true,
					kws: pars.kw === undefined ? undefined : [pars.kw],
				});
			} else {
				entry.ensure_initial_read ||= pars.ensure_initial_read ?? true;
				if (pars.kw === undefined || entry.kws === undefined) {
					entry.kws = undefined;
				} else if (!entry.kws.find((other_kw) => other_kw === pars.kw)) {
					entry.kws.push(pars.kw);
				}
			}
			if (this.m_pending_subscriptions.pending_flush === null) {
				this.m_pending_subscriptions.pending_flush = (async () => {
					while (this.m_pending_subscriptions.per_kwl.size !== 0) {
						while (true) {
							if (this.state && this.state.stage !== 0) break;
							await pause(new Duration(100, "ms"));
						}
						let n = 0;
						const outbound = [];
						for (const [kwl, entry] of this.m_pending_subscriptions.per_kwl) {
							if (entry.kws === null) {
								this.send([{ op: "subscribe", kwl }]);
							} else {
								this.send([{ op: "subscribe", kwl, kw: entry.kws }]);
							}
							n += 1;
							if (entry.ensure_initial_read) {
								this.send([{ op: "readAll", kwl }]);
								n += 1;
							}
							outbound.push([kwl, entry]);
							this.m_pending_subscriptions.per_kwl.delete(kwl);
							if (n > this.max_messages_per_burst) break;
						}
						try {
							await this.marker({ flush_subscriptions: false });
						} catch (e) {
							for (const [kwl, entry] of outbound) {
								this.m_pending_subscriptions.per_kwl.set(kwl, entry);
							}
							break;
						} finally {
							if (this.m_pending_subscriptions.per_kwl.size === 0) {
								this.m_pending_subscriptions.pending_flush = null;
							}
						}
					}
				})();
			}
		}
	}
	register_thumbnail_listener(handler, pars) {
		const state = this.get_state();
		const kwl = "monitoring.live_view";
		if (state.thumbnails === null) {
			this.send([
				{
					op: "subscribe",
					kwl,
					kw: ["thumbnail_header", "thumbnail"],
				},
			]);
			state.thumbnails = {
				header_data: undefined,
				prev_data: undefined,
				handlers: new Map(),
			};
		}
		enforce_nonnull(state.thumbnails);
		const id = StringIDPool.generate_or_use_id(
			enforce_nonnull(state.thumbnails).handlers,
			pars?.listener_id,
		);
		enforce_nonnull(state.thumbnails).handlers.set(id, handler);
		const listener_id = "_bogus_listener_thumbnails";
		this.listener_registry.register_kw_listener(
			{
				kwl,
				kw: "thumbnail",
				listener_id,
				cache: false,
				listener_type: 0,
				execution_strategy: 1,
				ensure_initial_read: false,
			},
			() => {
				return;
			},
		);
		return new ThumbnailListener(id, () => {
			if (state.thumbnails) {
				state.thumbnails.handlers.delete(id);
				if (state.thumbnails.handlers.size === 0) {
					state.thumbnails = null;
					this.listener_registry.safely_delete_handler({
						kwl,
						kw: "thumbnail",
						id: listener_id,
					});
					this.send([
						{
							op: "unsubscribe",
							kwl,
							kw: ["thumbnail_header", "thumbnail"],
						},
					]);
				}
			}
		});
	}
	handle_incoming(msg) {
		const state = this.state;
		switch (state.expected_message_type) {
			case 0:
				{
					const data = JSON.parse(msg.data);
					const len = data.length;
					for (let idx = 0; idx < len; idx++) {
						const obj = data[idx];
						if (Object.prototype.hasOwnProperty.call(obj, "kwl")) {
							if (!Object.prototype.hasOwnProperty.call(obj, "kw")) {
								this.handle_error(
									new Error("Encountered malformed message:" + obj),
								);
								continue;
							}
							if (this.connection_pars.cache_everything) {
								this.m_kwl_cache.register(obj);
							}
							for (const gl_entry of this.listener_registry.global_listeners) {
								try {
									gl_entry[1](obj);
								} catch (e) {
									this.handle_error(e);
								}
							}
							let unsubscribe_from_unlistened =
								this.connection_pars.remove_unhandled_subscriptions;
							const kwl_listeners = this.listener_registry.kwl_listeners.get(
								obj.kwl,
							);
							if (kwl_listeners === undefined) {
								if (unsubscribe_from_unlistened) {
									this.send([{ op: "unsubscribe", kwl: obj.kwl }]);
								}
								continue;
							}
							for (const id in kwl_listeners.complete) {
								unsubscribe_from_unlistened = false;
								const entry = kwl_listeners.complete[id];
								if (entry.execution_strategy === 0) {
									try {
										entry.handler();
									} catch (e) {
										this.handle_error(e);
									}
								} else {
									this.listener_registry.pending_lazy_updates.kwls.set(
										id,
										entry.handler,
									);
								}
							}
							for (const kw_name in obj.kw) {
								const payload = obj.kw[kw_name];
								const kw_listeners = kwl_listeners.partial[kw_name];
								if (kw_listeners === undefined) {
									if (unsubscribe_from_unlistened) {
										this.send([
											{ op: "unsubscribe", kwl: obj.kwl, kw: [kw_name] },
										]);
									}
								} else {
									for (const id in kw_listeners) {
										const entry = kw_listeners[id];
										if (entry.execution_strategy === 0) {
											try {
												entry.handler(payload);
											} catch (e) {
												this.handle_error(e);
											}
										} else {
											this.listener_registry.pending_lazy_updates.kws.set(id, [
												entry.handler,
												payload,
											]);
										}
									}
								}
							}
						} else if (Object.prototype.hasOwnProperty.call(obj, "file")) {
							state.expected_message_type =
								obj.file === "thumbnail_header" ? 1 : 3;
						} else if (Object.prototype.hasOwnProperty.call(obj, "thumbnail")) {
							state.expected_message_type = 2;
						} else if (Object.prototype.hasOwnProperty.call(obj, "marker")) {
							enforce(typeof obj.marker === "number");
							const id = obj.marker & 0x7fff;
							enforce(this.state.stage === 2);
							if (state.pending_markers.has(id)) {
								enforce_nonnull(state.pending_markers.get(id)).resolve();
								state.pending_markers.delete(id);
							} else {
								this.handle_error(
									new Error(`Received unexpected marker #${id}`),
								);
							}
						} else if (!("login" in obj && Object.keys(obj).length === 1)) {
							this.handle_error(
								new Error("Received unexpected message:" + obj),
							);
						}
					}
				}
				return;
			case 3:
				break;
			case 1:
				{
					enforce_eq(this.state.stage, 2);
					const state = this.state;
					if (state.thumbnails) {
						state.thumbnails.header_data = msg.data;
					}
				}
				break;
			case 2:
				{
					enforce_eq(this.state.stage, 2);
					const state = this.state;
					if (state.thumbnails) {
						state.thumbnails.prev_data = binary_concat(
							enforce_nonnull(state.thumbnails.header_data),
							msg.data,
						);
						for (const [_, handler] of state.thumbnails.handlers) {
							void _;
							try {
								handler(state.thumbnails.prev_data);
							} catch (e) {
								this.handle_error(e);
							}
						}
					}
				}
				break;
		}
		state.expected_message_type = 0;
	}
	async read(full_path, opts) {
		this.do_check_readonly(full_path.kwl, opts);
		if (opts?.use_cache_if_present ?? true) {
			const maybe_cached = this.query_cache(full_path);
			if (maybe_cached !== MissingData) {
				return maybe_cached[1];
			}
		}
		return await this.read_unchecked(full_path, opts?.timeout);
	}
	async wait_until_offline(pars) {
		const timeout =
			pars?.timeout ??
			Duration.max(this.connection_pars.timeout, new Duration(2, "min"));
		const deadline = new Date().valueOf() + timeout.ms();
		while (this.is_ready()) {
			await pause(new Duration(100, "ms"));
			if (new Date().valueOf() > deadline)
				throw new Error(
					`Timed out while waiting for ${this.identify()} to go offline`,
				);
		}
	}
	async reboot(pars) {
		await this.do_check_write("system", {
			check_component_liveness: true,
			check_towel: true,
		});
		this.get_state().expecting_close_because = "reboot";
		this.write_unchecked(
			{ kwl: "system", kw: "reboot" },
			pars?.command ?? "reboot",
		);
		await this.wait_until_offline();
		await this.recovery({ ...pars });
	}
	async reset(pars) {
		await this.do_check_write("system", {
			check_component_liveness: true,
			check_towel: true,
		});
		this.get_state().expecting_close_because = "reset";
		this.write_unchecked(
			{ kwl: "system", kw: "reset" },
			pars?.partition ?? "reset",
		);
		await this.wait_until_offline();
		await this.recovery({ ...pars });
	}
	async table_create_row(opts) {
		await this.subscription_flush();
		await this.do_check_write(opts.table_kwl, opts);
		const with_index = opts?.index !== undefined;
		const with_name = opts?.name !== undefined;
		const request_id = NumericIDPool.generate_id();
		const supports_request_id =
			this.protocol_features.create_row_request_id === true;
		const payload = (() => {
			if (
				with_index ||
				with_name ||
				opts?.allow_reuse_row !== undefined ||
				supports_request_id
			) {
				const result = {};
				if (with_index) {
					result.index = opts.index;
				}
				if (with_name) {
					result.name = opts.name;
				}
				if (opts?.allow_reuse_row !== undefined) {
					result.allow_reuse_row = opts.allow_reuse_row;
				}
				if (supports_request_id) {
					result.request_id = request_id;
				}
				return result;
			} else {
				return null;
			}
		})();
		const response = await new Promise((resolve) => {
			const timeout = VSocket.scaledTimeout(opts?.timeout);
			let watchdog = null;
			const listener = this.register_kw_listener(
				{
					kwl: opts.table_kwl,
					kw: "created_row",
					listener_type: 0,
					execution_strategy: 0,
				},
				(r) => {
					enforce(r !== null);
					enforce(Array.isArray(r));
					const directed_at_us = (() => {
						if (supports_request_id) {
							enforce(r.length === 3);
							return r[2] === request_id;
						} else {
							return true;
						}
					})();
					if (!directed_at_us) {
						return;
					}
					if (r[0] === null) {
						clearTimeout(watchdog);
						resolve(null);
					} else {
						if (
							(!with_index || opts?.index === r[0]) &&
							(!with_name || opts?.name === r[1])
						) {
							clearTimeout(watchdog);
							resolve([r[0], r[1]]);
						}
					}
				},
			);
			watchdog = setTimeout(() => {
				listener.unregister();
				resolve(null);
			}, timeout.ms());
			this.write_unchecked({ kwl: opts.table_kwl, kw: "create_row" }, payload);
		});
		if (response === null) {
			const failure = {
				type: "create-row",
				kwl: opts.table_kwl,
				opts,
			};
			this.connection_pars.failure_handler(failure);
			throw new Error(stringify_failure(failure, this));
		} else {
			return response;
		}
	}
	async table_rename_row(opts) {
		await this.subscription_flush();
		try {
			await this.write(
				{ kwl: opts.row_kwl, kw: "row_name_command" },
				opts.desired_name,
				{
					...opts,
					check_keyword_type: false,
				},
			);
		} catch (_) {
			this.connection_pars.failure_handler({
				type: "rename-row",
				kwl: opts.row_kwl,
				name: opts.desired_name,
			});
		}
	}
	async table_indices(opts) {
		await this.subscription_flush();
		this.do_check_readonly(opts.table_kwl, opts);
		const rowmask = await this.read(
			{
				kwl: opts.table_kwl,
				kw: "rowMask",
			},
			opts,
		);
		return indices_of_rowmask(rowmask);
	}
	async table_has_row(opts) {
		await this.subscription_flush();
		const indices = await this.table_indices(opts);
		return indices.indexOf(opts.index) >= 0;
	}
	async table_delete_row(opts) {
		await this.subscription_flush();
		const listener = this.register_kw_listener(
			{
				kwl: opts.table_kwl,
				kw: "rowMask",
				cache: true,
				listener_type: 0,
				execution_strategy: 0,
			},
			() => {},
		);
		try {
			await this.write(
				{
					kwl: `${opts.table_kwl}[${opts.index}]`,
					kw: "row_cmd",
				},
				"DELETE_ROW",
				{
					...opts,
					retry_until: {
						criterion: "custom",
						validator: async () => {
							return !(await this.table_has_row(opts));
						},
					},
					check_keyword_type: false,
				},
			);
		} catch (_) {
			this.connection_pars.failure_handler({
				type: "delete-rows",
				kwls: [`${opts.table_kwl}[${opts.index}]`],
			});
		} finally {
			listener.unregister();
		}
	}
	async table_delete_all_rows(opts) {
		await this.subscription_flush();
		const listener = this.register_kw_listener(
			{
				kwl: opts.table_kwl,
				kw: "rowMask",
				cache: true,
				listener_type: 0,
				execution_strategy: 0,
			},
			() => {},
		);
		try {
			await this.write(
				{
					kwl: opts.table_kwl,
					kw: "table_cmd",
				},
				"DELETE_ALL_ROWS",
				{
					...opts,
					retry_until: {
						criterion: "custom",
						validator: async () => {
							return (await this.table_indices(opts)).length === 0;
						},
					},
					check_keyword_type: false,
				},
			);
		} catch (e) {
			this.connection_pars.failure_handler({
				type: "delete-rows",
				kwls: (await this.table_indices(opts)).map(
					(i) => `${opts.table_kwl}[${i}]`,
				),
			});
		} finally {
			listener.unregister();
		}
	}
	async write(full_path, payload, opts) {
		let can_read = true;
		let desc = null;
		if (opts?.check_keyword_type ?? true) {
			desc = V2.find_keyword(
				this.schema,
				path_to_branch(full_path.kwl),
				full_path.kw,
			);
			enforce(payload !== null || desc.contents.optional);
			if (desc.kw_type === "status" || desc.kw_type === "driver-owned event") {
				this.connection_pars.failure_handler({
					type: "write",
					kwl: full_path.kwl,
					kw: full_path.kw,
					payload,
					reason: `this keyword is of type '${desc.kw_type}' and hence cannot be written to`,
				});
				return;
			}
			can_read = !(
				desc.kw_type === "event" ||
				(desc.kw_priority === "immediate-tracking" &&
					!this.protocol_features.readable_command_tracking)
			);
		}
		await this.do_check_write(full_path.kwl, opts);
		const is_duplex = full_path.kw.endsWith("_command");
		const status_kw = is_duplex
			? full_path.kw.replace(/_command$/, "_status")
			: full_path.kw;
		const timeout = VSocket.scaledTimeout(opts?.timeout);
		const retry_interval_ms = opts?.retry_interval?.ms() ?? 250;
		const indices =
			typeof payload === "object" && payload !== null && !Array.isArray(payload)
				? Object.keys(payload).map((s) => parseInt(s))
				: undefined;
		enforce(indices === undefined || indices.findIndex((x) => isNaN(x)) === -1);
		const wv =
			opts?.retry_until ??
			(indices === undefined
				? status_is(payload, desc)
				: status_sparsely_matches(payload));
		try {
			switch (wv.criterion) {
				case "return-immediately":
					this.write_unchecked(full_path, payload);
					break;
				case "custom": {
					const deadline = new Date().valueOf() + timeout.ms();
					let wait_interval_ms = opts?.first_write_interval?.ms() ?? 5;
					do {
						this.write_unchecked(full_path, payload);
						await pause(new Duration(wait_interval_ms, "ms"));
						if (await wv.validator()) {
							return;
						}
						wait_interval_ms = Math.min(
							retry_interval_ms,
							Math.ceil(1.2 * wait_interval_ms),
						);
					} while (new Date().valueOf() <= deadline);
					throw undefined;
				}
				case "status": {
					this.write_unchecked(full_path, payload);
					if (can_read) {
						const interval = setInterval(() => {
							this.write_unchecked(full_path, payload);
						}, retry_interval_ms);
						await this.wait_until(
							{ ...full_path, kw: status_kw },
							wv.validator,
							{
								check_component_liveness: false,
								timeout,
							},
						).finally(() => clearInterval(interval));
					}
				}
			}
		} catch (_) {
			this.connection_pars.failure_handler({
				type: "write",
				kwl: full_path.kwl,
				kw: full_path.kw,
				payload,
			});
		}
	}
	async wait_until(full_path, criterion, opts) {
		await this.subscription_flush();
		this.do_check_readonly(full_path.kwl, opts);
		const timeout = VSocket.scaledTimeout(opts?.timeout);
		const err = new Error(
			`wait_until(${full_path.kwl}, ${full_path.kw})@${this.identify()}: failed to satisfy the given criterion within ${Math.round(timeout.s())} s`,
		);
		const do_wait = new Promise((resolve, reject) => {
			let timer = undefined;
			const listener = this.register_kw_listener(
				{
					...full_path,
					execution_strategy: 0,
					listener_type: 0,
				},
				(payload) => {
					enforce(payload !== undefined);
					const verdict = criterion(payload);
					if (verdict === true) {
						did_resolve = true;
						listener.unregister();
						if (timer) {
							clearTimeout(timer);
						}
						resolve(payload);
					} else {
						enforce(!did_resolve);
					}
				},
			);
			let did_resolve = false;
			timer = setTimeout(() => {
				if (!did_resolve) {
					listener.unregister();
					reject(err);
				}
			}, timeout.ms());
			if (!(opts?.skip_initial_read ?? false)) {
				this.read(full_path, opts);
			}
		});
		return await do_wait;
	}
	watch_unchecked(full_path, handler, opts) {
		const listener = this.register_kw_listener(
			{
				...full_path,
				...opts,
				execution_strategy: 0,
				listener_type: 0,
			},
			handler,
		);
		return new Watcher(full_path, () => {
			listener.unregister();
		});
	}
	async watch(full_path, handler, opts) {
		await this.subscription_flush();
		this.do_check_readonly(full_path.kwl, opts);
		return this.watch_unchecked(full_path, handler, opts);
	}
	do_check_readonly(full_kwl, opts) {
		enforce(this.is_ready());
		if (
			(opts?.check_component_liveness ?? this.check_component_liveness) &&
			!this.module_registry.is_online(component_of_kwl(full_kwl))
		) {
			this.connection_pars.failure_handler({
				type: "access-denied",
				kwl: full_kwl,
				reason: `component ${component_of_kwl(full_kwl)} appears to be offline.`,
			});
		}
	}
}
function to_view_object(backing_store, kwl, desc) {
	switch (desc.container_type) {
		case 2:
			return new SubtreeArray(backing_store, kwl, desc);
		case 1:
			return desc.named_tables
				? new SubtreeNamedTable(backing_store, kwl, desc)
				: new SubtreeTable(backing_store, kwl, desc);
		case 0:
		case 5:
			return new Subtree(backing_store, kwl, desc);
	}
}
export class SubtreeArray {
	backing_store;
	kwl_basename;
	description;
	constructor(backing_store, kwl_basename, description) {
		this.backing_store = backing_store;
		this.kwl_basename = kwl_basename;
		this.description = description;
	}
	row(index) {
		if (index >= 0 && index < this.description.capacity) {
			return new Subtree(
				this.backing_store,
				`${this.kwl_basename}[${index}]`,
				this.description.contents,
			);
		}
		throw new Error(
			`Requested array element ${this.kwl_basename}[${index}] exceeds capacity of ${this.description.capacity}`,
		);
	}
	[Symbol.iterator]() {
		const N = this.description.capacity;
		const self = this;
		return (function* () {
			for (let i = 0; i < N; ++i) {
				yield self.row(i);
			}
		})();
	}
}
export class Subtree {
	backing_store;
	kwl;
	description;
	constructor(backing_store, kwl, description) {
		this.backing_store = backing_store;
		this.kwl = kwl;
		this.description = description;
		enforce(
			this.description.container_type === 5 ||
				this.description.data_type === "subtree",
		);
	}
	get parent() {
		switch (this.description.container_type) {
			case 5:
				return this.backing_store.root;
			default: {
				const parent_kwl = this.kwl.endsWith("]")
					? path_strip_trailing_index(this.kwl)
					: path_hd(this.kwl);
				return to_view_object(
					this.backing_store,
					parent_kwl,
					this.description.parent,
				);
			}
		}
	}
	get children() {
		return this.st_desc().children;
	}
	toString() {
		return `${this.kwl}@${this.backing_store.identify()}`;
	}
	to_full_kwl(kwl_relative) {
		return `${this.kwl}.${kwl_relative}`;
	}
	to_full_path(path_relative) {
		return {
			kwl:
				path_relative.kwl === undefined
					? this.kwl
					: this.to_full_kwl(path_relative.kwl),
			kw: path_relative.kw,
		};
	}
	st_desc() {
		return this.description.container_type === 5
			? this.description.contents
			: this.description;
	}
	find(relative_kwl) {
		const maybe_desc = V2.find_subtree_within(this.st_desc(), relative_kwl);
		if (maybe_desc === undefined) return undefined;
		return to_view_object(
			this.backing_store,
			this.to_full_kwl(relative_kwl),
			maybe_desc,
		);
	}
	query_cache(path) {
		return this.backing_store.query_cache(this.to_full_path(path));
	}
	query_cache_kwl(kwl) {
		return this.backing_store.query_cache_kwl(this.to_full_kwl(kwl));
	}
	write_unchecked(path, payload) {
		this.backing_store.write_unchecked(this.to_full_path(path), payload);
	}
	async read(path, opts) {
		return await this.backing_store.read(this.to_full_path(path), opts);
	}
	read_unchecked(path, timeout) {
		return this.backing_store.read_unchecked(this.to_full_path(path), timeout);
	}
	write_tree_unchecked(kwl, payload) {
		let payload_wrapped = payload;
		const segments = this.to_full_kwl(kwl).split(".");
		for (let i = segments.length - 1; i >= 0; i--) {
			payload_wrapped = { kwl: { [segments[i]]: payload_wrapped } };
		}
		this.backing_store.send([{ op: "tree", ...payload_wrapped }]);
	}
	async write(path, payload, opts) {
		await this.backing_store.write(this.to_full_path(path), payload, opts);
	}
	async wait_until(path, criterion, opts) {
		return await this.backing_store.wait_until(
			this.to_full_path(path),
			criterion,
			opts,
		);
	}
	watch_unchecked(path, handler, opts) {
		return this.backing_store.watch_unchecked(
			this.to_full_path(path),
			handler,
			opts,
		);
	}
	async watch(path, handler, opts) {
		return await this.backing_store.watch(
			this.to_full_path(path),
			handler,
			opts,
		);
	}
}
export async function deref(backing_store, path, _opts) {
	if (backing_store instanceof VSocket) {
		await backing_store.subscription_flush();
	}
	const target = await backing_store.read(path);
	if (!target) return null;
	const maybe_view_object = backing_store.root.find(target);
	if (
		maybe_view_object === undefined ||
		maybe_view_object instanceof SubtreeArray ||
		maybe_view_object instanceof SubtreeTable
	) {
		throw new Error(
			`Unable to dereference ${path.kwl}.${path.kw} -- doesn't point at a referenceable object`,
		);
	}
	return maybe_view_object;
}
export class Root {
	backing_store;
	constructor(backing_store) {
		this.backing_store = backing_store;
	}
	get components() {
		return this.backing_store.schema.keywords;
	}
	find(kwl) {
		const i = kwl.indexOf(".");
		const hd = i === -1 ? kwl : kwl.substring(0, i);
		for (const component of this.components) {
			if (component.contents.sys_name === hd) {
				if (i === -1) {
					return new Subtree(this.backing_store, hd, component);
				} else {
					const rest = kwl.substring(i + 1);
					const maybe_desc = V2.find_subtree_within(component.contents, rest);
					if (maybe_desc === undefined) return undefined;
					return to_view_object(this.backing_store, kwl, maybe_desc);
				}
			}
		}
		return undefined;
	}
}
export class SubtreeTable {
	backing_store;
	kwl;
	description;
	constructor(backing_store, kwl, description) {
		this.backing_store = backing_store;
		this.kwl = kwl;
		this.description = description;
		enforce(path_index(kwl) === undefined);
		enforce(this.description.container_type === 1);
	}
	get parent() {
		return new Subtree(
			this.backing_store,
			path_hd(this.kwl),
			this.description.parent,
		);
	}
	async allocated_indices(opts) {
		return await this.backing_store.table_indices({
			...opts,
			table_kwl: this.kwl,
		});
	}
	async is_allocated(index, opts) {
		return await this.backing_store.table_has_row({
			...(opts ?? {}),
			index,
			table_kwl: this.kwl,
		});
	}
	async rows(opts) {
		return (await this.allocated_indices(opts)).map((i) =>
			this.row_unchecked(i),
		);
	}
	row(index) {
		return this.row_unchecked(index);
	}
	row_unchecked(index) {
		return new SubtreeTableRow(
			this.backing_store,
			`${this.kwl}[${index}]`,
			this.description.contents,
		);
	}
	async *[Symbol.asyncIterator]() {
		const indices = await this.allocated_indices();
		const self = this;
		return (function* () {
			for (const i of indices) {
				yield self.row_unchecked(i);
			}
		})();
	}
}
export class SubtreeNamedTable extends SubtreeTable {
	constructor(backing_store, kwl, description) {
		super(backing_store, kwl, description);
		enforce(description.named_tables === true);
	}
	async create_row(opts) {
		const index = (
			await this.backing_store.table_create_row({
				...opts,
				table_kwl: this.kwl,
			})
		)[0];
		return this.row_unchecked(index);
	}
	async rows(opts) {
		return (await this.allocated_indices(opts)).map((i) =>
			this.row_unchecked(i),
		);
	}
	row(index, _) {
		void _;
		return this.row_unchecked(index);
	}
	row_unchecked(index) {
		return new SubtreeNamedTableRow(
			this.backing_store,
			`${this.kwl}[${index}]`,
			this.description.contents,
		);
	}
	async delete_all(opts) {
		await this.backing_store.table_delete_all_rows({
			table_kwl: this.kwl,
			...(opts ?? {}),
		});
	}
}
export class SubtreeTableRow extends Subtree {
	constructor(backing_store, kwl, description) {
		super(backing_store, kwl, description);
		enforce(description.parent.container_type === 1);
		enforce_nonnull(path_index(this.kwl));
	}
	get index() {
		return enforce_nonnull(path_index(this.kwl));
	}
}
export class SubtreeNamedTableRow extends SubtreeTableRow {
	constructor(backing_store, kwl, description) {
		super(backing_store, kwl, description);
		enforce(
			description.parent.container_type === 1 &&
				description.parent.named_tables === true,
		);
	}
	async rename(desired_name, opts) {
		await this.backing_store.table_rename_row({
			...opts,
			row_kwl: this.kwl,
			desired_name,
		});
	}
	async row_name(opts) {
		return await this.read({ kw: "row_name_status" }, opts);
	}
	async delete(opts) {
		await this.backing_store.table_delete_row({
			...opts,
			table_kwl: path_strip_trailing_index(this.kwl),
			index: this.index,
		});
	}
}
export var ComponentState;
(function (ComponentState) {
	ComponentState[(ComponentState["Unknown"] = 0)] = "Unknown";
	ComponentState[(ComponentState["Disabled"] = 1)] = "Disabled";
	ComponentState[(ComponentState["Uninitialized"] = 2)] = "Uninitialized";
	ComponentState[(ComponentState["Running"] = 3)] = "Running";
	ComponentState[(ComponentState["Crashed"] = 4)] = "Crashed";
})(ComponentState || (ComponentState = {}));
export class ModuleRegistry {
	static DISABLED = -2;
	data = {};
	sys_names = {};
	change_handlers = {};
	constructor(components, data) {
		for (const component of components) {
			this.sys_names[component.ua_name] = component.sys_name;
			enforce(component.owning_module !== undefined);
		}
		this.data = data;
	}
	on_change(f) {
		const id = StringIDPool.generate_id();
		this.change_handlers[id] = f;
		return new EventHandler(3, id, () => {
			delete this.change_handlers[id];
		});
	}
	static async initialize(components, socket, degraded_mode, context) {
		const data = {};
		for (const component of components) {
			data[component.sys_name] = {
				crashed: false,
				pid: -1,
				owning_module: component.owning_module,
				ua_name: component.ua_name,
			};
		}
		const mr = new ModuleRegistry(components, data);
		const mr_kwl = "arkona_internal_module_registry";
		const derive_state = (pid, crashed) => {
			if (pid === ModuleRegistry.DISABLED) {
				return ComponentState.Disabled;
			} else if (pid < 0) {
				return crashed ? ComponentState.Crashed : ComponentState.Uninitialized;
			} else {
				return crashed ? ComponentState.Crashed : ComponentState.Running;
			}
		};
		for (const component of components) {
			const pid_kw = `${component.sys_name}_pid`;
			socket.register_kw_listener(
				{
					listener_id: `ModuleRegistry:${pid_kw}`,
					kwl: mr_kwl,
					kw: pid_kw,
					listener_type: 0,
					execution_strategy: 0,
				},
				(pid) => {
					data[component.sys_name].pid = pid;
					for (const id in mr.change_handlers) {
						mr.change_handlers[id](
							component.sys_name,
							derive_state(
								data[component.sys_name].pid,
								data[component.sys_name].crashed,
							),
						);
					}
				},
			);
			const crashed_kw = `${component.sys_name}_crashed`;
			socket.register_kw_listener(
				{
					listener_id: `ModuleRegistry:${crashed_kw}`,
					kwl: mr_kwl,
					kw: crashed_kw,
					listener_type: 0,
					execution_strategy: 0,
				},
				(crashed) => {
					data[component.sys_name].crashed = crashed;
					for (const id in mr.change_handlers) {
						mr.change_handlers[id](
							component.sys_name,
							derive_state(
								data[component.sys_name].pid,
								data[component.sys_name].crashed,
							),
						);
					}
				},
			);
		}
		await new Promise((resolve, reject) => {
			let rejected = false;
			context.register_timeout_handler(() => {
				rejected = true;
				reject();
			});
			const event_handler = mr.on_change(() => {
				const missing_modules = new Set();
				for (const component_sys_name in data) {
					if (component_sys_name === "software_update") {
						continue;
					}
					const state = derive_state(
						data[component_sys_name].pid,
						data[component_sys_name].crashed,
					);
					if (
						state !== ComponentState.Running &&
						state !== ComponentState.Disabled &&
						!(state === ComponentState.Crashed && degraded_mode)
					) {
						missing_modules.add(data[component_sys_name].owning_module);
						context.desc = `while waiting for ${socket.identify()} to initialize the following modules: ${[
							...missing_modules,
						].join(", ")}`;
						break;
					}
				}
				if (missing_modules.size === 0 || (degraded_mode && !rejected)) {
					event_handler.unregister();
					context.unregister_timeout_handler(reject);
					resolve();
				}
			});
			try {
				socket.send([{ op: "readAll", kwl: mr_kwl }]);
			} catch (e) {
				rejected = true;
				reject(e);
			}
		});
		return mr;
	}
	is_online(component_sys_name) {
		return (
			this.data[component_sys_name].pid > 0 &&
			this.data[component_sys_name].crashed === false
		);
	}
	is_disabled(component_sys_name) {
		return this.data[component_sys_name].pid === ModuleRegistry.DISABLED;
	}
	get_status(component_sys_name) {
		return {
			pid: this.data[component_sys_name].pid,
			crashed: this.data[component_sys_name].crashed,
		};
	}
}
export class RuntimeConstants {
	data = {};
	constructor(data) {
		this.data = data;
	}
	get_constant(component_sys_name, constant_name) {
		if (
			!Object.prototype.hasOwnProperty.call(this.data, component_sys_name) ||
			!Object.prototype.hasOwnProperty.call(
				this.data[component_sys_name],
				constant_name,
			)
		) {
			throw new Error(
				`Unknown constant specifier {${component_sys_name}, ${constant_name}}`,
			);
		}
		return this.data[component_sys_name][constant_name];
	}
	equal(other) {
		const components_a = Object.keys(this.data).sort();
		const components_b = Object.keys(other.data).sort();
		if (components_a.length !== components_b.length) {
			return false;
		}
		for (let i = 0; i < components_a.length; ++i) {
			if (components_a[i] !== components_b[i]) {
				return false;
			}
			const component = components_a[i];
			const const_names_a = Object.keys(this.data[component]).sort();
			const const_names_b = Object.keys(other.data[component]).sort();
			if (const_names_a.length !== const_names_b.length) {
				return false;
			}
			for (let j = 0; j < const_names_a.length; ++j) {
				if (const_names_a[j] !== const_names_b[j]) {
					return false;
				}
				const const_name = const_names_a[j];
				if (
					this.data[component][const_name] !== other.data[component][const_name]
				) {
					return false;
				}
			}
		}
		return true;
	}
	static async initialize(schema, socket, timeout, context) {
		context.desc = "while waiting for runtime constants to initialize";
		const disabledComponentSysNames = new Set();
		const split = (orig_constant_specifier) => {
			const parts = orig_constant_specifier.split("::");
			enforce(parts.length === 2);
			const sysname_parts = enforce_nonnull(
				schema.constants[parts[0]][parts[1]],
			).path.split(".");
			enforce(sysname_parts.length === 2);
			return {
				component_sys_name: sysname_parts[0],
				constant_name: sysname_parts[1],
			};
		};
		if (socket.protocol_features.runtime_constants_json === true) {
			const raw_data = await download_json({
				protocol: socket.protocol === "ws" ? "http" : "https",
				reject_unauthorized: false,
				path: "/runtime_constants.json",
				ip: socket.ip,
				timeout,
			});
			for (const component of schema.keywords) {
				for (const constant of component.bound_to ?? []) {
					const { component_sys_name, constant_name } = split(constant);
					const b = enforce_nonnull(
						raw_data[component_sys_name][constant_name],
					);
					enforce(typeof b === "boolean");
					if (!b) {
						disabledComponentSysNames.add(component.sys_name);
						break;
					}
				}
			}
			return [new RuntimeConstants(raw_data), disabledComponentSysNames];
		}
		const data = {};
		const todo = {};
		const find_missing_constants = (component_sys_name) => {
			for (const c of schema.keywords) {
				if (c.sys_name === component_sys_name) {
					return (c.bound_to ?? []).map(split);
				}
			}
			throw new Error(`Unknown component ${component_sys_name}`);
		};
		for (const component_ua_name in schema.constants) {
			if (
				schema.keywords.find((comp) => {
					return comp.ua_name === component_ua_name;
				}) === undefined
			) {
				continue;
			}
			const component_sys_name = (() => {
				for (const c in schema.constants[component_ua_name]) {
					return schema.constants[component_ua_name][c].path.split(".")[0];
				}
				unreachable();
			})();
			data[component_sys_name] = {};
			const constants = Object.keys(schema.constants[component_ua_name]);
			enforce(constants.length > 0);
			todo[component_sys_name] = {
				missing_constants: find_missing_constants(component_sys_name),
				own_constants: constants,
			};
		}
		while (Object.keys(todo).length > 0) {
			const work_queue = [];
			const old_keys = Object.keys(todo);
			for (const component_sys_name of old_keys) {
				let verdict = "read";
				for (const mc of todo[component_sys_name].missing_constants) {
					enforce(verdict === "read");
					const cval = data?.[mc.component_sys_name]?.[mc.constant_name];
					if (cval === undefined) {
						verdict = "wait";
						break;
					} else {
						enforce(typeof cval === "boolean");
						if (cval === false) {
							verdict = "discard";
							break;
						}
					}
				}
				switch (verdict) {
					case "wait":
						break;
					case "discard":
						disabledComponentSysNames.add(component_sys_name);
						delete todo[component_sys_name];
						break;
					case "read": {
						const kwl = `${component_sys_name}.runtime_constants`;
						for (const constant_name of todo[component_sys_name]
							.own_constants) {
							work_queue.push(
								socket
									.read_unchecked({ kwl, kw: constant_name }, timeout)
									.then((val) => {
										data[component_sys_name][constant_name] = val;
									}),
							);
						}
						delete todo[component_sys_name];
					}
				}
			}
			await Promise.all(work_queue);
		}
		return [new RuntimeConstants(data), disabledComponentSysNames];
	}
}
function is_stored_named_table(x) {
	return "named-rows" in x;
}
export class VSettings {
	schema;
	build_info;
	m_identifier;
	m_data;
	listener_registry = new ListenerRegistry((err) =>
		setTimeout(() => {
			throw err;
		}, 0),
	);
	open_handlers = new Map();
	constructor(schema, build_info, json, m_identifier) {
		this.schema = schema;
		this.build_info = build_info;
		this.m_identifier = m_identifier;
		const bail = (msg) => {
			const suffix = msg ? `: ${msg}` : "";
			throw new Error(
				`The provided settings object does not conform to the expected file format${suffix}`,
			);
		};
		const is_record = (y) => {
			return y !== null && typeof y === "object";
		};
		if (!is_record(json)) {
			bail("should be a JSON dictionary");
		}
		if (!Object.hasOwnProperty.call(json, "format")) {
			bail("should have a `format` field ");
		}
		const expected_format = "V__matrix settings.json v1";
		if (json["format"] !== expected_format) {
			bail(
				`unknown format ${json["format"]} (expected to find ${expected_format})`,
			);
		}
		if (
			!Object.hasOwnProperty.call(json, "header") ||
			!is_record(json["header"])
		) {
			bail("should have a dictionary-valued `header` field");
		}
		if (
			!Object.hasOwnProperty.call(json["header"], "fpga") ||
			typeof json["header"]["fpga"] !== "string"
		) {
			bail('data["header"] should have a string-valued `fpga` field');
		}
		if (
			!Object.hasOwnProperty.call(json["header"], "version") ||
			!Array.isArray(json["header"]["version"]) ||
			json["header"]["version"].length !== 2 ||
			typeof json["header"]["version"][0] !== "string" ||
			typeof json["header"]["version"][1] !== "string"
		) {
			bail(
				'data["header"] should have a `version` field containing a two-string array [versionstring, build_date]',
			);
		}
		if (
			!Object.hasOwnProperty.call(json["header"], "flags") ||
			!Array.isArray(json["header"]["flags"]) ||
			json["header"]["flags"].findIndex((x) => typeof x !== "string") !== -1
		) {
			bail(
				'data["header"] should have a (possibly empty) string array called `flags`',
			);
		}
		if (!Object.hasOwnProperty.call(json, "components")) {
			bail("there should be a `components` dictionary holding the saved state");
		}
		const validate_keyword = (path, kw) => {
			if (
				!Object.hasOwnProperty.call(kw, "isDefault") ||
				typeof kw["isDefault"] !== "boolean"
			) {
				bail(
					`components.${path} should contain a boolean-valued isDefault field`,
				);
			}
			if (!Object.hasOwnProperty.call(kw, "data")) {
				bail(`components.${path} should contain a field named data`);
			}
		};
		const validate_named_table = (kwl_so_far, nt) => {
			const p = (i) => `components.${kwl_so_far}[${i}]`;
			for (let i = 0; i < nt["named-rows"].length; ++i) {
				const row = nt["named-rows"][i];
				if (typeof row.idx !== "number") {
					bail(`${p(i)} should have a numeric idx field`);
				}
				if (typeof row.id !== "string") {
					bail(`${p(i)} should have a string-valued id field`);
				}
				validate_subtree(`${kwl_so_far}[${i}]`, row);
			}
		};
		const validate_subtree = (kwl_so_far, st) => {
			for (const kw in st.kw ?? {}) {
				validate_keyword(`${kwl_so_far}.${kw}`, st.kw[kw]);
			}
			for (const kwl in st.kwl ?? {}) {
				if (is_stored_named_table(st.kwl[kwl])) {
					validate_named_table(`${kwl_so_far}.${kwl}`, st.kwl[kwl]);
				} else {
					validate_subtree(`${kwl_so_far}.${kwl}`, st.kwl[kwl]);
				}
			}
		};
		for (const sys_name in json["components"]) {
			validate_subtree(sys_name, json["components"][sys_name]);
		}
		this.m_data = json;
	}
	identify() {
		const flag_suffix =
			this.m_data.header.flags.length === 0
				? ""
				: `?${this.m_data.header.flags.join("&")}`;
		if (this.m_identifier) {
			return `${this.m_identifier} [settings.json${flag_suffix}, ${this.m_data.header.version[0]}/${this.m_data.header.fpga}]`;
		} else {
			return `settings.json${flag_suffix}, ${this.m_data.header.version[0]}/${this.m_data.header.fpga}`;
		}
	}
	get root() {
		return new Root(this);
	}
	is_ready() {
		return true;
	}
	marker() {
		return new Promise((resolve) => resolve());
	}
	find_subtree(kwl) {
		const segments = kwl.split(".");
		let cur_level = this.m_data.components[segments[0]];
		if (cur_level === undefined) return undefined;
		for (let i = 1; i < segments.length; ++i) {
			const segment = segments[i];
			if (Object.hasOwnProperty.call(cur_level.kwl, segment)) {
				cur_level = cur_level.kwl[segment];
				enforce(!("named-rows" in cur_level));
				continue;
			}
			if (
				segment.endsWith("]") &&
				Object.hasOwnProperty.call(
					cur_level.kwl,
					path_strip_trailing_index(segment),
				)
			) {
				const nt = cur_level.kwl[path_strip_trailing_index(segment)];
				enforce(is_stored_named_table(nt));
				const idx = path_index(segment);
				const maybe_row = nt["named-rows"].find((row) => row.idx === idx);
				if (maybe_row !== undefined) {
					cur_level = maybe_row;
					continue;
				}
			}
			return undefined;
		}
		return cur_level;
	}
	find_keyword(path) {
		const st = this.find_subtree(path.kwl);
		if (
			st === undefined ||
			is_stored_named_table(st) ||
			!Object.hasOwnProperty.call(st.kw, path.kw)
		) {
			return undefined;
		}
		return st.kw[path.kw];
	}
	query_cache(path) {
		const maybe_stored_kw = this.find_keyword(path);
		if (maybe_stored_kw === undefined) {
			return MissingData;
		}
		return [0, maybe_stored_kw.data];
	}
	query_cache_kwl(kwl) {
		const st = this.find_subtree(kwl);
		if (st === undefined) return MissingData;
		const result = new Map();
		if (!is_stored_named_table(st)) {
			for (const kw in st.kw) {
				result.set(kw, [0, st.kw[kw].data]);
			}
		}
		return [0, result];
	}
	get_runtime_constant(constant_string) {
		const parts = constant_string.split("::");
		enforce(parts.length === 2);
		const kwl = `${snake_case(parts[0])}.runtime_constants`;
		const kw = this.find_keyword({ kwl, kw: parts[1] });
		if (kw === undefined) {
			throw new Error(`Unknown runtime constant '${constant_string}'`);
		}
		enforce(typeof kw.data === "number" || typeof kw.data === "boolean");
		return kw.data;
	}
	register_kw_listener(pars, handler) {
		return this.listener_registry.register_kw_listener(pars, handler);
	}
	register_kwl_listener(pars, handler) {
		return this.listener_registry.register_kwl_listener(pars, handler);
	}
	register_global_listener(handler, pars) {
		return this.listener_registry.register_global_listener(handler, pars);
	}
	on_open(f, suggested_id) {
		const id = StringIDPool.generate_or_use_id(
			this.open_handlers,
			suggested_id,
		);
		this.open_handlers.set(id, f);
		return new EventHandler(0, id, () => {
			this.open_handlers.delete(id);
		});
	}
	write_unchecked(path, payload) {
		this.send([
			{
				op: "data",
				kwl: path.kwl,
				kw: { [path.kw]: payload },
			},
		]);
	}
	async write(path, payload, _) {
		void _;
		this.write_unchecked(path, payload);
	}
	send(requests) {
		for (const request of requests) {
			switch (request.op) {
				case "data":
					for (const kw in request.kw) {
						const x = this.find_keyword({
							kwl: request.kwl,
							kw: kw,
						});
						if (x === undefined)
							throw new Error(
								`This settings file contains no keyword named ${request.kwl}.${kw}`,
							);
						x.data = request.kw[kw];
					}
					break;
				case "flags":
				case "marker":
				case "readAll":
				case "subscribe":
				case "unsubscribe":
					break;
				case "tree":
					throw new Error("TODO");
			}
		}
	}
	async read_unchecked(path) {
		const maybe_data = this.query_cache(path);
		if (maybe_data === MissingData)
			throw new Error(
				`This settings file contains no data for ${path.kwl}.${path.kw}`,
			);
		return maybe_data[1];
	}
	async read(full_path, opts) {
		void opts;
		return await this.read_unchecked(full_path);
	}
	async table_create_row(opts) {
		void opts;
		throw new Error("TODO");
	}
	async table_rename_row(opts) {
		void opts;
		throw new Error("TODO");
	}
	async table_indices(opts) {
		void opts;
		throw new Error("TODO");
	}
	async table_has_row(opts) {
		void opts;
		throw new Error("TODO");
	}
	async table_delete_row(opts) {
		void opts;
		throw new Error("TODO");
	}
	async table_delete_all_rows(opts) {
		void opts;
		throw new Error("TODO");
	}
	async wait_until(full_path, criterion, opts) {
		void full_path;
		void criterion;
		void opts;
		throw new Error("TODO");
	}
	watch_unchecked(full_path, handler, opts) {
		const listener = this.register_kw_listener(
			{
				...full_path,
				execution_strategy: 0,
				listener_type: 0,
				ensure_initial_read: opts?.ensure_initial_read ?? false,
			},
			handler,
		);
		return new Watcher(full_path, () => {
			listener.unregister();
		});
	}
	async watch(full_path, handler, opts) {
		return this.watch_unchecked(full_path, handler, opts);
	}
}
//# sourceMappingURL=data:application/json;base64,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
