import { enforce } from "./utilities.js";
export class UnorderedArrayImpl {
	data = [];
	register(interval, handler) {
		this.data.push([interval[0], interval[1], handler]);
	}
	find(key) {
		const result = [];
		for (const el of this.data) {
			if (el[0] <= key && el[1] >= key) {
				result.push(el[2]);
			}
		}
		return result;
	}
}
export class OrderedArrayImpl {
	data = [];
	register(interval, handler) {
		this.data.push([interval[0], interval[1], handler]);
		this.data.sort((a, b) => a[0] - b[0]);
	}
	find(key) {
		const result = [];
		if (this.data.length !== 0 && this.data[0][0] <= key) {
			for (let i = 0; i < this.data.length; ++i) {
				if (this.data[i][0] <= key && this.data[i][1] >= key) {
					result.push(this.data[i][2]);
				} else if (this.data[i][0] > key) {
					break;
				}
			}
		}
		return result;
	}
}
export class OrderedTypedArrayImpl {
	sentinel;
	intervals = new Uint32Array(3 * 8);
	handlers = [];
	dummy_handler = (_, __) => {};
	compaction_timer = null;
	max_interval_size = 0;
	constructor(sentinel) {
		this.sentinel = sentinel;
		this.intervals.fill(this.sentinel);
	}
	get size() {
		return this.handlers.length;
	}
	grow() {
		let new_intervals = new Uint32Array(this.intervals.length * 2);
		new_intervals.fill(this.sentinel);
		new_intervals.set(this.intervals);
		this.intervals = new_intervals;
	}
	find_i(key) {
		if (this.intervals[3 * (this.size - 1)] <= key) return this.size;
		if (this.intervals[0] > key) return 0;
		let i_lower = 0;
		let i_upper = this.size;
		while (i_lower + 1 < i_upper) {
			const i_mid = (i_lower + i_upper) >> 1;
			if (this.intervals[3 * i_mid] <= key) {
				i_lower = i_mid;
			} else {
				i_upper = i_mid;
			}
		}
		return i_upper;
	}
	delete(key, h) {
		for (let i = this.find_i(key) - 1; i >= 0; --i) {
			if (this.handlers[this.intervals[3 * i + 2]] === h) {
				this.handlers[this.intervals[3 * i + 2]] = this.dummy_handler;
				this.intervals.copyWithin(3 * i, 3 * i + 3, 3 * this.size + 3);
				if (this.compaction_timer) {
					clearTimeout(this.compaction_timer);
				}
				this.compaction_timer = setTimeout(() => {
					this.compact();
					this.compaction_timer = null;
				}, 200);
				break;
			} else if (this.intervals[3 * i] < key) {
				break;
			}
		}
	}
	compact() {
		let max_interval_size = 0;
		const deltas = [{ index: 0, deficit: 0 }];
		const new_handlers = [];
		let cur_deficit = 0;
		for (let i = 0; i < this.handlers.length; ++i) {
			if (this.handlers[i] === this.dummy_handler) {
				cur_deficit += 1;
				deltas.push({ index: i + 1, deficit: cur_deficit });
			} else {
				new_handlers.push(this.handlers[i]);
			}
		}
		for (let i = 0; this.intervals[i] !== this.sentinel; i += 3) {
			max_interval_size = Math.max(
				max_interval_size,
				this.intervals[i + 1] - this.intervals[i],
			);
			const handler_index = this.intervals[i + 2];
			let cur_deficit = 0;
			for (const { index, deficit } of deltas) {
				if (index > handler_index) break;
				cur_deficit = deficit;
			}
			enforce(handler_index >= cur_deficit);
			this.intervals[i + 2] -= cur_deficit;
		}
		this.max_interval_size = max_interval_size;
		this.handlers = new_handlers;
	}
	print() {
		let s = "[";
		for (let i = 0; i < this.size; ++i) {
			if (i > 0) {
				s += ", ";
			}
			s += `(${this.intervals[3 * i]}, ${this.intervals[3 * i + 1]}, ${this.intervals[3 * i + 2]})`;
		}
		console.log(s + "]");
	}
	register(interval, handler) {
		enforce(interval[0] > 0);
		enforce(interval[1] >= interval[0]);
		if (3 * (this.size + 2) >= this.intervals.length) {
			this.grow();
			enforce(3 * (this.size + 2) < this.intervals.length);
		}
		const i_insert = this.find_i(interval[0]);
		this.intervals.copyWithin(
			3 * (i_insert + 1),
			3 * i_insert,
			3 * this.size + 3,
		);
		this.intervals[3 * i_insert] = interval[0];
		this.intervals[3 * i_insert + 1] = interval[1];
		this.intervals[3 * i_insert + 2] = this.size;
		this.handlers.push(handler);
		this.max_interval_size = Math.max(
			this.max_interval_size,
			interval[1] - interval[0],
		);
		return {
			unregister: () => {
				this.delete(interval[0], handler);
			},
		};
	}
	find(id) {
		let result = [];
		for (
			let i =
				id > this.max_interval_size + 1
					? 3 * this.find_i(id - this.max_interval_size - 1)
					: 0;
			this.intervals[i] <= id;
			i += 3
		) {
			if (this.intervals[i + 1] >= id) {
				result.push(this.handlers[this.intervals[i + 2]]);
			}
		}
		return result;
	}
	invoke(id, payload) {
		for (
			let i =
				id > this.max_interval_size + 1
					? 3 * this.find_i(id - this.max_interval_size - 1)
					: 0;
			this.intervals[i] <= id;
			i += 3
		) {
			if (this.intervals[i + 1] >= id) {
				this.handlers[this.intervals[i + 2]](payload, id);
			}
		}
	}
}
export class OrderedTypedSOAImpl {
	sentinel;
	mins = new Uint32Array(8);
	maxs = new Uint32Array(8);
	handler_ids = new Uint32Array(8);
	handlers = [];
	dummy_handler = (_, __) => {};
	compaction_timer = null;
	max_interval_size = 0;
	constructor(sentinel) {
		this.sentinel = sentinel;
		this.mins.fill(this.sentinel);
	}
	get size() {
		return this.handlers.length;
	}
	grow() {
		let new_mins = new Uint32Array(this.mins.length * 2);
		new_mins.fill(this.sentinel);
		new_mins.set(this.mins);
		this.mins = new_mins;
		let new_maxs = new Uint32Array(this.maxs.length * 2);
		new_maxs.fill(this.sentinel);
		new_maxs.set(this.maxs);
		this.maxs = new_maxs;
		let new_handler_ids = new Uint32Array(this.handler_ids.length * 2);
		new_handler_ids.set(this.handler_ids);
		this.handler_ids = new_handler_ids;
	}
	find_i(key) {
		if (this.mins[this.size - 1] <= key) return this.size;
		if (this.mins[0] > key) return 0;
		let i_lower = 0;
		let i_upper = this.size;
		while (i_lower + 1 < i_upper) {
			const i_mid = (i_lower + i_upper) >> 1;
			if (this.mins[i_mid] <= key) {
				i_lower = i_mid;
			} else {
				i_upper = i_mid;
			}
		}
		return i_upper;
	}
	delete(key, h) {
		for (let i = this.find_i(key) - 1; i >= 0; --i) {
			if (this.handlers[this.handler_ids[i]] === h) {
				this.handlers[this.handler_ids[i]] = this.dummy_handler;
				this.mins.copyWithin(i, i + 1, this.size + 1);
				this.maxs.copyWithin(i, i + 1, this.size + 1);
				this.handler_ids.copyWithin(i, i + 1, this.size + 1);
				if (this.compaction_timer) {
					clearTimeout(this.compaction_timer);
				}
				this.compaction_timer = setTimeout(() => {
					this.compact();
					this.compaction_timer = null;
				}, 200);
				break;
			} else if (this.mins[i] < key) {
				break;
			}
		}
	}
	compact() {
		let max_interval_size = 0;
		const deltas = [{ index: 0, deficit: 0 }];
		const new_handlers = [];
		let cur_deficit = 0;
		for (let i = 0; i < this.handlers.length; ++i) {
			if (this.handlers[i] === this.dummy_handler) {
				cur_deficit += 1;
				deltas.push({ index: i + 1, deficit: cur_deficit });
			} else {
				new_handlers.push(this.handlers[i]);
			}
		}
		for (let i = 0; this.mins[i] !== this.sentinel; i++) {
			max_interval_size = Math.max(
				max_interval_size,
				this.maxs[i] - this.mins[i],
			);
			const handler_index = this.handler_ids[i];
			let cur_deficit = 0;
			for (const { index, deficit } of deltas) {
				if (index > handler_index) break;
				cur_deficit = deficit;
			}
			enforce(handler_index >= cur_deficit);
			this.handler_ids[i] -= cur_deficit;
		}
		this.max_interval_size = max_interval_size;
		this.handlers = new_handlers;
	}
	print() {
		console.log(
			"mins:",
			this.mins,
			"maxs:",
			this.maxs,
			"handler_ids:",
			this.handler_ids,
		);
	}
	register(interval, handler) {
		enforce(interval[0] > 0);
		enforce(interval[1] >= interval[0]);
		if (this.size + 2 >= this.mins.length) {
			this.grow();
			enforce(this.size + 2 < this.mins.length);
		}
		const i_insert = this.find_i(interval[0]);
		this.mins.copyWithin(i_insert + 1, i_insert, this.size + 1);
		this.maxs.copyWithin(i_insert + 1, i_insert, this.size + 1);
		this.handler_ids.copyWithin(i_insert + 1, i_insert, this.size + 1);
		this.mins[i_insert] = interval[0];
		this.maxs[i_insert] = interval[1];
		this.handler_ids[i_insert] = this.size;
		this.handlers.push(handler);
		this.max_interval_size = Math.max(
			this.max_interval_size,
			interval[1] - interval[0],
		);
		return {
			unregister: () => {
				this.delete(interval[0], handler);
			},
		};
	}
	find(id) {
		let result = [];
		for (
			let i =
				id > this.max_interval_size + 1
					? this.find_i(id - this.max_interval_size - 1)
					: 0;
			this.mins[i] <= id;
			i++
		) {
			if (this.maxs[i] >= id) {
				result.push(this.handlers[this.handler_ids[i]]);
			}
		}
		return result;
	}
	invoke(id, payload) {
		for (
			let i =
				id > this.max_interval_size + 1
					? this.find_i(id - this.max_interval_size - 1)
					: 0;
			this.mins[i] <= id;
			i++
		) {
			if (this.maxs[i] >= id) {
				this.handlers[this.handler_ids[i]](payload, id);
			}
		}
	}
}
//# sourceMappingURL=data:application/json;base64,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
