import { enforce, enforce_nonnull } from "./utilities.js";
import { keyword_iter_sync } from "./schema_reflection.js";
export function unpack(payload, issue_descriptions) {
	const issues = [];
	let num_errors = 0;
	let num_warnings = 0;
	let num_total = 0;
	for (let i = 0, rem = payload; rem !== 0; i++, rem >>= 1) {
		if ((rem & 1) !== 0) {
			num_total++;
			const issue = issue_descriptions[i];
			issues.push(issue);
			switch (issue.severity) {
				case "Alert":
				case "Critical":
				case "Emergency":
				case "Error":
					num_errors++;
					break;
				case "Warning":
					num_warnings++;
					break;
			}
		}
	}
	return {
		num_errors,
		num_warnings,
		num_total,
		issues,
		raw: payload,
	};
}
export function diff(prev, cur, issue_descriptions) {
	if (prev === null) {
		return {
			lowered: [],
			raised: cur.issues,
		};
	}
	const intersection_raw = cur.raw & prev.raw;
	const raised_raw = cur.raw ^ intersection_raw;
	const lowered_raw = prev.raw ^ intersection_raw;
	return {
		raised: unpack(raised_raw, issue_descriptions).issues,
		lowered: unpack(lowered_raw, issue_descriptions).issues,
	};
}
export class Tracker {
	vsocket;
	m_handler;
	m_watchers = [];
	m_cur_issues = new Map();
	constructor(vsocket, m_handler) {
		this.vsocket = vsocket;
		this.m_handler = m_handler;
		keyword_iter_sync({
			backing_store: vsocket,
			handler: (_, desc, kwl) => {
				if (
					desc.contents.container_type === 0 &&
					desc.contents.data_type === "alerts"
				) {
					const alerts = desc.contents.alerts;
					const kw = desc.contents.sys_name;
					this.m_watchers.push(
						vsocket.watch_unchecked({ kwl, kw }, (payload) => {
							const prev = this.get(kwl, kw) ?? null;
							const current = unpack(payload, alerts);
							const { raised, lowered } = diff(prev, current, alerts);
							this.m_handler(kwl, kw, { current, raised, lowered });
							if (!this.m_cur_issues.has(kwl))
								this.m_cur_issues.set(kwl, new Map());
							enforce_nonnull(this.m_cur_issues.get(kwl)).set(kw, current);
						}),
					);
				}
				return "recurse";
			},
		});
	}
	get(kwl, kw) {
		return this.m_cur_issues.get(kwl)?.get(kw);
	}
	get_all() {
		const result = [];
		for (const [kwl, x] of this.m_cur_issues) {
			for (const [kw, unpacked] of x) {
				if (unpacked.num_total !== 0) {
					result.push([{ kwl, kw }, unpacked]);
				}
			}
		}
		return result;
	}
	close() {
		this.m_watchers.forEach((w) => w.unwatch());
		this.m_watchers.splice(0, this.m_watchers.length);
		enforce(this.m_watchers.length === 0);
	}
}
//# sourceMappingURL=data:application/json;base64,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
