import { StringIDPool } from "./id_pool.js";
import { GlobalListener, KeywordListener, KWLListener } from "./pervasives.js";
import { enforce_nonnull } from "./utilities.js";
export class ListenerRegistry {
	m_error_handler;
	global_listeners = new Map();
	kwl_listeners = new Map();
	lazy_handler_interval = null;
	pending_lazy_updates = { kwls: new Map(), kws: new Map() };
	constructor(m_error_handler) {
		this.m_error_handler = m_error_handler;
		this.lazy_handler_interval = setInterval(() => {
			this.execute_lazy_handlers();
		}, 98);
	}
	dump_stats() {
		let n_complete = 0;
		let n_partial = 0;
		const multi = new Set();
		for (const [kwl, { complete, partial }] of this.kwl_listeners) {
			n_complete += Object.keys(complete).length;
			for (const kw_name in partial) {
				const n = Object.keys(partial[kw_name]).length;
				n_partial += n;
				if (n > 1 && multi.size < 20) multi.add(`${kwl}/${kw_name}`);
			}
		}
		console.log(
			`Have ${this.global_listeners.size} global listeners, ${this.kwl_listeners.size} kwl listeners (${n_complete}/${n_partial})`,
		);
		console.log(
			`got multiple partial subscriptions for ${[...multi].join(", ")}`,
		);
	}
	close() {
		clearInterval(enforce_nonnull(this.lazy_handler_interval));
	}
	prune_listeners(keep_if) {
		this.execute_lazy_handlers();
		for (const [listener_id, _] of this.global_listeners) {
			if (!keep_if({ listener_id })) this.global_listeners.delete(listener_id);
		}
		for (const [kwl, { complete, partial }] of this.kwl_listeners) {
			for (const id in complete) {
				if (!keep_if({ kwl, listener_id: id })) delete complete[id];
			}
			for (const kw_name in partial) {
				for (const id in partial[kw_name]) {
					if (!keep_if({ kwl, listener_id: id, kw: kw_name }))
						delete partial[kw_name][id];
				}
				if (Object.keys(partial[kw_name]).length === 0) delete partial[kw_name];
			}
			if (
				Object.keys(complete).length === 0 &&
				Object.keys(partial).length === 0
			)
				this.kwl_listeners.delete(kwl);
		}
	}
	execute_lazy_handlers() {
		for (const [listener_id, handler] of this.pending_lazy_updates.kwls) {
			void listener_id;
			try {
				handler();
			} catch (e) {
				this.m_error_handler(e);
			}
		}
		for (const [listener_id, entry] of this.pending_lazy_updates.kws) {
			void listener_id;
			try {
				entry[0](entry[1]);
			} catch (e) {
				this.m_error_handler(e);
			}
		}
		this.pending_lazy_updates.kwls.clear();
		this.pending_lazy_updates.kws.clear();
	}
	safely_delete_handler(pars) {
		if (
			pars.kwl.includes("live_view") &&
			(this.pending_lazy_updates.kwls.size !== 0 ||
				this.pending_lazy_updates.kws.size !== 0)
		)
			debugger;
		const entry = this.kwl_listeners.get(pars.kwl);
		if (!entry) return;
		if (pars.kw !== undefined) {
			if (Object.prototype.hasOwnProperty.call(entry.partial, pars.kw)) {
				delete entry.partial[pars.kw][pars.id];
				if (Object.keys(entry.partial[pars.kw]).length === 0) {
					delete entry.partial[pars.kw];
				}
			}
		} else {
			delete entry.complete[pars.id];
		}
		if (
			Object.keys(entry.complete).length === 0 &&
			Object.keys(entry.partial).length === 0
		) {
			this.kwl_listeners.delete(pars.kwl);
		}
		if (this.pending_lazy_updates.kwls.has(pars.id)) {
			this.execute_lazy_handlers();
		}
	}
	do_get_listeners(kwl) {
		if (!this.kwl_listeners.has(kwl)) {
			this.kwl_listeners.set(kwl, {
				complete: Object.create(null),
				partial: Object.create(null),
			});
		}
		return this.kwl_listeners.get(kwl);
	}
	have_permanent_listener(pars) {
		const entry = this.kwl_listeners.get(pars.kwl);
		if (!entry) return false;
		if (pars.kw === undefined) {
			for (const id in entry.complete) {
				if (entry.complete[id].permanent) return true;
			}
		} else {
			for (const id in entry.partial[pars.kw]) {
				if (entry.partial[pars.kw][id].permanent) return true;
			}
		}
		return false;
	}
	register_kw_listener(pars, handler) {
		const partial_listeners = this.do_get_listeners(pars.kwl).partial;
		if (!Object.prototype.hasOwnProperty.call(partial_listeners, pars.kw)) {
			partial_listeners[pars.kw] = Object.create(null);
		}
		const listeners = partial_listeners[pars.kw];
		const id = StringIDPool.generate_or_use_id(listeners, pars.listener_id);
		listeners[id] = {
			handler:
				pars.listener_type === 0
					? handler
					: (payload) => {
							handler(payload);
							this.safely_delete_handler({ ...pars, id });
						},
			execution_strategy: pars.execution_strategy,
			permanent: pars.listener_type === 0,
		};
		return new KeywordListener(pars.kwl, pars.kw, id, new WeakRef(this));
	}
	register_kwl_listener(pars, handler) {
		const listeners = this.do_get_listeners(pars.kwl).complete;
		const id = StringIDPool.generate_or_use_id(listeners, pars.listener_id);
		listeners[id] = {
			handler:
				pars.listener_type === 0
					? handler
					: () => {
							handler();
							this.safely_delete_handler({ ...pars, id });
						},
			execution_strategy: pars.execution_strategy,
			permanent: pars.listener_type === 0,
		};
		return new KWLListener(pars.kwl, id, new WeakRef(this));
	}
	register_global_listener(handler, pars) {
		const id = StringIDPool.generate_or_use_id(
			this.global_listeners,
			pars?.listener_id,
		);
		this.global_listeners.set(id, handler);
		const ref_to_self = new WeakRef(this);
		return new GlobalListener(id, (self) => {
			ref_to_self.deref()?.global_listeners.delete(self.id);
		});
	}
}
//# sourceMappingURL=data:application/json;base64,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
