import { ListenerRegistry } from "./listener_registry.js";
import { WSCloseEvent, WSErrorEvent } from "./polyfills/polyfill_ws.js";
import { Duration } from "./time.js";
import { Tagged } from "./utilities.js";
export type SysName = Tagged<string, "sysname">;
export type KeywordDataType =
	| "bool"
	| "int"
	| "float"
	| "float duration"
	| "int duration"
	| "string"
	| "timestamp"
	| "ipaddress"
	| "enum"
	| "timecode"
	| "ref"
	| "device_tree_node";
export type DataType = "subtree" | "atomic subtree" | KeywordDataType;
export type KeywordType =
	| "event"
	| "driver-owned event"
	| "status"
	| "command"
	| "duplex";
export type BranchName = Tagged<string, "branch">;
export type TargetTypeIdentifier = Tagged<string, "typename">;
export type PathScope = "full" | "relative";
export type KWLName<PathScope extends "full" | "relative"> = Tagged<
	string,
	PathScope
>;
export declare const MissingData: unique symbol;
export type MaybeMissing<T> = T | typeof MissingData;
export type Path<ps extends PathScope> = ps extends "full"
	? {
			kwl: KWLName<"full">;
			kw: SysName;
		}
	: {
			kwl?: KWLName<"relative">;
			kw: SysName;
		};
export declare function to_full_kwl(
	this_kwl: null | KWLName<"full">,
	relative_kwl?: KWLName<"relative">,
): KWLName<"full">;
export declare function component_of_kwl(kwl: KWLName<"full">): string;
export type KeywordPriority = "coalescing" | "immediate" | "immediate-tracking";
export interface CommonKeywordProperties<Stage extends "raw" | "annotated"> {
	kw_type: Stage extends "raw" ? KeywordType | "virtual" : KeywordType;
	kw_priority?: KeywordPriority;
	persistent: boolean;
}
export interface CommonConstantProperties<DataType> {
	data_type: DataType;
	path: string;
}
export interface ConstantBoolDescription
	extends CommonConstantProperties<"bool"> {}
export interface ConstantIntDescription
	extends CommonConstantProperties<"int"> {
	min: number;
	max: number;
}
export interface ConstantStringDescription
	extends CommonConstantProperties<"string"> {
	maxlength: number;
}
export interface ConstantDurationDescription
	extends CommonConstantProperties<"int duration" | "float duration"> {
	min: number;
	max: number;
}
export interface ConstantIPAddressDescription
	extends CommonConstantProperties<"ipaddress"> {}
export interface ConstantTimestampDescription
	extends CommonConstantProperties<"timestamp"> {}
export type ConstantDescription =
	| ConstantBoolDescription
	| ConstantIntDescription
	| ConstantStringDescription
	| ConstantDurationDescription
	| ConstantIPAddressDescription
	| ConstantTimestampDescription;
export type ElementCount<Stage extends "raw" | "annotated"> =
	Stage extends "raw" ? number | string : number;
export type SubtreeMultiplicityProperties<Stage extends "raw" | "annotated"> =
	| {}
	| {
			array_size: ElementCount<Stage>;
	  }
	| {
			table_size: ElementCount<Stage>;
			named_rows: boolean;
	  };
export type ConstantsDictionary = {
	[component_name: string]: {
		[constant_name: string]: ConstantDescription;
	};
};
export interface CommonCommandOptions {
	check_component_liveness?: boolean;
	timeout?: Duration;
}
export type WaitUntilOptions = CommonCommandOptions & {
	skip_initial_read?: boolean;
};
export interface DeviceTreeNodePayload {
	node: string;
	offset: number;
	count: number;
}
export interface TimecodePayload {
	hours: number;
	minutes: number;
	seconds: number;
	frames: number;
	raw: [number, number];
}
export type KeywordPayload =
	| null
	| boolean
	| string
	| number
	| DeviceTreeNodePayload
	| TimecodePayload
	| KeywordPayload[]
	| {
			[array_element_index: number]: KeywordPayload;
	  }
	| [number, KeywordPayload];
export type MaybeSparseArray<T> =
	| T[]
	| {
			[i: string]: T;
	  };
export type KeywordWritePayload =
	| KeywordPayload
	| {
			[i: string]: KeywordPayload;
	  };
export interface SubtreePayload {
	kw: {
		[name: string]: KeywordPayload;
	};
	kwl: {
		[name: string]: SubtreePayload;
	};
}
export interface CommonWriteOptions extends CommonCommandOptions {
	check_keyword_type?: boolean;
	check_towel?: boolean;
	retry_interval?: Duration;
	first_write_interval?: Duration;
}
export type WriteValidator<T> =
	| {
			criterion: "status";
			validator: (status: T) => boolean;
	  }
	| {
			criterion: "custom";
			validator: () => Promise<boolean>;
	  }
	| {
			criterion: "return-immediately";
	  };
export type WriteOptions<T> = CommonWriteOptions & {
	retry_until?: WriteValidator<T>;
};
export type RowMask = string;
export type RowView<S> = S & {
	readonly index: number;
};
export declare class Watcher {
	readonly path: Path<"full">;
	private readonly destructor;
	constructor(path: Path<"full">, destructor: () => void);
	unwatch(): void;
}
export type IDHandler = (payload: KeywordPayload, id: number) => void;
export interface IDIntervalListener {
	unregister(): void;
}
export interface IDIntervalMap<T> {
	register(interval: [number, number], handler: IDHandler): T;
	find(keyword_id: number): IDHandler[];
}
export declare const enum ExecutionStrategy {
	Immediate = 0,
	Lazy = 1,
}
export declare const enum KeywordListenerType {
	Permanent = 0,
	OneShot = 1,
}
export type ListenerID = UniqueID<string, "listener-id">;
export interface Listener {
	unregister: () => void;
}
export declare class GlobalListener implements Listener {
	readonly id: ListenerID;
	private readonly destructor;
	constructor(id: ListenerID, destructor: (self: GlobalListener) => void);
	unregister(): void;
}
export declare class KWLListener implements Listener {
	readonly kwl: KWLName<"full">;
	readonly id: ListenerID;
	private readonly listener_registry;
	constructor(
		kwl: KWLName<"full">,
		id: ListenerID,
		listener_registry: WeakRef<ListenerRegistry>,
	);
	unregister(): void;
}
export declare class KeywordListener implements Listener {
	readonly kwl: KWLName<"full">;
	readonly kw: SysName;
	readonly id: ListenerID;
	readonly listener_registry: WeakRef<ListenerRegistry>;
	constructor(
		kwl: KWLName<"full">,
		kw: SysName,
		id: ListenerID,
		listener_registry: WeakRef<ListenerRegistry>,
	);
	unregister(): void;
}
export declare const enum EventHandlerType {
	Open = 0,
	Close = 1,
	Error = 2,
	ComponentStateChange = 3,
}
export declare class EventHandler {
	readonly handler_type: EventHandlerType;
	readonly id: UniqueID<string, "event-handler-id">;
	private readonly destructor;
	constructor(
		handler_type: EventHandlerType,
		id: UniqueID<string, "event-handler-id">,
		destructor: () => void,
	);
	unregister(): void;
}
export type GlobalKeywordHandler = (data: {
	op: "data";
	kwl: KWLName<"full">;
	kw: {
		[sys_name: string]: KeywordPayload;
	};
}) => void;
export type Request =
	| {
			op: "subscribe" | "unsubscribe" | "readAll";
			kwl: KWLName<"full">[] | KWLName<"full">;
			kw?: string[] | string;
	  }
	| {
			op: "data";
			kwl: KWLName<"full">;
			kw: {
				[kw_name: string]: KeywordPayload;
			};
	  }
	| {
			op: "tree";
			kwl: SubtreePayload;
	  }
	| {
			op: "readAll";
			kwl: KWLName<"full">[] | KWLName<"full">;
			kw: string[] | string;
	  }
	| {
			op: "flags";
			flags: {
				"report errors"?: boolean;
			};
	  }
	| {
			op: "marker";
			marker: number;
	  };
export interface BuildInfo {
	bsp_commit: string;
	built_by: string;
	changelog: string;
	commit: string;
	config: string;
	hardware_model?: string;
	timestamp: string;
	version: string;
	vapi_version?: string;
}
export interface WebServerBuildInfo {
	date: string;
	time: string;
}
export type KeywordHandler<T = KeywordPayload> = (value: T) => void;
export type KWLHandler = () => void;
export interface IListenerRegistry {
	register_kw_listener<T extends KeywordPayload = KeywordPayload>(
		pars: {
			kwl: KWLName<"full">;
			kw: SysName;
			listener_id?: ListenerID;
			cache?: boolean;
			listener_type: KeywordListenerType;
			execution_strategy: ExecutionStrategy;
			ensure_initial_read?: boolean;
		},
		handler: KeywordHandler<T>,
	): KeywordListener;
	register_kwl_listener(
		pars: {
			kwl: KWLName<"full">;
			listener_id?: ListenerID;
			cache?: boolean;
			listener_type: KeywordListenerType;
			execution_strategy: ExecutionStrategy;
			ensure_initial_read?: boolean;
		},
		handler: KWLHandler,
	): KWLListener;
	register_global_listener(
		handler: GlobalKeywordHandler,
		pars?: {
			listener_id?: ListenerID;
		},
	): GlobalListener;
}
export type Generation = Tagged<number, "generation">;
export type Generational<T> = [gen: Generation, payload: T];
export type UniqueID<T, tag extends string> = Tagged<T, `__tag-${tag}`>;
export interface WebSocketParameters {
	ip?: string;
	towel?: string;
	protocol?: "ws" | "wss";
	port?: number;
	cache_everything?: boolean;
	reject_unauthorized?: boolean;
	max_payload_size?: number;
	login?: {
		user: string;
		password: string;
	} | null;
}
export interface InternalWebSocketParameters {
	timeout: Duration;
	on_first_msg: (first_message: string) => void;
	on_further_msg: (msg: MessageEvent) => void;
	on_close: (ev: WSCloseEvent) => void;
	reject: (err: Error) => void;
	on_error: (ev: WSErrorEvent) => void;
}
export declare class TimeoutContext {
	private state;
	desc: null | string;
	constructor(state: {
		timed_out: boolean;
		timeout_handlers: Array<() => void>;
	});
	get timed_out(): boolean;
	register_timeout_handler(f: () => void): void;
	unregister_timeout_handler(f: () => void): void;
}
export declare function with_timeout<T>(
	timeout: Duration,
	handler: (context: TimeoutContext) => Promise<T>,
): Promise<T>;
