import { enforce_nonnull } from "./utilities.js";
export const MissingData = Symbol("missing");
export function to_full_kwl(this_kwl, relative_kwl) {
	if (relative_kwl !== undefined) {
		const prefix = this_kwl === null ? "" : `${this_kwl}.`;
		return `${prefix}${relative_kwl}`;
	} else {
		return enforce_nonnull(this_kwl);
	}
}
export function component_of_kwl(kwl) {
	const i = kwl.indexOf(".");
	if (i !== -1) {
		return kwl.substring(0, i);
	}
	return kwl;
}
export class Watcher {
	path;
	destructor;
	constructor(path, destructor) {
		this.path = path;
		this.destructor = destructor;
	}
	unwatch() {
		this.destructor();
	}
}
export var ExecutionStrategy;
(function (ExecutionStrategy) {
	ExecutionStrategy[(ExecutionStrategy["Immediate"] = 0)] = "Immediate";
	ExecutionStrategy[(ExecutionStrategy["Lazy"] = 1)] = "Lazy";
})(ExecutionStrategy || (ExecutionStrategy = {}));
export var KeywordListenerType;
(function (KeywordListenerType) {
	KeywordListenerType[(KeywordListenerType["Permanent"] = 0)] = "Permanent";
	KeywordListenerType[(KeywordListenerType["OneShot"] = 1)] = "OneShot";
})(KeywordListenerType || (KeywordListenerType = {}));
export class GlobalListener {
	id;
	destructor;
	constructor(id, destructor) {
		this.id = id;
		this.destructor = destructor;
	}
	unregister() {
		this.destructor(this);
	}
}
export class KWLListener {
	kwl;
	id;
	listener_registry;
	constructor(kwl, id, listener_registry) {
		this.kwl = kwl;
		this.id = id;
		this.listener_registry = listener_registry;
	}
	unregister() {
		this.listener_registry
			.deref()
			?.safely_delete_handler({ kwl: this.kwl, id: this.id });
	}
}
export class KeywordListener {
	kwl;
	kw;
	id;
	listener_registry;
	constructor(kwl, kw, id, listener_registry) {
		this.kwl = kwl;
		this.kw = kw;
		this.id = id;
		this.listener_registry = listener_registry;
	}
	unregister() {
		this.listener_registry
			.deref()
			?.safely_delete_handler({ kwl: this.kwl, kw: this.kw, id: this.id });
	}
}
export var EventHandlerType;
(function (EventHandlerType) {
	EventHandlerType[(EventHandlerType["Open"] = 0)] = "Open";
	EventHandlerType[(EventHandlerType["Close"] = 1)] = "Close";
	EventHandlerType[(EventHandlerType["Error"] = 2)] = "Error";
	EventHandlerType[(EventHandlerType["ComponentStateChange"] = 3)] =
		"ComponentStateChange";
})(EventHandlerType || (EventHandlerType = {}));
export class EventHandler {
	handler_type;
	id;
	destructor;
	constructor(handler_type, id, destructor) {
		this.handler_type = handler_type;
		this.id = id;
		this.destructor = destructor;
	}
	unregister() {
		this.destructor();
	}
}
export class TimeoutContext {
	state;
	desc = null;
	constructor(state) {
		this.state = state;
	}
	get timed_out() {
		return this.state.timed_out;
	}
	register_timeout_handler(f) {
		this.state.timeout_handlers.push(f);
	}
	unregister_timeout_handler(f) {
		const remaining = [];
		for (const g of this.state.timeout_handlers) {
			if (f !== g) remaining.push(g);
		}
		this.state.timeout_handlers = remaining;
	}
}
export function with_timeout(timeout, handler) {
	return new Promise((resolve, reject) => {
		const internal_state = {
			timed_out: false,
			timeout_handlers: [],
		};
		const context = new TimeoutContext(internal_state);
		const timer = setTimeout(() => {
			const suffix = context.desc === null ? "" : ` (${context.desc})`;
			internal_state.timed_out = true;
			for (const f of internal_state.timeout_handlers.reverse()) {
				f();
			}
			reject(new Error(`Timeout after ${timeout.ms()} ms${suffix}`));
		}, timeout.ms());
		handler(context)
			.then((success) => {
				clearTimeout(timer);
				if (!internal_state.timed_out) {
					resolve(success);
				}
			})
			.catch((err) => {
				clearTimeout(timer);
				if (!internal_state.timed_out) {
					reject(err);
				}
			});
	});
}
//# sourceMappingURL=data:application/json;base64,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
