import { get as http_get } from "http";
import { get as https_get } from "https";
import { gunzipSync } from "zlib";
import { enforce } from "../utilities.js";
export function download(pars, kind) {
	const finish_buffer = (x) => {
		if (x instanceof Buffer)
			return (kind ?? "string") !== "string" ? x : x.toString("utf-8");
		else {
			enforce(typeof x === "string");
			return (kind ?? "string") === "string" ? x : Buffer.from(x, "utf-8");
		}
	};
	let data = undefined;
	const port = pars.port ?? (pars.protocol === "http" ? 80 : 443);
	return new Promise((resolve, reject) => {
		const cb = (res) => {
			if (res.statusCode === 200) {
				res
					.on("data", (chunk) => {
						if (data === undefined) {
							data = chunk;
						} else {
							enforce(typeof data === typeof chunk);
							if (typeof data === "string") {
								res.setEncoding("utf8");
								data += chunk;
							} else {
								enforce(data instanceof Buffer);
								data = Buffer.concat([data, chunk]);
							}
						}
					})
					.on("end", () => {
						if ((res.headers["content-encoding"] ?? "") == "gzip") {
							enforce(data instanceof Buffer);
							resolve(finish_buffer(gunzipSync(data)));
						} else if (data instanceof Buffer) {
							resolve(finish_buffer(data));
						} else {
							enforce(typeof data == "string");
							resolve(finish_buffer(data));
						}
					});
			} else {
				let msg =
					res.statusMessage ??
					(res.statusCode ? `Error ${res.statusCode}` : "");
				if (msg.length > 0) {
					msg = ": " + msg;
				}
				reject(
					`Unable to download ${pars.protocol}://${pars.ip}:${port}/${pars.path}${msg}`,
				);
			}
		};
		const common_pars = {
			hostname: pars.ip,
			port: port,
			protocol: pars.protocol + ":",
			path: pars.path.startsWith("/") ? pars.path : `/${pars.path}`,
			headers: { "Accept-Encoding": "gzip" },
			timeout: pars.timeout.ms(),
			...(pars.protocol === "http"
				? {}
				: {
						rejectUnauthorized: pars.reject_unauthorized ?? true,
					}),
		};
		const inject_error_handlers = (req) => {
			const prefix = `Unable to download ${pars.protocol}://${pars.ip}:${port}/${pars.path}`;
			req
				.on("error", (err) => reject(`${prefix}: ${err.message}`))
				.on("close", () => reject(`${prefix}: Connection unexpectedly closed`))
				.on("abort", () =>
					reject(`${prefix}: Connection aborted unexpectedly`),
				);
		};
		switch (pars.protocol) {
			case "http":
				inject_error_handlers(http_get(common_pars, cb));
				break;
			case "https":
				inject_error_handlers(
					https_get(
						{
							...common_pars,
							rejectUnauthorized: pars.reject_unauthorized ?? true,
						},
						cb,
					),
				);
				break;
			default:
				enforce(false);
		}
	});
}
export async function download_json(pars) {
	return JSON.parse(await download(pars));
}
//# sourceMappingURL=data:application/json;base64,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
