import { digest } from "./polyfills/polyfill_digest.js";
import { Subtree, SubtreeNamedTable } from "./data_views.js";
import {
	children_of_raw_atomic_subtree,
	children_of_raw_subtree,
	kernel_of_child,
} from "./schema_v2.js";
import { Duration } from "./time.js";
import {
	asyncIter,
	enforce,
	enforce_eq,
	enforce_nonnull,
	path_hd,
	unreachable,
	WorkQueue,
} from "./utilities.js";
export async function subtree_iter(pars) {
	const wq = pars.work_queue ?? new WorkQueue({ num_workers: 128 });
	const worker = async (kwl, desc) => {
		const verdict = pars.on_subtree
			? await pars.on_subtree(new Subtree(pars.backing_store, kwl, desc))
			: "recurse";
		if (verdict === "recurse") {
			for (const child of desc.children) {
				const kernel = kernel_of_child(child);
				if (kernel.data_type !== "subtree") continue;
				switch (child.container_type) {
					case 4:
						unreachable();
					case 0:
						await worker(`${kwl}.${kernel.sys_name}`, kernel);
						break;
					case 2:
						enforce_eq(child.contents.container_type, 0);
						for (let i = 0; i < child.capacity; ++i) {
							wq.push(async () => {
								await worker(`${kwl}.${kernel.sys_name}[${i}]`, kernel);
							});
						}
						break;
					case 1:
						enforce_eq(child.contents.container_type, 0);
						{
							const table_kwl = `${kwl}.${kernel.sys_name}`;
							const verdict =
								pars.on_named_table && child.named_tables
									? await pars.on_named_table(
											new SubtreeNamedTable(
												pars.backing_store,
												table_kwl,
												child,
											),
										)
									: "recurse";
							if (verdict === "do-not-recurse") break;
							if ((pars.include ?? "allocated") === "allocated") {
								const allocated_indices =
									await pars.backing_store.table_indices({
										table_kwl,
										check_component_liveness: pars.check_component_liveness,
									});
								for (const i of allocated_indices) {
									wq.push(async () => {
										await worker(`${table_kwl}[${i}]`, kernel);
									});
								}
							} else {
								enforce(pars.include === "everything");
								for (let i = 0; i < child.capacity; ++i) {
									wq.push(async () => {
										await worker(`${table_kwl}[${i}]`, kernel);
									});
								}
							}
						}
				}
			}
		}
	};
	for (const component of pars.backing_store.schema.keywords) {
		if (component.enabled || pars.include_disabled === true) {
			wq.push(async () => {
				await worker(component.contents.sys_name, component.contents);
			});
		}
	}
	await wq.drain();
}
export function subtree_iter_sync(pars) {
	const worker = (kwl, desc) => {
		if (pars.handler(desc, kwl) === "recurse") {
			for (const child of desc.children) {
				const kernel = kernel_of_child(child);
				if (kernel.data_type !== "subtree") continue;
				switch (child.container_type) {
					case 4:
						unreachable();
					case 0:
						worker(`${kwl}.${kernel.sys_name}`, kernel);
						break;
					case 2:
						enforce_eq(child.contents.container_type, 0);
						for (let i = 0; i < child.capacity; ++i) {
							worker(`${kwl}.${kernel.sys_name}[${i}]`, kernel);
						}
						break;
					case 1:
						enforce_eq(child.contents.container_type, 0);
						const table_kwl = `${kwl}.${kernel.sys_name}`;
						enforce(pars.include === "everything");
						for (let i = 0; i < child.capacity; ++i) {
							worker(`${table_kwl}[${i}]`, kernel);
						}
				}
			}
		}
	};
	for (const component of pars.backing_store.schema.keywords) {
		if (component.enabled || pars.include_disabled === true) {
			worker(component.contents.sys_name, component.contents);
		}
	}
}
export async function subtree_iter_typewise(backing_store, pars) {
	await subtree_iter({
		...pars,
		backing_store,
		include: pars.include ?? "allocated",
		on_subtree: async (st) => {
			enforce(!Array.isArray(st.description));
			if (st.description.type_identifier === pars.type_identifier) {
				await pars.handler(pars.lift(st), st.kwl);
			}
			return "recurse";
		},
	});
}
export async function keyword_iter(pars) {
	const include_rowname_keywords = pars.include_rowname_keywords ?? false;
	const wq = pars.work_queue ?? new WorkQueue({ num_workers: 32 });
	await subtree_iter({
		...pars,
		include: pars.include ?? "allocated",
		on_subtree: async (st) => {
			for (const child of st.children) {
				if (child.container_type === 4) {
					await pars.handler(pars.backing_store, child, st.kwl);
				}
			}
			if (include_rowname_keywords) {
				const maybe_rnd = maybe_rowname_desc(
					st.description.container_type === 5
						? st.description.contents
						: st.description,
				);
				if (maybe_rnd)
					wq.push(() =>
						pars.handler(pars.backing_store, maybe_rnd, path_hd(st.kwl)),
					);
			}
			return "recurse";
		},
	});
	await wq.drain();
}
function maybe_rowname_desc(st) {
	if (st.parent.container_type === 1 && st.parent.named_tables) {
		const kw_desc = {
			container_type: 4,
			kw_type: "duplex",
			parent: st,
			persistent: true,
			contents: {
				container_type: 0,
				data_type: "string",
				max_length: 32,
				optional: false,
				sys_name: "row_name",
				parent: undefined,
			},
		};
		kw_desc.contents.parent = kw_desc;
		return kw_desc;
	}
	return null;
}
export function keyword_iter_sync(pars) {
	const include_rowname_keywords = pars.include_rowname_keywords ?? false;
	const st_handler = (st, kwl) => {
		for (const child of st.children) {
			if (child.container_type === 4) {
				pars.handler(pars.backing_store, child, kwl);
			}
		}
		if (include_rowname_keywords) {
			const maybe_rnd = maybe_rowname_desc(st);
			if (maybe_rnd) pars.handler(pars.backing_store, maybe_rnd, path_hd(kwl));
		}
		return "recurse";
	};
	subtree_iter_sync({
		...pars,
		include: pars.include ?? "everything",
		handler: (x, kwl) =>
			st_handler(x.container_type === 5 ? x.contents : x, kwl),
	});
}
export async function branch_iter(pars) {
	const branchWorker = async (branch, desc) => {
		const verdict = await pars.handler(branch, desc);
		if (verdict === "recurse") {
			const kernel = kernel_of_child(desc);
			if (kernel.data_type !== "subtree") {
				unreachable();
			}
			for (const child of kernel.children) {
				const kernel = kernel_of_child(child);
				if (kernel.data_type === "subtree") {
					enforce(child.container_type !== 4);
					enforce(child.container_type !== 5);
					await branchWorker(`${branch}.${kernel.sys_name}`, child);
				}
			}
		}
	};
	for (const component of pars.backing_store.schema.keywords) {
		if (component.enabled || pars.include_disabled === true) {
			await branchWorker(component.contents.sys_name, component.contents);
		}
	}
}
export function branch_iter_sync(pars) {
	const branchWorker = (branch, desc) => {
		const verdict = pars.handler(branch, desc);
		if (verdict === "recurse") {
			const kernel = kernel_of_child(desc);
			if (kernel.data_type !== "subtree") {
				unreachable();
			}
			for (const child of kernel.children) {
				const kernel = kernel_of_child(child);
				if (kernel.data_type === "subtree") {
					enforce(child.container_type !== 4);
					enforce(child.container_type !== 5);
					branchWorker(`${branch}.${kernel.sys_name}`, child);
				}
			}
		}
	};
	for (const component of pars.backing_store.schema.keywords) {
		if (component.enabled || pars.include_disabled === true) {
			branchWorker(component.contents.sys_name, component.contents);
		}
	}
}
function do_reset_subtree_to_defaults(
	backing_store,
	kwl,
	st,
	wq,
	named_tables,
	pars,
) {
	const recursive = pars?.recursive ?? true;
	const get_default = (desc) => {
		if (desc.optional) {
			return null;
		}
		switch (desc.container_type) {
			case 3:
				return [];
			case 0:
				switch (desc.data_type) {
					case "device_tree_node":
						throw new Error(`dtnode keywords are read-only`);
					case "atomic subtree":
						return desc.children.map((child) => get_default(child));
					case "alerts":
						return 0;
					default:
						return desc.default_value ?? null;
				}
			case 2: {
				const per_element = get_default(desc.contents);
				return Array.from({ length: desc.capacity }, () => per_element);
			}
		}
	};
	const unroll = (kwl, cst) => {
		switch (cst.container_type) {
			case 0:
				enforce(cst.data_type === "subtree");
				do_reset_subtree_to_defaults(
					backing_store,
					kwl,
					cst,
					wq,
					named_tables,
					pars,
				);
				break;
			case 2:
				for (let i = 0; i < cst.capacity; ++i) {
					unroll(`${kwl}[${i}]`, cst.contents);
				}
				break;
			case 1:
				{
					if (cst.named_tables && cst.capacity > 0) {
						named_tables.push(kwl);
					}
					wq.push(async () => {
						const indices = await backing_store.table_indices({
							table_kwl: kwl,
						});
						for (const i of indices) {
							unroll(`${kwl}[${i}]`, cst.contents);
						}
					});
				}
				break;
		}
	};
	for (const child of st.children) {
		const kernel = kernel_of_child(child);
		switch (child.container_type) {
			case 4:
				let maybe_kw = null;
				switch (child.kw_type) {
					case "command":
						maybe_kw = kernel.sys_name;
						break;
					case "duplex":
						maybe_kw = kernel.sys_name + "_command";
						break;
				}
				if (maybe_kw !== null) {
					const kw = maybe_kw;
					const payload = get_default(child.contents);
					wq.push(async () => {
						await backing_store
							.write({ kwl, kw }, payload, {
								timeout: new Duration(3, "s"),
								retry_interval: new Duration(50, "ms"),
							})
							.catch((_) => {});
					});
				}
				break;
			default:
				if (recursive) unroll(`${kwl}.${kernel.sys_name}`, child);
		}
		if (recursive && kernel.data_type === "subtree") {
			unroll(`${kwl}.${kernel.sys_name}`, child);
		}
	}
}
export async function reset_subtree_to_defaults(backing_store, kwl, st, pars) {
	const recursive = pars?.recursive ?? true;
	const wq = pars?.wq ?? new WorkQueue({ num_workers: 32 });
	const named_tables = [];
	do_reset_subtree_to_defaults(backing_store, kwl, st, wq, named_tables, {
		recursive,
	});
	if (pars?.drain_wq ?? true) await wq.drain();
	await asyncIter(named_tables, async (kwl) => {
		await backing_store.write({ kwl, kw: "table_cmd" }, "DELETE_ALL_ROWS", {
			check_component_liveness: false,
			check_keyword_type: false,
			retry_until: {
				criterion: "custom",
				validator: async () =>
					(await backing_store.table_indices({ table_kwl: kwl })).length === 0,
			},
		});
	});
}
function variants_of_raw_variant(schema, desc) {
	if (desc.variants !== undefined) return desc.variants;
	const td = enforce_nonnull(
		schema.typedefs[enforce_nonnull(desc.type_identifier)],
	);
	enforce(td.data_type === "variant");
	return enforce_nonnull(td.variants);
}
function* iter_atomic_properties(schema) {
	function* add_subtree(branch_name, bound_to, upstream_multiplicities, desc) {
		const enclosing_subtree_multiplicities = (() => {
			if ("array_size" in desc) {
				return [...upstream_multiplicities, { array_size: desc.array_size }];
			} else if ("table_size" in desc) {
				return [
					...upstream_multiplicities,
					{ table_size: desc.table_size, named_rows: desc.named_rows },
				];
			}
			return upstream_multiplicities;
		})();
		for (const child of children_of_raw_subtree(schema, desc)) {
			if (child.data_type === "subtree") {
				for (const x of add_subtree(
					branch_name + "." + desc.sys_name,
					bound_to,
					enclosing_subtree_multiplicities,
					child,
				)) {
					yield x;
				}
			} else {
				const keyword_properties = child.persistent
					? {
							kw_type: child.kw_type,
							persistent: true,
							atomic: child,
						}
					: {
							kw_type: child.kw_type,
							persistent: false,
							sys_name: child.sys_name,
						};
				const enclosing_subtree_properties = {
					bound_to,
					branch_name,
					total_multiplicity: enclosing_subtree_multiplicities,
				};
				yield { enclosing_subtree_properties, keyword_properties };
			}
		}
	}
	for (const component of schema.keywords) {
		for (const x of add_subtree(
			component.sys_name,
			component.bound_to ?? [],
			[],
			component,
		)) {
			yield x;
		}
	}
}
export async function compute_persistence_hash(schema) {
	let s = "";
	const add = (property_name, value) => {
		s += property_name;
		s += typeof value === "string" ? value : JSON.stringify(value);
	};
	for (const {
		enclosing_subtree_properties,
		keyword_properties,
	} of iter_atomic_properties(schema)) {
		add("bound_to", enclosing_subtree_properties.bound_to);
		add(
			"enclosing_subtree_multiplicities",
			enclosing_subtree_properties.total_multiplicity,
		);
		add("kw_type", keyword_properties.kw_type);
		add("persistent", keyword_properties.persistent);
		if (keyword_properties.persistent) {
			const field_worker = (field) => {
				add("atomic_multiplicity", field.atomic_multiplicity);
				add("data_type", field.data_type);
				add("sys_name", field.sys_name);
				switch (field.data_type) {
					case "alerts":
						add("alerts", field.alerts);
						break;
					case "enum":
						add("enums", field.enum_values);
						break;
					case "float duration":
					case "float":
					case "int duration":
					case "int":
					case "timestamp":
						add("min", field.min);
						add("max", field.max);
						break;
					case "string":
						add("maxlength", field.max_length);
						break;
					case "ref":
						add("ref_properties", {
							ref_perm: field.ref_perm,
							target_type: field.target_type,
							target_type_identifier: field.target_type_identifier,
						});
						break;
					case "atomic subtree":
						for (const child of children_of_raw_atomic_subtree(schema, field)) {
							field_worker(child);
						}
						break;
					case "variant":
						for (const [constructor_name, variant] of variants_of_raw_variant(
							schema,
							field,
						)) {
							add("constructor_name", constructor_name);
							field_worker(variant);
						}
						break;
					case "bool":
					case "ipaddress":
						break;
					default:
						console.log(field);
						unreachable();
				}
			};
			field_worker(keyword_properties.atomic);
		} else {
			add("sys_name", keyword_properties.sys_name);
		}
	}
	return await digest(s);
}
function is_record(x) {
	return typeof x === "object" && x !== null;
}
function equal(x, y) {
	const aprops_seen = new Set();
	const worker = (a, b) => {
		if (is_record(a)) {
			if (!is_record(b)) return false;
			const rec_a = a;
			const rec_b = b;
			for (const k in rec_a) {
				if (a === x) {
					if (aprops_seen.has(k)) continue;
					aprops_seen.add(k);
				}
				if (rec_b[k] === undefined) return false;
				if (!worker(rec_a[k], rec_b[k])) return false;
			}
			for (const k in rec_b) {
				if (rec_a[k] === undefined) return false;
			}
			return true;
		}
		if (Array.isArray(a)) {
			if (!Array.isArray(b)) return false;
			const arr_a = a;
			const arr_b = a;
			if (arr_a.length !== arr_b.length) return false;
			for (let i = 0; i < arr_a.length; ++i) {
				if (!worker(arr_a[i], arr_b[i])) return false;
			}
			return true;
		}
		return JSON.stringify(a) === JSON.stringify(b);
	};
	return worker(x, y);
}
function diff_recursively(a, b, pars) {
	const result = {};
	for (const key in a) {
		if (key === "genlock") debugger;
		if (
			(pars?.ignored_properties?.has(key) ?? false) &&
			pars?.cur_key !== "children"
		)
			continue;
		if (b[key] === undefined) result[key] = { kind: "removed", was: a[key] };
		else {
			if (
				is_record(a[key]) &&
				is_record(b[key]) &&
				!(pars?.dontrecurse?.has(key) ?? false)
			) {
				const as_record = (x) => {
					if (key === "children") {
						const result = {};
						enforce(Array.isArray(x));
						for (const child of x) {
							result[enforce_nonnull(child.sys_name)] = child;
						}
						return result;
					}
					return x;
				};
				const changes = diff_recursively(as_record(a[key]), as_record(b[key]), {
					...pars,
					cur_key: key,
				});
				if (Object.keys(changes).length > 0)
					result[key] = {
						kind: "record-changed",
						changes,
					};
			} else if (!equal(a[key], b[key])) {
				result[key] = { kind: "scalar-changed", was: a[key], is: b[key] };
			}
		}
	}
	for (const key in b) {
		if (a[key] === undefined && !(pars?.ignored_properties?.has(key) ?? false))
			result[key] = { kind: "added", is: b[key] };
	}
	return result;
}
export function extract_persistent_parts(schema) {
	const subtree_worker = (st) => {
		const remaining_children = [];
		for (const child of children_of_raw_subtree(schema, st)) {
			if (child.data_type === "subtree") {
				const maybe_st = subtree_worker(child);
				if (maybe_st) remaining_children.push(maybe_st);
			} else if (child.persistent) remaining_children.push(child);
		}
		return remaining_children.length === 0
			? null
			: { ...st, children: remaining_children };
	};
	const remaining_components = [];
	for (const component of schema.keywords) {
		const maybe_rest = subtree_worker(component);
		if (maybe_rest) remaining_components.push(maybe_rest);
	}
	const referenced_typedefs = new Set();
	const add_typedef = (td_identifier) => {
		if (!referenced_typedefs.has(td_identifier)) {
			const td = schema.typedefs[td_identifier];
			if (!td) return;
			referenced_typedefs.add(td_identifier);
			switch (td.data_type) {
				case "subtree":
				case "atomic subtree":
					include_typedefs(td);
			}
		}
	};
	const include_typedefs = (st) => {
		if (st.type_identifier) add_typedef(st.type_identifier);
		for (const child of st.data_type === "subtree"
			? children_of_raw_subtree(schema, st)
			: children_of_raw_atomic_subtree(schema, st)) {
			switch (child.data_type) {
				case "subtree":
					include_typedefs(child);
					break;
				case "atomic subtree":
					include_typedefs(child);
					break;
				default:
					if (child.type_identifier) {
						add_typedef(child.type_identifier);
					}
			}
		}
	};
	remaining_components.forEach((comp) => include_typedefs(comp));
	const typedefs = {};
	for (const td of referenced_typedefs) {
		typedefs[td] = enforce_nonnull(schema.typedefs[td]);
	}
	return {
		...schema,
		typedefs,
		keywords: remaining_components,
	};
}
export function reify_typedefs(schema) {
	const st_worker = (st) => {
		const all_children =
			st.data_type === "subtree"
				? children_of_raw_subtree(schema, st)
				: children_of_raw_atomic_subtree(schema, st);
		return {
			...st,
			children: all_children.map((child) => {
				switch (child.data_type) {
					case "subtree":
					case "atomic subtree":
						return st_worker(child);
					default:
						return child;
				}
			}),
		};
	};
	return { ...schema, keywords: schema.keywords.map(st_worker) };
}
export function schema_diff(a, b, pars) {
	const kw_dict = (schema) => {
		const result = {};
		for (const component of schema.keywords) {
			result[component.sys_name] = component;
		}
		return result;
	};
	debugger;
	return {
		constants: diff_recursively(a.constants, b.constants, pars),
		typedefs: diff_recursively(a.typedefs, b.typedefs, pars),
		keywords: diff_recursively(kw_dict(a), kw_dict(b), pars),
	};
}
//# sourceMappingURL=data:application/json;base64,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
