import * as V1 from "./schema_v1.js";
function convert_schema(schema) {
	const atomic_multiplicity_of = (m, optional) => {
		const atomic_multiplicity = [];
		if ("vector_size" in m) {
			atomic_multiplicity.push({ type: "vector", capacity: m.vector_size });
		}
		if ("array_size" in m) {
			atomic_multiplicity.push({ type: "array", capacity: m.array_size });
		}
		if (optional) {
			atomic_multiplicity.push({ type: "optional" });
		}
		return atomic_multiplicity;
	};
	const convert_atomic_field = (af) => {
		const atomic_multiplicity = atomic_multiplicity_of(af, af.optional);
		const tmp = (() => {
			switch (af.data_type) {
				case "atomic subtree":
					return {
						...af,
						children: V1.children_of_raw_atomic_subtree(schema, af).map(
							convert_atomic_field,
						),
						atomic_multiplicity,
					};
				default:
					return {
						...af,
						atomic_multiplicity,
					};
			}
		})();
		delete tmp["optional"];
		return tmp;
	};
	const convert_keyword = (kw) => {
		const atomic_multiplicity = atomic_multiplicity_of(kw, kw.optional);
		switch (kw.data_type) {
			case "atomic subtree":
				return {
					...kw,
					children: V1.children_of_raw_atomic_subtree(schema, kw).map(
						convert_atomic_field,
					),
					atomic_multiplicity,
				};
			default:
				return {
					...kw,
					atomic_multiplicity,
				};
		}
	};
	const convert_typedef = (td) => {
		switch (td.data_type) {
			case "subtree":
				return {
					...td,
					children: V1.children_of_raw_subtree(schema, td).map((child) => {
						switch (child.data_type) {
							case "subtree":
								return convert_subtree(child);
							default:
								return convert_keyword(child);
						}
					}),
				};
			case "atomic subtree":
				return {
					...td,
					atomic_multiplicity: atomic_multiplicity_of(td, td.optional),
					children: V1.children_of_raw_atomic_subtree(schema, td).map(
						convert_atomic_field,
					),
				};
			default:
				return {
					...td,
					atomic_multiplicity: atomic_multiplicity_of(td, td.optional),
				};
		}
	};
	const convert_subtree = (st) => {
		return {
			...st,
			children: V1.children_of_raw_subtree(schema, st).map((child) => {
				switch (child.data_type) {
					case "subtree":
						return convert_subtree(child);
					default:
						return convert_keyword(child);
				}
			}),
		};
	};
	const convert_component = (comp) => {
		const children = [];
		for (const child of V1.children_of_raw_subtree(schema, comp)) {
			switch (child.data_type) {
				case "subtree":
					children.push(convert_subtree(child));
					break;
				default:
					children.push(convert_keyword(child));
			}
		}
		return {
			...comp,
			children,
		};
	};
	const result = {
		versions: { format: 2 },
		constants: schema.constants,
		error_feedback_channels: schema.error_feedback_channels.map((efc) => {
			return {
				kwl: efc.kwl,
				owner: efc.owner,
			};
		}),
		keywords: schema.keywords.map(convert_component),
		typedef_identifiers: schema.typedef_identifiers,
		typedefs: {},
		units: {},
	};
	for (const td in schema.typedefs) {
		result.typedefs[td] = convert_typedef(schema.typedefs[td]);
	}
	return result;
}
export function is_v2(schema) {
	return "versions" in schema;
}
export function ensure_v2(schema) {
	if (is_v2(schema)) {
		return schema;
	} else {
		return convert_schema(schema);
	}
}
//# sourceMappingURL=data:application/json;base64,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
