import {
	enforce,
	enforce_nonnull,
	path_index,
	path_strip_trailing_index,
	unreachable,
} from "./utilities.js";
export var ContainerType;
(function (ContainerType) {
	ContainerType[(ContainerType["None"] = 0)] = "None";
	ContainerType[(ContainerType["Table"] = 1)] = "Table";
	ContainerType[(ContainerType["Array"] = 2)] = "Array";
	ContainerType[(ContainerType["Vector"] = 3)] = "Vector";
	ContainerType[(ContainerType["Keyword"] = 4)] = "Keyword";
	ContainerType[(ContainerType["Component"] = 5)] = "Component";
})(ContainerType || (ContainerType = {}));
export function children_of_raw_subtree(schema, st) {
	if ("children" in st) {
		return enforce_nonnull(st.children);
	} else {
		const td = schema.typedefs[enforce_nonnull(st.type_identifier)];
		if (td.data_type === "subtree") {
			return td.children ?? [];
		}
		unreachable();
	}
}
export function children_of_raw_atomic_subtree(schema, st) {
	if ("children" in st) {
		return enforce_nonnull(st.children);
	} else {
		const td = schema.typedefs[enforce_nonnull(st.type_identifier)];
		if (td.data_type === "atomic subtree") {
			return td.children ?? [];
		}
		unreachable();
	}
}
function annotate_atomic_field(
	child,
	schema,
	units,
	constants_resolver,
	field_parent,
) {
	const clear_atomic_multiplicity = (c) => {
		return { ...c, atomic_multiplicity: [] };
	};
	const first_kernel = ((kernel_parent) => {
		switch (child.data_type) {
			case "atomic subtree": {
				const raw_children = enforce_nonnull(
					children_of_raw_atomic_subtree(schema, child),
				);
				const result = {
					...clear_atomic_multiplicity(child),
					optional: false,
					container_type: 0,
					children: [],
					parent: kernel_parent,
				};
				result.children = raw_children.map((child) =>
					annotate_atomic_field(
						child,
						schema,
						units,
						constants_resolver,
						result,
					),
				);
				return result;
			}
			case "variant": {
				enforce(
					child.variants !== undefined || child.type_identifier !== undefined,
				);
				const raw_variants =
					child.variants ??
					schema.typedefs[enforce_nonnull(child.type_identifier)].variants ??
					[];
				const result = {
					...clear_atomic_multiplicity(child),
					optional: false,
					container_type: 0,
					variants: [],
					parent: kernel_parent,
				};
				for (const [constructor_name, variant_desc] of raw_variants) {
					result.variants.push([
						constructor_name,
						annotate_atomic_field(
							variant_desc,
							schema,
							units,
							constants_resolver,
							result,
						),
					]);
				}
				return result;
			}
			case "int":
			case "float": {
				return {
					...clear_atomic_multiplicity(child),
					container_type: 0,
					optional: false,
					parent: kernel_parent,
					unit:
						child.other_properties?.unit === undefined
							? null
							: enforce_nonnull(units[child.other_properties.unit]),
				};
			}
			default:
				return {
					...clear_atomic_multiplicity(child),
					container_type: 0,
					optional: false,
					parent: kernel_parent,
				};
		}
	})(field_parent);
	let result = first_kernel;
	const add_container_layer = (container_type, capacity) => {
		const container = {
			container_type,
			capacity: enforce_nonnull(
				typeof capacity === "string" ? constants_resolver(capacity) : capacity,
			),
			contents: result,
			parent: field_parent,
			optional: false,
		};
		result.parent = container;
		result = container;
	};
	for (let i = child.atomic_multiplicity.length - 1; i >= 0; --i) {
		const level = child.atomic_multiplicity[i];
		switch (level.type) {
			case "optional":
				if (enforce_nonnull(result.optional) === false) {
					result.optional = true;
				} else {
					throw new Error(
						"Invalid atomic multiplicity: multiple consecutive levels of optionality are not supported",
					);
				}
				break;
			case "array":
				add_container_layer(2, enforce_nonnull(level.capacity));
				break;
			case "vector":
				add_container_layer(3, enforce_nonnull(level.capacity));
				break;
		}
	}
	return result;
}
function populate_raw_subtree(
	parent,
	st_raw,
	schema,
	units,
	numeric_constants_resolver,
) {
	for (const raw_child of enforce_nonnull(
		children_of_raw_subtree(schema, st_raw),
	)) {
		if (raw_child.data_type !== "subtree") {
			const temporary_bogus_parent = null;
			if (parent.container_type === 0) {
				if (raw_child.kw_type !== "virtual") {
					const c = {
						container_type: 4,
						kw_type: raw_child.kw_type,
						persistent: raw_child.persistent,
						contents: annotate_atomic_field(
							raw_child,
							schema,
							units,
							numeric_constants_resolver,
							temporary_bogus_parent,
						),
						parent,
						...(raw_child.kw_priority === undefined
							? {}
							: { kw_priority: raw_child.kw_priority }),
					};
					c.contents.parent = c;
					parent.children.push(c);
				}
			} else {
				unreachable();
			}
		} else {
			const kernel = {
				data_type: "subtree",
				other_properties: raw_child.other_properties,
				sys_name: raw_child.sys_name,
				brief: raw_child.brief,
				desc: raw_child.desc,
				type_identifier: raw_child.type_identifier,
				parent,
				container_type: 0,
				children: [],
			};
			const get_capacity = (maybe_constant) => {
				if (typeof maybe_constant === "string") {
					return numeric_constants_resolver(maybe_constant);
				} else {
					return maybe_constant;
				}
			};
			if (parent.container_type !== 0) {
				unreachable();
			} else {
				if ("array_size" in raw_child) {
					const array_container = {
						parent,
						capacity: get_capacity(raw_child.array_size),
						container_type: 2,
						contents: kernel,
					};
					kernel.parent = array_container;
					parent.children.push(array_container);
				} else if ("table_size" in raw_child) {
					const table_container = {
						parent,
						capacity: get_capacity(raw_child.table_size),
						container_type: 1,
						named_tables: raw_child.named_rows,
						contents: kernel,
					};
					kernel.parent = table_container;
					parent.children.push(table_container);
				} else {
					parent.children.push(kernel);
				}
			}
			populate_raw_subtree(
				kernel,
				raw_child,
				schema,
				units,
				numeric_constants_resolver,
			);
		}
	}
}
export function annotate(schema, disabledComponents, constants_resolver) {
	const keywords = [];
	const numeric_constants_resolver = (constant_name) => {
		const result = constants_resolver(constant_name);
		enforce(typeof result === "number");
		return result;
	};
	const dummy_parent = {
		container_type: 5,
		bound_to: [],
		enabled: false,
		owning_module: "",
		ua_name: "",
		contents: {},
	};
	const units = Object.fromEntries(
		Object.entries(schema.units ?? {}).map(([name, udef]) => {
			const exceptions_map = new Map();
			for (const [value, s] of udef.exceptions) {
				exceptions_map.set(value, s);
			}
			return [name, { ...udef, exceptions_map }];
		}),
	);
	for (const raw_component of schema.keywords) {
		if (disabledComponents.has(raw_component.sys_name)) {
			continue;
		}
		const component = {
			container_type: 5,
			bound_to: raw_component.bound_to ?? [],
			enabled:
				(raw_component.bound_to ?? []).findIndex(
					(constantName) => constants_resolver(constantName) === false,
				) === -1,
			owning_module: raw_component.owning_module,
			ua_name: raw_component.ua_name,
			contents: {
				parent: dummy_parent,
				children: [],
				container_type: 0,
				data_type: "subtree",
				other_properties: raw_component.other_properties,
				sys_name: raw_component.sys_name,
				brief: raw_component.brief,
				desc: raw_component.desc,
				type_identifier: raw_component.type_identifier,
			},
		};
		component.contents.parent = component;
		populate_raw_subtree(
			component.contents,
			raw_component,
			schema,
			units,
			numeric_constants_resolver,
		);
		keywords.push(component);
	}
	const typedefs = {};
	for (const typedefName of Object.keys(schema.typedefs)) {
		const td = schema.typedefs[typedefName];
		switch (td.data_type) {
			case "subtree":
				{
					const dummy_st = {
						container_type: 0,
						children: [],
						data_type: "subtree",
						sys_name: "",
						other_properties: {},
						parent: null,
					};
					populate_raw_subtree(
						dummy_st,
						{
							data_type: "subtree",
							children: [td],
							sys_name: "",
							other_properties: {},
						},
						schema,
						units,
						numeric_constants_resolver,
					);
					enforce(dummy_st.children.length === 1);
					const c = dummy_st.children[0];
					enforce(
						c.container_type === 0 ||
							c.container_type === 1 ||
							c.container_type === 2,
					);
					typedefs[typedefName] = {
						...c,
						common: td.common,
					};
				}
				break;
			default:
				enforce(Array.isArray(td.atomic_multiplicity));
				typedefs[typedefName] = {
					...annotate_atomic_field(
						td,
						schema,
						units,
						numeric_constants_resolver,
						undefined,
					),
					common: td.common,
				};
				break;
		}
	}
	return {
		...schema,
		typedefs,
		keywords,
		units,
	};
}
export function kernel_of_child(c) {
	let result = c;
	while ("contents" in result) {
		result = result.contents;
	}
	enforce(result);
	enforce(result.container_type === 0 && !("contents" in result));
	return result;
}
export function kernel_of_field(f) {
	let result = f;
	while ("contents" in result) {
		result = result.contents;
	}
	enforce(result);
	enforce(result.container_type === 0 && !("contents" in result));
	return result;
}
export function find_branch(annotated_schema, branch_name) {
	const segments = branch_name.split(".");
	enforce(segments.length > 0);
	for (const component of annotated_schema.keywords) {
		if (component.contents.sys_name === segments[0]) {
			let cur_st = component.contents;
			if (segments.length === 1) {
				return cur_st;
			}
			for (let i = 1; i < segments.length - 1; ++i) {
				for (const child of cur_st.children) {
					const kernel = kernel_of_child(child);
					if (kernel.sys_name === segments[i]) {
						if (kernel.data_type === "subtree") {
							cur_st = kernel;
							break;
						} else {
							throw new Error(`Unable to look up ${branch_name}`);
						}
					}
				}
			}
			const base_name = segments[segments.length - 1].replace(
				/_(command|status)$/,
				"",
			);
			for (const child of cur_st.children) {
				const kernel = kernel_of_child(child);
				enforce(kernel.data_type === "subtree" || child.container_type === 4);
				if (
					(kernel.sys_name === segments[segments.length - 1] &&
						kernel.data_type === "subtree") ||
					(child.container_type === 4 &&
						((child.kw_type === "duplex" && kernel.sys_name === base_name) ||
							(child.kw_type !== "duplex" &&
								kernel.sys_name === segments[segments.length - 1])))
				) {
					return child;
				}
			}
		}
	}
	throw new Error(`Unknown branch ${branch_name}`);
}
export function find_keyword(annotated_schema, branch_name, kw_name) {
	const child = find_branch(annotated_schema, `${branch_name}.${kw_name}`);
	if (child.container_type !== 4) {
		throw new Error(
			`Mistyped result (${branch_name} should have been a keyword, but is a subtree)`,
		);
	}
	return child;
}
export function find_subtree_within(desc, relative_kwl) {
	let cur_desc = desc;
	for (const top of relative_kwl.split(".")) {
		const sys_name = path_strip_trailing_index(top);
		const kernel = kernel_of_child(cur_desc);
		if (kernel.data_type !== "subtree") {
			return undefined;
		}
		const child_st = (() => {
			for (const child of kernel.children) {
				const child_kernel = kernel_of_child(child);
				if (child_kernel.sys_name === sys_name) {
					if (child_kernel.data_type === "subtree") {
						enforce(child.container_type !== 4);
						return child;
					} else {
						return undefined;
					}
				}
			}
			return undefined;
		})();
		if (child_st === undefined) {
			return undefined;
		}
		cur_desc = child_st;
		const maybe_index = path_index(top);
		switch (child_st.container_type) {
			case 2:
			case 1:
				if (maybe_index !== undefined) {
					if (maybe_index >= child_st.capacity) {
						return undefined;
					} else {
						enforce("contents" in cur_desc);
						cur_desc = cur_desc.contents;
					}
				}
				break;
			case 0:
				if (maybe_index !== undefined) {
					return undefined;
				}
		}
	}
	enforce(!Array.isArray(cur_desc));
	return cur_desc;
}
//# sourceMappingURL=data:application/json;base64,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
