import { enforce, enforce_nonnull } from "./utilities.js";
export class Duration {
	unit;
	value;
	static abs(x) {
		return new Duration(Math.abs(x.value), x.unit);
	}
	constructor(x, unit) {
		switch (unit) {
			case "h":
				this.value = 3600 * x;
				this.unit = "s";
				break;
			case "min":
				this.value = 60 * x;
				this.unit = "s";
				break;
			default:
				this.value = x;
				this.unit = unit === "us" ? "µs" : unit;
		}
	}
	static from_string(raw_input) {
		const m = raw_input.trim().match(/^(.*[0-9.]+)\s*(.*)$/);
		if (m === null) return undefined;
		const num_str = enforce_nonnull(m[1]);
		const num = parseFloat(num_str);
		if (isNaN(num)) return undefined;
		const unit_str = enforce_nonnull(m[2]);
		switch (unit_str) {
			case "ns":
			case "µs":
			case "ms":
			case "s":
				return new Duration(num, unit_str);
			case "us":
				return new Duration(num, "µs");
			case "min":
				return new Duration(num / 60, "s");
			case "h":
				return new Duration(num / 3600, "s");
			default:
				return undefined;
		}
	}
	plus(other) {
		if (other.unit === this.unit) {
			return new Duration(other.value + this.value, this.unit);
		} else {
			return new Duration(other.s() + this.s(), "s");
		}
	}
	minus(other) {
		if (other.unit === this.unit) {
			return new Duration(this.value - other.value, this.unit);
		} else {
			return new Duration(this.s() - other.s(), "s");
		}
	}
	divide_by(x) {
		if (typeof x === "number") {
			return new Duration(this.value / x, this.unit);
		} else {
			enforce(x instanceof Duration);
			return this.s() / x.s();
		}
	}
	times(x) {
		return new Duration(this.value * x, this.unit);
	}
	ns_rounded() {
		return Math.round(this.ns());
	}
	ns() {
		switch (this.unit) {
			case "ns":
				return this.value;
			case "µs":
				return this.value * 1e3;
			case "ms":
				return this.value * 1e6;
			case "s":
				return this.value * 1e9;
		}
	}
	us() {
		switch (this.unit) {
			case "ns":
				return this.value * 1e-3;
			case "µs":
				return this.value;
			case "ms":
				return this.value * 1e3;
			case "s":
				return this.value * 1e6;
		}
	}
	ms() {
		switch (this.unit) {
			case "ns":
				return this.value * 1e-6;
			case "µs":
				return this.value * 1e-3;
			case "ms":
				return this.value;
			case "s":
				return this.value * 1e3;
		}
	}
	s() {
		switch (this.unit) {
			case "ns":
				return this.value * 1e-9;
			case "µs":
				return this.value * 1e-6;
			case "ms":
				return this.value * 1e-3;
			case "s":
				return this.value;
		}
	}
	toString(mode) {
		if (mode === undefined || mode === "precise") {
			let v = this.value;
			let u = this.unit;
			while (u !== "s" && v % 1000 === 0) {
				v /= 1000;
				switch (u) {
					case "ns":
						u = "µs";
						break;
					case "µs":
						u = "ms";
						break;
					case "ms":
						u = "s";
						break;
				}
			}
			return `${v} ${u}`;
		} else {
			const ns = this.ns();
			const abs_ns = Math.abs(ns);
			const rough_time = (raw_seconds) => {
				console.assert(raw_seconds > 0);
				const seconds = Math.round(raw_seconds);
				const hours = Math.floor(seconds / 3600);
				const minutes = Math.floor((seconds - 3600 * hours) / 60);
				const rem = Math.round(seconds - 3600 * hours - 60 * minutes);
				const pad = (num) => ("0" + num).slice(-2);
				return `${pad(hours)}:${pad(minutes)}:${pad(rem)}`;
			};
			const to_fixed = (x, digits) => x.toFixed(digits).replace(/\.0+$/, "");
			if (abs_ns < 1e3) {
				return `${to_fixed(ns, 1)} ns`;
			} else if (abs_ns < 1e6) {
				return `${to_fixed(ns / 1000, 1)} µs`;
			} else if (abs_ns < 1e9) {
				return `${to_fixed(ns / 1000000, 1)} ms`;
			} else if (abs_ns < 60e9) {
				return `${to_fixed(ns / 1000000000, 3)} s`;
			} else if (abs_ns < 86400e9) {
				return `${ns < 0 ? "-" : ""}${rough_time(abs_ns * 1e-9)}`;
			} else {
				const days = Math.floor(abs_ns / 86400e9);
				const prefix = `${ns < 0 ? "-" : ""}${days}d`;
				if (abs_ns * 1e-9 - days * 86400 >= 0.5) {
					return `${prefix} ${rough_time(abs_ns * 1e-9 - days * 86400)}`;
				} else {
					return prefix;
				}
			}
		}
	}
	equal(other) {
		const abs_diff_s = Math.abs(this.s() - other.s());
		return (
			abs_diff_s < 1e-9 ||
			abs_diff_s < 1e-16 * Math.max(Math.abs(this.s()), Math.abs(other.s()))
		);
	}
	static max(x, ...ys) {
		let greatest = x;
		for (const y of ys) {
			if (y.s() > greatest.s()) greatest = y;
		}
		return greatest;
	}
	static min(x, ...ys) {
		let smallest = x;
		for (const y of ys) {
			if (y.s() < smallest.s()) smallest = y;
		}
		return smallest;
	}
}
export class Timestamp {
	ns_total;
	constructor(x) {
		switch (typeof x) {
			case "bigint":
				this.ns_total = x;
				break;
			case "number":
				this.ns_total = BigInt(Math.round(x * 1e9));
				break;
			case "string": {
				const parts = x.split(".");
				switch (parts.length) {
					case 1:
						this.ns_total = BigInt(parts[0]) * BigInt(1_000_000_000);
						break;
					case 2:
						this.ns_total =
							BigInt(parts[0]) * BigInt(1_000_000_000) + BigInt(parts[1]);
						if (this.ns_total < 0)
							throw new Error(`Unable to interpret '${x}' as a timestamp`);
						break;
					default:
						throw new Error(`Unable to interpret '${x}' as a timestamp`);
				}
			}
		}
	}
	nanoseconds() {
		return Number(this.ns_total % BigInt(1_000_000_000));
	}
	seconds() {
		return Number(this.ns_total / BigInt(1_000_000_000));
	}
	distance_to(other) {
		return new Duration(Number(other.ns_total - this.ns_total), "ns");
	}
	distance_from(other) {
		return other.distance_to(this);
	}
	plus(d) {
		return new Timestamp(this.ns_total + BigInt(d.ns_rounded()));
	}
	minus(d) {
		return this.plus(d.times(-1));
	}
	toString(mode) {
		const secs = this.ns_total / BigInt(1_000_000_000);
		const ns = this.ns_total % BigInt(1_000_000_000);
		if (mode === "convenient" && ns === BigInt(0)) {
			return secs.toString();
		} else {
			return secs.toString() + "." + ns.toString().padStart(9, "0");
		}
	}
	toNumber() {
		return Number(this.ns_total) * 1e-9;
	}
	toNSBigInt() {
		return this.ns_total;
	}
	toJSON() {
		return this.toString("backend-style");
	}
	equal(other) {
		return other.ns_total === this.ns_total;
	}
}
//# sourceMappingURL=data:application/json;base64,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
