import { Duration } from "./time.js";
import { enforce_nonnull, unreachable, enforce } from "./asserts.js";
export * from "./asserts.js";
export function pause(t) {
	return new Promise((resolve) => {
		setTimeout(() => {
			resolve();
		}, t.ms());
	});
}
export function path_hd(path) {
	const i = path.lastIndexOf(".");
	if (i !== -1) {
		return path.substring(0, i);
	}
	return "";
}
export function path_tl(path) {
	const i = path.lastIndexOf(".");
	if (i !== -1) {
		return path.substring(i + 1);
	}
	return path;
}
export function path_to_branch(path) {
	return path.replace(/\[[0-9]+\]/g, "");
}
export function path_strip_trailing_index(path) {
	return path.replace(/\[[0-9]+\]$/, "");
}
export function path_index(path) {
	if (path.charAt(path.length - 1) !== "]") return undefined;
	const i = path.lastIndexOf("[");
	if (i === -1) {
		return undefined;
	}
	return parseInt(path.substr(i + 1, path.length - i - 2));
}
export function path_compare(a, b) {
	const by_branch = path_to_branch(a).localeCompare(path_to_branch(b));
	if (by_branch !== 0) return by_branch;
	const num_re = /(\d+)/;
	let m_a = num_re.exec(a);
	let m_b = num_re.exec(b);
	while (!!m_a) {
		enforce(!!m_b);
		const i_a = parseInt(enforce_nonnull(m_a[1]));
		const i_b = parseInt(enforce_nonnull(m_b[1]));
		enforce(!isNaN(i_a) && !isNaN(i_b));
		const diff = i_a - i_b;
		if (diff !== 0) return diff;
		m_a = num_re.exec(a.substring(m_a.index + enforce_nonnull(m_a[1]).length));
		m_b = num_re.exec(b.substring(m_b.index + enforce_nonnull(m_b[1]).length));
	}
	return 0;
}
export function snake_case(s) {
	let result = "";
	for (const C of s) {
		const c = C.toLowerCase();
		if (C !== c && result.length !== 0) {
			result += "_";
		}
		result += c;
	}
	return result;
}
export class WorkQueue {
	num_workers;
	num_running = 0;
	todo = [];
	finalizers = [];
	thrown_exceptions = [];
	m_waiting = true;
	constructor(pars) {
		this.num_workers = pars.num_workers;
		if (pars.start_immediately ?? true) {
			this.start();
		}
	}
	get size() {
		return this.todo.length + this.num_running;
	}
	get stats() {
		return { waiting: this.todo.length, running: this.num_running };
	}
	tick() {
		const n = Math.min(this.num_workers - this.num_running, this.todo.length);
		const pop = this.todo.splice(0, n);
		this.num_running += n;
		for (const f of pop) {
			f()
				.catch((e) => this.thrown_exceptions.push(e))
				.finally(() => {
					this.num_running -= 1;
					if (!this.m_waiting) this.tick();
				});
		}
		if (this.idle()) {
			for (const [resolve, reject] of this.finalizers) {
				if (this.thrown_exceptions.length === 0) {
					resolve();
				} else {
					reject(this.thrown_exceptions);
				}
			}
			this.finalizers = [];
		}
	}
	start() {
		this.m_waiting = false;
		this.tick();
	}
	idle() {
		return this.num_running === 0 && this.todo.length === 0;
	}
	push(f) {
		this.todo.push(f);
		if (!this.m_waiting) this.tick();
	}
	clear() {
		this.todo = [];
		if (!this.m_waiting) this.tick();
	}
	drain() {
		this.m_waiting = false;
		return new Promise((resolve, reject) => {
			this.finalizers.push([resolve, reject]);
			this.tick();
		});
	}
}
export async function poll(criterion, pars) {
	const ms0 = new Date().valueOf();
	while (true) {
		if (await criterion()) {
			return;
		}
		const timeout = pars?.timeout ?? new Duration(5, "s");
		if (new Date().valueOf() > ms0 + timeout.ms()) {
			throw new Error(`Failed to satisfy ${criterion} within ${timeout} s`);
		}
		await pause(pars?.interval ?? new Duration(200, "ms"));
	}
}
export async function asyncFilter(xs, f, pars) {
	const wq = new WorkQueue({ num_workers: pars?.num_workers ?? 32 });
	const result = [];
	xs.forEach((x, i) => {
		wq.push(async () => {
			if (await f(x)) {
				result.push([x, i]);
			}
		});
	});
	await wq.drain();
	result.sort((a, b) => a[1] - b[1]);
	return result.map((a) => a[0]);
}
export async function asyncFilterMap(xs, f, pars) {
	const wq = new WorkQueue({ num_workers: pars?.num_workers ?? 32 });
	const result = [];
	xs.forEach((x, i) => {
		wq.push(async () => {
			result.push([await f(x), i]);
		});
	});
	await wq.drain();
	result.sort((a, b) => a[1] - b[1]);
	return result.filter((x) => x[0] !== undefined).map((a) => a[0]);
}
export async function asyncMap(xs, f, pars) {
	const result = [];
	const wq = new WorkQueue({ num_workers: pars?.num_workers ?? 32 });
	xs.forEach((x, i) => {
		wq.push(async () => {
			result.push([await f(x, i), i]);
		});
	});
	await wq.drain();
	result.sort((a, b) => a[1] - b[1]);
	return result.map((a) => a[0]);
}
export async function poll_until(f, pars) {
	const ms0 = new Date().valueOf();
	while (true) {
		const x = await f();
		if (x.satisfied === true) {
			return x.result;
		}
		await pause(pars.pollInterval ?? new Duration(1, "s"));
		if (new Date().valueOf() > ms0 + pars.timeout.ms()) {
			throw new Error(
				`Failed to satisfy criterion ${f} within ${pars.timeout} s`,
			);
		}
	}
}
export async function asyncFind(xs, criterion) {
	for (const x of xs) {
		if (await criterion(x)) {
			return x;
		}
	}
	return undefined;
}
export async function asyncGet(xs, criterion) {
	const maybeResult = await asyncFind(xs, criterion);
	if (maybeResult === undefined) {
		throw new Error(`Found no list element satisfying ${criterion}`);
	}
	return maybeResult;
}
export async function asyncFindMap(xs, f) {
	for (const x of xs) {
		const y = await f(x);
		if (y !== undefined) {
			return y;
		}
	}
	return undefined;
}
export async function asyncGetMap(xs, f) {
	const maybeResult = await asyncFindMap(xs, f);
	if (maybeResult === undefined) {
		throw new Error(`Found no list element satisfying ${f}`);
	}
	return maybeResult;
}
export async function asyncIter(things, f, opts) {
	const wq = new WorkQueue(opts ?? { num_workers: 32 });
	for (let i = 0; i < things.length; ++i) {
		wq.push(() => f(things[i], i));
	}
	await wq.drain();
}
export async function asyncZip(left_things, right_things, f, opts) {
	await asyncIter(
		Array.from(
			{ length: Math.min(left_things.length, right_things.length) },
			(_, i) => i,
		),
		(i) => f(left_things[i], right_things[i], i),
		opts,
	);
}
export function parse_machine_address(str) {
	const m = str.match(/(.*):\/\/(.*)/);
	if (m) {
		switch (enforce_nonnull(m[1]).toLowerCase()) {
			case "https":
			case "wss":
				return { protocol: "wss", ip: enforce_nonnull(m[2]) };
			case "http":
			case "ws":
				return { protocol: "ws", ip: enforce_nonnull(m[2]) };
			default:
				unreachable();
		}
	} else {
		return { protocol: "ws", ip: str };
	}
}
export function numstring_with_precision(payload, precision, force_precise) {
	return force_precise || Number.isInteger(payload)
		? payload.toString()
		: payload.toPrecision(precision).replace(/0+e/, "e");
}
export function scale_format(payload, scale, precision) {
	let i = 0;
	while (i < scale.length - 1 && scale[i + 1][0] < Math.abs(payload)) ++i;
	enforce(scale[i][0] > 0);
	return `${numstring_with_precision(payload / scale[i][0], precision, false)} ${scale[i][1]}`;
}
export function format_with_unit(payload, unit, precision) {
	return (
		unit.exceptions_map.get(payload) ??
		scale_format(payload, unit.scale, precision)
	);
}
export function addrs_equal_up_to_port(pars) {
	const command = pars.command ?? "";
	const status = pars.status ?? "";
	if (command.match(/:[0-9]+$/)) {
		return command.trim() === status.trim();
	} else {
		return command.trim() === status.trim().replace(/:[0-9]+$/, "");
	}
}
//# sourceMappingURL=data:application/json;base64,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
