import {
	Subtree,
	SubtreeArray,
	SubtreeNamedTable,
	SubtreeTable,
} from "./data_views.js";
import { Duration, Timestamp } from "./time.js";
import { kernel_of_child } from "./schema_v2.js";
import {
	enforce,
	enforce_nonnull,
	path_hd,
	path_index,
	path_strip_trailing_index,
	path_tl,
	pause,
} from "./utilities.js";
import {
	RefinedIndexedMutatingReference,
	RefinedIndexedNonmutatingReference,
	RefinedMutatingReference,
	RefinedNonmutatingReference,
} from "./voverlays.js";
export function get_subtree(socket, kwl) {
	const desc = socket.root.find(kwl);
	enforce(desc instanceof Subtree);
	return desc;
}
export function component_checked(socket, expected_sys_name) {
	const child_desc = socket.schema.keywords.find(
		(comp) => comp.contents.sys_name === expected_sys_name && comp.enabled,
	);
	if (child_desc === undefined) return null;
	return new Subtree(socket, expected_sys_name, child_desc.contents);
}
function child_not_found_errmsg(expected_sys_name, subtree) {
	const children_suffix =
		subtree.description.container_type === 0
			? ` (children available at the current level are: ${subtree.description.children
					.map((c) => kernel_of_child(c).sys_name)
					.join(", ")})`
			: "";
	const vsocket = subtree.backing_store;
	return `Unable to find child '${expected_sys_name}' ${children_suffix}. This might be caused by a version mismatch between the vapi module you're currently using and the software version ${vsocket.build_info.version}[model: ${vsocket.build_info.hardware_model}] running on the target machine ${vsocket.ip}`;
}
export function subtree_child_checked(subtree, child_index, expected_sys_name) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			child_desc.container_type === 0 &&
			child_desc.sys_name === expected_sys_name
		) {
			return new Subtree(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
		}
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function subtree_table_child_checked(
	subtree,
	child_index,
	expected_sys_name,
) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			child_desc.container_type === 1 &&
			child_desc.contents.sys_name === expected_sys_name
		) {
			return new SubtreeTable(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
		}
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function subtree_named_table_child_checked(
	subtree,
	child_index,
	expected_sys_name,
) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			child_desc.container_type === 1 &&
			child_desc.named_tables === true &&
			child_desc.contents.sys_name === expected_sys_name
		) {
			return new SubtreeNamedTable(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
		}
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function subtree_array_child_checked(
	subtree,
	child_index,
	expected_sys_name,
) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			child_desc.container_type === 2 &&
			child_desc.contents.sys_name === expected_sys_name
		) {
			return new SubtreeArray(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
		}
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function liftOrLowerNullable(f) {
	return (maybe_payload, socket) =>
		maybe_payload === null ? null : f(maybe_payload, socket);
}
export function liftNanoseconds(payload, _) {
	return new Duration(payload, "ns");
}
export function liftSeconds(payload, _) {
	return new Duration(payload, "s");
}
export function liftTimestamp(payload, _) {
	return new Timestamp(payload);
}
export function lowerTimestamp(ts, _) {
	return ts.toString("backend-style");
}
export function lowerNanoseconds(payload, _) {
	return payload.ns_rounded();
}
export function lowerSeconds(payload, _) {
	return payload.s();
}
export function liftArray(f) {
	return (raw_elems, socket) => raw_elems.map((x) => f(x, socket));
}
export function mapMaybeSparseArray(raw_elems, f) {
	if (Array.isArray(raw_elems)) return raw_elems.map((x) => f(x));
	const result = {};
	for (const k in raw_elems) {
		result[k] = f(raw_elems[k]);
	}
	return result;
}
export function liftMaybeSparseArray(f) {
	return (raw_elems, socket) =>
		mapMaybeSparseArray(raw_elems, (x) => f(x, socket));
}
export function liftNullable(f) {
	return (maybe_x, socket) => (maybe_x === null ? null : f(maybe_x, socket));
}
export function lowerArray(f) {
	return (elems, socket) => elems.map((x) => f(x, socket));
}
export function lowerNullableArray(f) {
	return (raw_elems, socket) =>
		raw_elems === null ? null : raw_elems.map((x) => f(x, socket));
}
export function liftRefinedMutatingReference(f, member_name, type_converters) {
	return (path, backing_store) => {
		if (path === null) return null;
		const parent = f(path_hd(path), backing_store);
		enforce(path_tl(path) === member_name);
		return new RefinedMutatingReference(parent, member_name, type_converters);
	};
}
export function liftRefinedIndexedMutatingReference(
	f,
	member_name,
	type_converters,
) {
	return (path, backing_store) => {
		if (path === null) return null;
		const parent = f(path_hd(path), backing_store);
		const index = enforce_nonnull(path_index(path));
		enforce(index !== undefined && !isNaN(index));
		enforce(path_strip_trailing_index(path_tl(path)) === member_name);
		return new RefinedIndexedMutatingReference(
			parent,
			member_name,
			index,
			type_converters,
		);
	};
}
export function liftRefinedNonmutatingReference(
	f,
	member_name,
	type_converters,
) {
	return (path, backing_store) => {
		if (path === null) return null;
		const parent = f(path_hd(path), backing_store);
		enforce(path_tl(path) === member_name);
		return new RefinedNonmutatingReference(
			parent,
			member_name,
			type_converters,
		);
	};
}
export function liftRefinedIndexedNonmutatingReference(
	f,
	member_name,
	type_converters,
) {
	return (path, backing_store) => {
		if (path === null) return null;
		const parent = f(path_hd(path), backing_store);
		const index = enforce_nonnull(path_index(path));
		enforce(!isNaN(index));
		enforce(path_strip_trailing_index(path_tl(path)) === member_name);
		return new RefinedIndexedNonmutatingReference(
			parent,
			member_name,
			index,
			type_converters,
		);
	};
}
export function lowerRefinedReference() {
	return (maybe_ref, _) => {
		if (maybe_ref === null) return null;
		return `${maybe_ref.enclosing_subtree.raw.kwl}.${maybe_ref.kw()}`;
	};
}
export function lowerRefinedIndexedReference() {
	return (maybe_ref, _) => {
		if (maybe_ref === null) return null;
		return `${maybe_ref.enclosing_subtree.raw.kwl}.${maybe_ref.kw()}[${maybe_ref.index}]`;
	};
}
export async function legacy_table_resize(pars) {
	const retry_interval = pars.retry_interval ?? new Duration(200, "ms");
	let indices = await pars.table.allocated_indices();
	while (indices.length < pars.n) {
		for (let i = 0; i < pars.n - indices.length; ++i) {
			await pars.add_row.write("Click");
		}
		await pause(retry_interval);
		indices = await pars.table.allocated_indices();
	}
	while (indices.length > pars.n) {
		for (let i = pars.n; i < indices.length; ++i) {
			await pars.delete_row(i).write("Click");
		}
		await pause(retry_interval);
		indices = await pars.table.allocated_indices();
	}
}
//# sourceMappingURL=data:application/json;base64,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
