import { VSettings, VSocket } from "./data_views.js";
import { reset_subtree_to_defaults } from "./schema_reflection.js";
import { Timestamp } from "./time.js";
import { asyncIter, enforce, enforce_nonnull } from "./utilities.js";
export async function reset_to_defaults(x) {
	enforce(!Array.isArray(x.raw.description));
	await reset_subtree_to_defaults(
		x.raw.backing_store,
		enforce_nonnull(x.raw.kwl),
		x.raw.description,
	);
}
export function same(a, b) {
	if (a !== null && b !== null) {
		if (a.raw.kwl !== b.raw.kwl) return false;
		if (
			a.raw.backing_store instanceof VSocket &&
			b.raw.backing_store instanceof VSocket
		) {
			return a.raw.backing_store.ip === b.raw.backing_store.ip;
		} else if (
			a.raw.backing_store instanceof VSettings &&
			b.raw.backing_store instanceof VSettings
		) {
			return a.raw.backing_store === b.raw.backing_store;
		}
		return false;
	} else {
		return (a === null) === (b === null);
	}
}
export function nullableLiftOrLower(liftOrLower) {
	return (payload, backing_store) =>
		payload === null ? null : liftOrLower(payload, backing_store);
}
export class rKeyword {
	parent;
	m_kw;
	m_read_converters;
	constructor(parent, m_kw, m_read_converters) {
		this.parent = parent;
		this.m_kw = m_kw;
		this.m_read_converters = m_read_converters;
	}
	async read(opts) {
		return this.m_read_converters.lift(
			await this.parent.raw.read({ kw: this.m_kw }, opts),
			this.parent.raw.backing_store,
		);
	}
	async wait_until(criterion, opts) {
		const f = this.m_read_converters
			? (payload) =>
					criterion(
						this.m_read_converters.lift(payload, this.parent.raw.backing_store),
					)
			: criterion;
		const result = await this.parent.raw.wait_until({ kw: this.m_kw }, f, opts);
		return this.m_read_converters.lift(result, this.parent.raw.backing_store);
	}
	async watch(handler, opts) {
		const f = this.m_read_converters
			? (payload) =>
					handler(
						this.m_read_converters.lift(payload, this.parent.raw.backing_store),
					)
			: handler;
		return await this.parent.raw.watch({ kw: this.m_kw }, f, opts);
	}
}
export class rKeywordReferenceable extends rKeyword {
	m_scalar_read_converters;
	constructor(parent, m_kw, m_scalar_read_converters, m_read_converters) {
		super(parent, m_kw, m_read_converters);
		this.m_scalar_read_converters = m_scalar_read_converters;
	}
	reference_to_index(index) {
		return new RefinedIndexedNonmutatingReference(
			this.parent,
			this.m_kw,
			index,
			this.m_scalar_read_converters,
		);
	}
}
async function do_write(kw, payload, parent, type_converters, opts) {
	let lifted_validator;
	if (type_converters && opts?.retry_until?.criterion === "status") {
		const v = opts.retry_until.validator;
		lifted_validator = (status) =>
			v(type_converters.lift(status, parent.raw.backing_store));
	}
	if (lifted_validator === undefined && payload instanceof Timestamp) {
		lifted_validator = (status) => {
			return (
				(status === null && payload === null) ||
				payload.equal(type_converters.lift(status, parent.raw.backing_store))
			);
		};
	}
	await parent.raw.write(
		{ kw },
		type_converters
			? type_converters.lower(payload, parent.raw.backing_store)
			: payload,
		{
			...opts,
			...(lifted_validator
				? {
						retry_until: { criterion: "status", validator: lifted_validator },
					}
				: {}),
		},
	);
}
async function do_write_sparse(
	kw,
	payload,
	parent,
	non_nullable_type_converters,
	opts,
) {
	const indices = Object.keys(payload).map((s) => parseInt(s));
	enforce(indices.findIndex((i) => isNaN(i)) === -1);
	let lifted_validator;
	if (opts?.retry_until?.criterion === "status") {
		const v = opts.retry_until.validator;
		lifted_validator = (status) => {
			if (status === null) return false;
			for (const i of indices) {
				if (status.length <= i) return false;
				if (
					v(
						non_nullable_type_converters.lift(
							status[i],
							parent.raw.backing_store,
						),
					)
				)
					return false;
			}
			return true;
		};
	}
	const converted_payload = {};
	for (const i of indices) {
		converted_payload[i] = non_nullable_type_converters.lower(
			payload[i],
			parent.raw.backing_store,
		);
	}
	await parent.raw.write({ kw }, converted_payload, {
		...opts,
		...(lifted_validator
			? {
					retry_until: { criterion: "status", validator: lifted_validator },
				}
			: {}),
	});
}
export class riKeyword {
	parent;
	m_kw;
	index;
	m_lift;
	extract(payload, backing_store) {
		if (
			payload === null ||
			payload.length <= this.index ||
			payload[this.index] === null
		) {
			return null;
		}
		return this.m_lift(payload[this.index], backing_store);
	}
	constructor(parent, m_kw, index, read_converters) {
		this.parent = parent;
		this.m_kw = m_kw;
		this.index = index;
		this.m_lift = (payload, backing_store) => {
			if (payload === null) return null;
			return read_converters.lift(payload, backing_store);
		};
	}
	async read(opts) {
		const raw = await this.parent.raw.read({ kw: this.m_kw }, opts);
		if (raw === null || raw.length <= this.index) return null;
		return this.m_lift(raw[this.index], this.parent.raw.backing_store);
	}
	async wait_until(criterion, opts) {
		const raw_result = await this.parent.raw.wait_until(
			{ kw: this.m_kw },
			(payload) =>
				criterion(this.extract(payload, this.parent.raw.backing_store)),
			opts,
		);
		return this.extract(raw_result, this.parent.raw.backing_store);
	}
	async watch(handler, opts) {
		return await this.parent.raw.watch(
			{ kw: this.m_kw },
			(payload) => {
				handler(this.extract(payload, this.parent.raw.backing_store));
			},
			opts,
		);
	}
}
export class wKeyword {
	parent;
	m_kw;
	m_write_converters;
	m_anchored_write_validator;
	constructor(parent, m_kw, m_write_converters, badly_typed_validator) {
		this.parent = parent;
		this.m_kw = m_kw;
		this.m_write_converters = m_write_converters;
		this.m_anchored_write_validator = badly_typed_validator;
	}
	async write(payload, opts) {
		return do_write(this.m_kw, payload, this.parent, this.m_write_converters, {
			retry_until: { criterion: "custom", validator: async () => true },
			...(this.m_anchored_write_validator === null
				? {}
				: {
						retry_until: this.m_anchored_write_validator(this.parent, payload),
					}),
			...opts,
		});
	}
}
export class wKeywordReferenceable extends wKeyword {
	m_scalar_converters;
	constructor(
		parent,
		m_kw,
		m_scalar_converters,
		m_write_converters,
		badly_typed_write_validator,
	) {
		super(parent, m_kw, m_write_converters, badly_typed_write_validator);
		this.m_scalar_converters = m_scalar_converters;
	}
	reference_to_index(index) {
		return new RefinedIndexedNonmutatingReference(
			this.parent,
			this.m_kw,
			index,
			this.m_scalar_converters,
		);
	}
}
export class rwiKeyword extends riKeyword {
	m_lower;
	constructor(parent, kw, index, type_converters) {
		super(parent, kw, index, type_converters);
		this.m_lower = (payload, backing_store) => {
			if (payload === null) return null;
			return type_converters.lower(payload, backing_store);
		};
	}
	async write(payload, opts) {
		await do_write_sparse(
			this.m_kw,
			{ [this.index]: payload },
			this.parent,
			{ lift: this.m_lift, lower: this.m_lower },
			opts,
		);
	}
}
export class rwKeyword extends rKeyword {
	m_write_converters;
	m_anchored_write_validator;
	constructor(
		parent,
		m_kw,
		read_converters,
		m_write_converters,
		badly_typed_write_validator,
	) {
		super(parent, m_kw, read_converters);
		this.m_write_converters = m_write_converters;
		this.m_anchored_write_validator = badly_typed_write_validator;
	}
	async write(payload, opts) {
		return do_write(this.m_kw, payload, this.parent, this.m_write_converters, {
			...(this.m_anchored_write_validator
				? { retry_until: this.m_anchored_write_validator(this.parent, payload) }
				: {}),
			...(opts ?? {}),
		});
	}
}
export class rwKeywordReferenceable extends rwKeyword {
	m_scalar_read_converters;
	m_scalar_write_converters;
	constructor(
		parent,
		m_kw,
		m_scalar_read_converters,
		m_scalar_write_converters,
		read_converters,
		write_converters,
		badly_typed_write_validator,
	) {
		super(
			parent,
			m_kw,
			read_converters,
			write_converters,
			badly_typed_write_validator,
		);
		this.m_scalar_read_converters = m_scalar_read_converters;
		this.m_scalar_write_converters = m_scalar_write_converters;
		void this.m_scalar_write_converters;
	}
	reference_to_index(index) {
		return new RefinedIndexedNonmutatingReference(
			this.parent,
			this.m_kw,
			index,
			this.m_scalar_read_converters,
		);
	}
}
export class dKeyword {
	status;
	command;
	constructor(
		parent,
		kw,
		read_converters,
		write_converters,
		badly_typed_write_validator,
	) {
		this.status = new rKeyword(parent, `${kw}_status`, read_converters);
		this.command = new rwKeyword(
			parent,
			`${kw}_command`,
			read_converters,
			write_converters,
			badly_typed_write_validator,
		);
	}
	get parent() {
		return this.status.parent;
	}
}
export class dKeywordReferenceable {
	m_scalar_read_converters;
	m_scalar_write_converters;
	status;
	command;
	constructor(
		parent,
		kw,
		m_scalar_read_converters,
		m_scalar_write_converters,
		read_converters,
		write_converters,
		badly_typed_write_validator,
	) {
		this.m_scalar_read_converters = m_scalar_read_converters;
		this.m_scalar_write_converters = m_scalar_write_converters;
		void this.m_scalar_write_converters;
		this.status = new rKeywordReferenceable(
			parent,
			`${kw}_status`,
			this.m_scalar_read_converters,
			read_converters,
		);
		this.command = new rwKeyword(
			parent,
			`${kw}_command`,
			read_converters,
			write_converters,
			badly_typed_write_validator,
		);
	}
	get parent() {
		return this.status.parent;
	}
}
export class StronglyTypedNamedTable {
	raw;
	m_lift;
	constructor(raw, m_lift) {
		this.raw = raw;
		this.m_lift = m_lift;
	}
	async allocated_indices(opts) {
		return this.raw.allocated_indices(opts);
	}
	capacity() {
		enforce(this.raw.description.container_type === 1);
		return this.raw.description.capacity;
	}
	async is_allocated(index, opts) {
		return this.raw.is_allocated(index, opts);
	}
	async create_row(opts) {
		return this.m_lift(await this.raw.create_row(opts), this.raw.backing_store);
	}
	async rows(opts) {
		return (await this.raw.rows(opts)).map((st) =>
			this.m_lift(st, this.raw.backing_store),
		);
	}
	async delete_all() {
		await this.raw.delete_all();
	}
	async ensure_allocated(n, mode) {
		let rows = await this.rows();
		if (rows.length < n) {
			await asyncIter(
				Array.from({ length: n - rows.length }, () => null),
				async () => {
					await this.create_row();
				},
			);
			rows = await this.rows();
		} else if (rows.length > n && mode === "exactly") {
			await asyncIter(rows.slice(n), async (row) => {
				await row.delete();
			});
			rows = rows.slice(0, n);
		}
		return rows;
	}
	row(index, opts) {
		return this.m_lift(this.raw.row(index, opts), this.raw.backing_store);
	}
	row_unchecked(index) {
		return this.m_lift(this.raw.row_unchecked(index), this.raw.backing_store);
	}
}
export class StronglyTypedTable {
	raw;
	m_lift;
	constructor(raw, m_lift) {
		this.raw = raw;
		this.m_lift = m_lift;
	}
	async rows(opts) {
		return (await this.raw.rows(opts)).map((st) =>
			this.m_lift(st, this.raw.backing_store),
		);
	}
	async allocated_indices(opts) {
		return this.raw.allocated_indices(opts);
	}
	async is_allocated(index, opts) {
		return this.raw.is_allocated(index, opts);
	}
	row(index) {
		return this.m_lift(this.raw.row(index), this.raw.backing_store);
	}
	async *[Symbol.asyncIterator]() {
		const indices = await this.raw.allocated_indices();
		const self = this;
		return (function* () {
			for (const i of indices) {
				yield self.m_lift(self.raw.row_unchecked(i), self.raw.backing_store);
			}
		})();
	}
}
export class StronglyTypedArray {
	raw;
	m_lift;
	constructor(raw, m_lift) {
		this.raw = raw;
		this.m_lift = m_lift;
	}
	row(index) {
		return this.m_lift(this.raw.row(index), this.raw.backing_store);
	}
	get size() {
		return this.raw.description.capacity;
	}
	[Symbol.iterator]() {
		const N = this.size;
		const self = this;
		return (function* () {
			for (let i = 0; i < N; ++i) {
				yield self.row(i);
			}
		})();
	}
}
export class RefinedMutatingReference extends rwKeyword {
	enclosing_subtree;
	constructor(enclosing_subtree, kw, type_converters) {
		super(enclosing_subtree, kw, type_converters, type_converters, null);
		this.enclosing_subtree = enclosing_subtree;
	}
	kw() {
		return this.m_kw;
	}
}
export class RefinedNonmutatingReference extends rKeyword {
	enclosing_subtree;
	constructor(enclosing_subtree, kw, type_converters) {
		super(enclosing_subtree, kw, type_converters);
		this.enclosing_subtree = enclosing_subtree;
	}
	kw() {
		return this.m_kw;
	}
}
export class RefinedIndexedMutatingReference extends rwiKeyword {
	enclosing_subtree;
	constructor(enclosing_subtree, kw, index, type_converters) {
		super(enclosing_subtree, kw, index, type_converters);
		this.enclosing_subtree = enclosing_subtree;
	}
	kw() {
		return this.m_kw;
	}
}
export class RefinedIndexedNonmutatingReference extends riKeyword {
	enclosing_subtree;
	constructor(enclosing_subtree, kw, index, type_converters) {
		super(enclosing_subtree, kw, index, type_converters);
		this.enclosing_subtree = enclosing_subtree;
	}
	kw() {
		return this.m_kw;
	}
}
//# sourceMappingURL=data:application/json;base64,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
