import { deepEqual } from "./polyfills/polyfill_util.js";
import { enforce, unreachable } from "./utilities.js";
import { Timestamp } from "./time.js";
export function atomic_equal(x, y, desc) {
	if (x === null && y === null) {
		enforce(desc.optional);
		return true;
	}
	if (x !== null && y !== null) {
		switch (desc.container_type) {
			case 0:
				switch (desc.data_type) {
					case "alerts":
					case "timecode":
						unreachable();
					case "atomic subtree":
						enforce(Array.isArray(x) && Array.isArray(y));
						enforce(x.length === y.length && x.length === desc.children.length);
						for (let i = 0; i < desc.children.length; ++i) {
							if (!atomic_equal(x[i], y[i], desc.children[i])) return false;
						}
						return true;
					case "variant":
						enforce(Array.isArray(x) && Array.isArray(y));
						enforce(x.length === 2 && y.length === 2);
						enforce(typeof x[0] === "number" && typeof y[0] === "number");
						if (x[0] !== y[0]) return false;
						return atomic_equal(x[1], y[1], desc.variants[x[0]][1]);
					case "bool":
						enforce(typeof x === "boolean" && typeof y === "boolean");
						return x === y;
					case "enum":
					case "ipaddress":
					case "ref":
					case "string":
						enforce(typeof x === "string" && typeof y === "string");
						return x === y;
					case "device_tree_node":
						return deepEqual(x, y);
					case "float duration":
					case "float":
						enforce(typeof x === "number" && typeof y === "number");
						return Math.abs(x - y) <= 1e-9;
					case "int":
					case "int duration":
						enforce(typeof x === "number" && typeof y === "number");
						return x === y;
					case "timestamp":
						enforce(typeof x === "number" || typeof x === "string");
						enforce(typeof y === "number" || typeof y === "string");
						return new Timestamp(x).equal(new Timestamp(y));
				}
			case 2:
			case 3:
				enforce(Array.isArray(x) && Array.isArray(y));
				if (x.length !== y.length) {
					enforce(desc.container_type === 3);
					return false;
				}
				for (let i = 0; i < x.length; ++i) {
					if (!atomic_equal(x[i], y[i], desc.contents)) return false;
				}
				return true;
		}
	}
	enforce((x === null) !== (y === null));
	enforce(desc.optional);
	return false;
}
function default_validator(x, maybe_desc) {
	if (!!x && typeof x === "object" && !Array.isArray(x)) {
		return (y) => {
			if (y === null) return false;
			enforce(Array.isArray(y));
			for (const i in x) {
				const y_el = y[i];
				const x_el = x[i];
				if (maybe_desc !== null) {
					enforce(maybe_desc.contents.container_type === 2, `${x}`);
					if (!atomic_equal(y_el, x_el, maybe_desc.contents.contents))
						return false;
				} else {
					if (!deepEqual(y_el, x_el)) return false;
				}
			}
			return true;
		};
	}
	if (maybe_desc) {
		return (y) => atomic_equal(x, y, maybe_desc.contents);
	}
	return (y) => deepEqual(x, y);
}
export function status_is(x, maybe_desc) {
	return { criterion: "status", validator: default_validator(x, maybe_desc) };
}
export function status_sparsely_matches(x) {
	return {
		criterion: "status",
		validator: (y) => {
			if (y !== null && x !== null) {
				enforce(Array.isArray(y));
				for (const i in x) {
					if (parseInt(i, 10) >= y.length || !deepEqual(y[i], x[i]))
						return false;
				}
				return true;
			} else {
				return (x === null) === (y === null);
			}
		},
	};
}
export function max_abs_deviation(epsilon) {
	return (x) => {
		return {
			criterion: "status",
			validator: (y) =>
				(y === null && x === null) ||
				(y !== null && x !== null && Math.abs(x - y) <= epsilon),
		};
	};
}
export function max_rel_deviation(epsilon) {
	return (x) => {
		return {
			criterion: "status",
			validator: (y) =>
				(y === null && x === null) ||
				(y !== null &&
					x !== null &&
					Math.abs(x - y) <= epsilon * Math.max(Math.abs(x), Math.abs(y))),
		};
	};
}
//# sourceMappingURL=data:application/json;base64,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
