import { read_error_message } from "./error_channels.js";
import { NumericIDPool, StringIDPool } from "./id_pool.js";
import { KWLCache } from "./kwl_cache.js";
import { ListenerRegistry } from "./listener_registry.js";
import { download_json } from "./polyfills/polyfill_http.js";
import * as WebSocket from "./polyfills/polyfill_ws.js";
import { binary_concat } from "./polyfills/polyfill_ws.js";
import { get_env } from "./polyfills/polyfill_globals.js";
import {
	component_of_kwl,
	EventHandler,
	MissingData,
	Watcher,
	with_timeout,
} from "./pervasives.js";
import { ResourceCache } from "./resource_cache.js";
import { indices as indices_of_rowmask } from "./rowmask.js";
import { ensure_v2 } from "./schema_v1_to_v2.js";
import * as V2 from "./schema_v2.js";
import { Duration } from "./time.js";
import {
	enforce,
	enforce_eq,
	enforce_nonnull,
	path_hd,
	path_index,
	path_strip_trailing_index,
	path_to_branch,
	pause,
	snake_case,
	unreachable,
} from "./utilities.js";
import { status_is, status_sparsely_matches } from "./write_validators.js";
export class ThumbnailListener {
	id;
	destructor;
	constructor(id, destructor) {
		(this.id = id), (this.destructor = destructor);
	}
	unregister() {
		this.destructor(this);
	}
}
var ConnectionStage, TowelState, ExpectedMessageType;
!(function (ConnectionStage) {
	(ConnectionStage[(ConnectionStage.Disconnected = 0)] = "Disconnected"),
		(ConnectionStage[(ConnectionStage.ConnectionPending = 1)] =
			"ConnectionPending"),
		(ConnectionStage[(ConnectionStage.ConnectionEstablished = 2)] =
			"ConnectionEstablished");
})(ConnectionStage || (ConnectionStage = {})),
	(function (TowelState) {
		(TowelState[(TowelState.Blocking = 0)] = "Blocking"),
			(TowelState[(TowelState.Set = 1)] = "Set"),
			(TowelState[(TowelState.Unset = 2)] = "Unset");
	})(TowelState || (TowelState = {}));
export function stringify_failure(failure, vsocket) {
	switch (failure.type) {
		case "access-denied":
			return `Unable to access ${failure.kwl}${void 0 !== failure.kw ? `.${failure.kw}` : ""} at ${vsocket.ip}: ${failure.reason}`;
		case "create-row":
			return (
				`Unable to create new row within ${failure.kwl} at ${vsocket.ip}` +
				(void 0 !== failure.opts.allow_reuse_row ||
				void 0 !== failure.opts.index ||
				void 0 !== failure.opts.name
					? ` (options were ${JSON.stringify(failure.opts)})`
					: "")
			);
		case "rename-row":
			return `Unable to rename ${failure.kwl} at ${vsocket.ip} to ${failure.name}`;
		case "delete-rows":
			return `Unable to delete ${failure.kwls.join(", ")} at ${vsocket.ip}`;
		case "write":
			return (
				`Unable to change ${failure.kwl}.${failure.kw} at ${vsocket.ip} to ${failure.payload}` +
				(failure.reason ? `: ${failure.reason}` : "")
			);
		case "read":
			return `Unable to read ${failure.kwl}.${failure.kw} from ${vsocket.ip}`;
		case "wait-until":
			return `wait_until(${failure.kwl}.${failure.kw}, ${failure.criterion}) at ${vsocket.ip} failed to resolve within ${failure.timeout.toString("convenient")}`;
	}
}
function make_throwing_failure_handler(vsocket) {
	return (failure) => {
		throw new Error(stringify_failure(failure, vsocket));
	};
}
function generate_random_towel() {
	let res = "random-towel-";
	for (let i = 0; i < 20; ++i)
		res += Math.round(36 * Math.random()).toString(36);
	return res;
}
!(function (ExpectedMessageType) {
	(ExpectedMessageType[(ExpectedMessageType.JSON = 0)] = "JSON"),
		(ExpectedMessageType[(ExpectedMessageType.ThumbnailHeader = 1)] =
			"ThumbnailHeader"),
		(ExpectedMessageType[(ExpectedMessageType.Thumbnail = 2)] = "Thumbnail"),
		(ExpectedMessageType[(ExpectedMessageType.File = 3)] = "File");
})(ExpectedMessageType || (ExpectedMessageType = {}));
export class VSocket {
	static INTERVAL_SECS = 1;
	static schema_cache = new ResourceCache();
	static s_timeout_scale = 1;
	handle_error(error) {
		this.connection_pars.event_handler({ event_type: "error", error: error });
	}
	m_pending_subscriptions = {
		burstcount: 0,
		pending_flush: null,
		per_kwl: new Map(),
	};
	m_kwl_cache = new KWLCache();
	state = { stage: 0, reconnect_interval_ms: null };
	max_messages_per_burst = 1e3;
	get check_component_liveness() {
		return !this.connection_pars.permit_degraded_mode;
	}
	static scaledTimeout(suggested) {
		return (suggested ?? new Duration(5, "s")).times(VSocket.s_timeout_scale);
	}
	dump_stats() {
		this.m_kwl_cache.dump_stats(), this.listener_registry.dump_stats();
	}
	static setTimeoutScale(factor) {
		this.s_timeout_scale = factor;
	}
	connection_pars;
	outbound_snooper = null;
	listener_registry = new ListenerRegistry((error) =>
		this.connection_pars.event_handler({ event_type: "error", error: error }),
	);
	prune_listeners(keep_if) {
		this.listener_registry.prune_listeners(keep_if);
	}
	constructor(pars) {
		const protocol = pars.protocol ?? "ws",
			ip = pars.ip ?? get_env().IP;
		if (!ip)
			throw new Error(
				"No IP address has been specified, and no default address has been set through an IP=... environment parameter",
			);
		this.connection_pars = {
			...pars,
			ip: ip,
			login: pars.login ?? null,
			protocol: protocol,
			towel: pars.towel ?? generate_random_towel(),
			port: pars.port ?? ("ws" === protocol ? 80 : 443),
			cache_everything: pars.cache_everything ?? !1,
			reject_unauthorized: pars.reject_unauthorized ?? !1,
			remove_unhandled_subscriptions: pars.remove_unhandled_subscriptions ?? !0,
			timeout: VSocket.scaledTimeout(pars.timeout),
			max_payload_size: pars.max_payload_size ?? 4194304,
			ignored_modules: pars.ignored_modules ?? [],
			check_towel_on_write: pars.check_towel_on_write ?? !0,
			event_handler:
				pars.event_handler ??
				((event) => {
					if ("error" === event.event_type)
						setTimeout(() => {
							throw event.error;
						}, 0);
				}),
			permit_degraded_mode: pars.permit_degraded_mode ?? !1,
			failure_handler:
				pars.failure_handler ?? make_throwing_failure_handler(this),
		};
	}
	async do_check_write(full_kwl, opts) {
		if (
			!(opts?.check_component_liveness ?? this.check_component_liveness) ||
			this.module_registry.is_online(component_of_kwl(full_kwl))
		) {
			if (opts?.check_towel ?? this.connection_pars.check_towel_on_write) {
				switch (this.current_towel.state) {
					case 0:
						return void this.connection_pars.failure_handler({
							type: "access-denied",
							kwl: full_kwl,
							reason: `blocked by towel '${this.current_towel.value}'`,
						});
					case 1:
						return;
					case 2:
						await this.place_towel();
				}
				enforce_eq(this.current_towel.state, 1);
			}
		} else
			this.connection_pars.failure_handler({
				type: "access-denied",
				kwl: full_kwl,
				reason: `component \`${component_of_kwl(full_kwl)}\` appears to be offline.`,
			});
	}
	is_ready() {
		return 2 === this.state.stage;
	}
	get_state() {
		if (2 !== this.state.stage)
			throw new Error(
				`Unable to access connection state, as this VSocket currently isn't connected to ${this.connection_pars.ip}`,
			);
		return this.state;
	}
	get protocol_features() {
		return this.state.protocol_features;
	}
	set_snooper(outbound) {
		this.outbound_snooper = outbound;
	}
	set_failure_handler(failure_handler) {
		this.connection_pars.failure_handler =
			failure_handler ?? make_throwing_failure_handler(this);
	}
	identify() {
		return `${this.protocol}://${this.ip}`;
	}
	get current_towel() {
		return this.get_state().current_towel;
	}
	get build_info() {
		return this.get_state().build_info;
	}
	get ip() {
		return this.connection_pars.ip;
	}
	get login() {
		return this.connection_pars.login
			? { ...this.connection_pars.login }
			: null;
	}
	get protocol() {
		return this.connection_pars.protocol;
	}
	get port() {
		return this.connection_pars.port;
	}
	get towel() {
		return this.connection_pars.towel;
	}
	get runtime_constants() {
		return this.get_state().runtime_constants;
	}
	get schema() {
		return this.get_state().schema;
	}
	get module_registry() {
		return this.get_state().module_registry;
	}
	get root() {
		return new Root(this);
	}
	async place_towel(pars) {
		const state = this.get_state();
		if (0 === state.current_towel.state && !pars?.override_preexisting_towel)
			return void this.connection_pars.failure_handler({
				type: "access-denied",
				kwl: "system.usrinfo",
				kw: "towel",
				reason:
					"currently blocked by preexisting towel. If you wish to clear it anyway, use await place_towel({ override_preexisting_towel: true })",
			});
		const old_towel = { ...state.current_towel };
		state.current_towel = { value: this.towel, state: 1 };
		try {
			await this.write({ kwl: "system.usrinfo", kw: "towel" }, this.towel, {
				check_towel: !1,
			});
		} catch (e) {
			throw ((state.current_towel = old_towel), e);
		}
	}
	async subscription_flush() {
		this.m_pending_subscriptions.pending_flush &&
			(await this.m_pending_subscriptions.pending_flush);
	}
	async watch_error_channels(handler, opts) {
		const result = [];
		return (
			await Promise.all(
				this.schema.error_feedback_channels.map(async (channel) => {
					const owningModule = channel.owner;
					void 0 !==
						this.schema.keywords.find(
							(comp) =>
								comp.owning_module === owningModule && !0 === comp.enabled,
						) &&
						result.push(
							await this.watch(
								{ kwl: channel.kwl, kw: "error_feedback" },
								(raw) => {
									const msg = read_error_message(raw);
									msg && handler(msg);
								},
								{
									check_component_liveness: !1,
									...(opts?.listener_id
										? { listener_id: `_err_${channel.kwl}_${opts.listener_id}` }
										: {}),
								},
							),
						);
				}),
			),
			result
		);
	}
	recover_subscriptions() {
		for (const [kwl, { complete: complete, partial: partial }] of this
			.listener_registry.kwl_listeners)
			if (0 !== Object.keys(complete).length) this.subscribe_to({ kwl: kwl });
			else for (const kw in partial) this.subscribe_to({ kwl: kwl, kw: kw });
	}
	async marker(pars) {
		const state = this.get_state(),
			id = state.next_marker_id,
			did_collide = state.pending_markers.has(id),
			pending_markers = state.pending_markers;
		return (
			(state.next_marker_id = (id + 1) % 32768),
			(pars?.flush_subscriptions ?? 1) && (await this.subscription_flush()),
			new Promise((resolve, reject) => {
				if (did_collide) reject("Marker IDs exhausted");
				else {
					this.send([{ op: "marker", marker: 32768 | id }]);
					const expiryDateMS =
						new Date().valueOf() + VSocket.scaledTimeout(pars?.timeout).ms();
					pending_markers.set(id, {
						expiryDateMS: expiryDateMS,
						resolve: resolve,
						reject: reject,
					});
				}
			})
		);
	}
	query_cache(path) {
		return this.m_kwl_cache.query_kw(path);
	}
	query_cache_kwl(kwl) {
		return this.m_kwl_cache.query_kwl(kwl);
	}
	do_get_runtime_constant(constants, constant_string, runtime_constants) {
		const parts = constant_string.split("::"),
			[ua_name, constant_name] = parts;
		if (!constants[ua_name][constant_name])
			throw new Error(
				`Unable to read runtime constant ${ua_name}.${constant_name}; perhaps you are trying to connect to an old software build?`,
			);
		const sysname_parts = constants[ua_name][constant_name].path.split(".");
		return runtime_constants.get_constant(sysname_parts[0], sysname_parts[1]);
	}
	get_runtime_constant(constant_string) {
		return this.do_get_runtime_constant(
			this.schema.constants,
			constant_string,
			this.runtime_constants,
		);
	}
	send(req) {
		if (0 === this.state.stage)
			throw new Error(
				`Unable to dispatch request ${JSON.stringify(req)}, as this socket is currently not connected to ${this.connection_pars.ip}`,
			);
		this.state.websocket.send(JSON.stringify(req)),
			this.outbound_snooper?.(req);
	}
	async read_unchecked(path, raw_timeout) {
		await this.subscription_flush();
		const timeout = VSocket.scaledTimeout(raw_timeout);
		return await new Promise((resolve, reject) => {
			let watchdog = null;
			const listener = this.register_kw_listener(
				{ kwl: path.kwl, kw: path.kw, listener_type: 1, execution_strategy: 0 },
				(payload) => {
					clearTimeout(watchdog), resolve(payload);
				},
			);
			(watchdog = setTimeout(() => {
				listener.unregister(),
					reject(
						new Error(`Unable to read ${path.kwl}.${path.kw} from ${this.ip}`),
					);
			}, timeout.ms())),
				this.send([{ op: "readAll", kwl: path.kwl, kw: path.kw }]);
		});
	}
	write_unchecked(path, payload) {
		this.send([{ op: "data", kwl: path.kwl, kw: { [path.kw]: payload } }]);
	}
	static async open(pars) {
		const result = new VSocket(pars);
		return await result.do_open_connection("first-connect"), result;
	}
	do_open_connection(mode) {
		const aborted = Symbol("aborted"),
			timeout = VSocket.scaledTimeout(this.connection_pars.timeout);
		let did_open_at_least_once = "reconnect" === mode,
			timer_pending = !1;
		const arm_reconnect_timer = () => {
				if (!timer_pending) {
					if (0 !== this.state.stage)
						try {
							this.state.websocket.close();
						} catch (_) {
						} finally {
							this.state = { stage: 0, reconnect_interval_ms: 250 };
						}
					if (
						0 !== this.state.stage ||
						null !== this.state.reconnect_interval_ms
					) {
						const reconnect_interval_ms =
							0 !== this.state.stage
								? 250
								: Math.min(
										Math.round(1.5 * this.state.reconnect_interval_ms),
										2e3,
									);
						(this.state = {
							stage: 0,
							reconnect_interval_ms: reconnect_interval_ms,
						}),
							setTimeout(async () => {
								if (
									((timer_pending = !1),
									null !== this.state.reconnect_interval_ms)
								) {
									try {
										await this.do_open_connection("reconnect"),
											this.recover_subscriptions();
									} catch (_) {
										return;
									}
									this.connection_pars.event_handler?.({
										event_type: "connection-reopened",
									});
								} else
									this.connection_pars.event_handler({
										event_type: "info",
										msg: "Aborting reconnect attempt (based on the assumption that this socket has been closed deliberately)",
									});
							}, reconnect_interval_ms),
							(timer_pending = !0);
					}
				}
			},
			open_websocket = () =>
				new Promise((resolve, reject) => {
					const websocket = new WebSocket.Adapter({
						...this.connection_pars,
						timeout: timeout,
						on_close: () => {
							try {
								this.m_kwl_cache.clear(),
									this.connection_pars.event_handler(
										2 === this.state.stage &&
											null !== this.state.expecting_close_because
											? {
													event_type: "expected-close",
													reason: this.state.expecting_close_because,
												}
											: { event_type: "unexpected-close" },
									);
							} finally {
								did_open_at_least_once && arm_reconnect_timer();
							}
						},
						on_error: () => {
							try {
								this.connection_pars.event_handler({
									event_type: "websocket-error",
								});
							} finally {
								did_open_at_least_once && arm_reconnect_timer();
							}
						},
						reject: (err) => reject(err),
						on_first_msg: (msg_data) => {
							try {
								let obj = JSON.parse(msg_data);
								if (!Array.isArray(obj))
									return void reject(
										`Got unexpected first message: should have been an array [{ "webserver_buildinfo": <...> }]), but was '${msg_data}'`,
									);
								if (
									((obj = obj[0]),
									!Object.prototype.hasOwnProperty.call(
										obj,
										"webserver_buildinfo",
									))
								)
									throw new Error(
										`Got unexpected first message from ${this.connection_pars.ip}: should have contained a field named 'webserver_buildinfo'`,
									);
								Object.prototype.hasOwnProperty.call(obj, "protocol_features")
									? resolve([
											obj.webserver_buildinfo,
											{
												create_row_request_id:
													obj.protocol_features.create_row_request_id ?? !1,
												runtime_constants_json:
													obj.protocol_features.runtime_constants_json ?? !1,
												readable_command_tracking:
													obj.protocol_features.readable_command_tracking ?? !1,
												vscriptd: obj.protocol_features.vscriptd ?? !1,
												http_network_config:
													obj.protocol_features.http_network_config ?? !1,
											},
											websocket,
										])
									: resolve([
											obj.webserver_buildinfo,
											{
												create_row_request_id: !1,
												runtime_constants_json: !1,
												readable_command_tracking: !1,
												vscriptd: obj.protocol_features.vscriptd ?? !1,
												http_network_config:
													obj.protocol_features.http_network_config ?? !1,
											},
											websocket,
										]);
							} catch (e) {
								reject(e);
							}
						},
						on_further_msg: (msg) => this.handle_incoming(msg),
					});
				});
		return with_timeout(timeout, async (context) => {
			context.desc = `while trying to open WebSocket connection to ${this.identify()}`;
			let websocket = null;
			try {
				const [webserver_build_info, protocol_features, ws] =
					await open_websocket();
				(websocket = ws),
					(this.state = {
						stage: 1,
						expected_message_type: 0,
						websocket: websocket,
						webserver_build_info: webserver_build_info,
						protocol_features: protocol_features,
					});
				const protocol =
					"ws" === this.connection_pars.protocol ? "http" : "https";
				if (context.timed_out) throw aborted;
				context.desc = `while trying to download data/build_info.json from ${this.identify()}`;
				const build_info = await download_json({
					ip: this.connection_pars.ip,
					protocol: protocol,
					port: this.connection_pars.port,
					path: "/data/build_info.json",
					reject_unauthorized: this.connection_pars.reject_unauthorized,
					timeout: timeout,
				});
				if (context.timed_out) throw aborted;
				const key_suffix = build_info.commit.endsWith("+")
						? "|" + build_info.timestamp
						: "",
					key =
						build_info.commit +
						key_suffix +
						(build_info.hardware_model ?? "<unknown hardware model>");
				if (
					((context.desc = `while trying to download data/schema.json from ${this.identify()}`),
					context.timed_out)
				)
					throw aborted;
				const raw_schema = ensure_v2(
					await VSocket.schema_cache.retrieve({
						ip: this.connection_pars.ip,
						protocol: protocol,
						port: this.connection_pars.port,
						path: "/data/schema.json",
						key: key,
						reject_unauthorized: this.connection_pars.reject_unauthorized,
						timeout: timeout,
					}),
				);
				if (context.timed_out) throw aborted;
				const ignored_component_ua_names = raw_schema.keywords
					.filter(
						(comp) =>
							-1 !==
							this.connection_pars.ignored_modules.findIndex(
								(m) => m === comp.owning_module,
							),
					)
					.map((comp) => comp.ua_name);
				if (0 !== this.connection_pars.ignored_modules.length) {
					raw_schema.keywords = raw_schema.keywords.filter(
						(comp) =>
							-1 ===
							this.connection_pars.ignored_modules.findIndex(
								(m) => m === comp.owning_module,
							),
					);
					const is_ignored = (td_name) => {
							const comp_ua_name = td_name.split("::")[0];
							return (
								-1 !==
								ignored_component_ua_names.findIndex(
									(ua) => comp_ua_name === ua,
								)
							);
						},
						old_typedefs = raw_schema.typedefs,
						old_typedef_identifiers = raw_schema.typedef_identifiers;
					(raw_schema.typedefs = {}), (raw_schema.typedef_identifiers = []);
					for (const td_name in old_typedefs)
						is_ignored(td_name) ||
							(raw_schema.typedefs[td_name] = old_typedefs[td_name]);
					for (const td_name of old_typedef_identifiers)
						is_ignored(td_name) || raw_schema.typedef_identifiers.push(td_name);
				}
				if (
					((context.desc = `while trying to initialize module registry for ${this.identify()}`),
					context.timed_out)
				)
					throw aborted;
				const module_registry = await ModuleRegistry.initialize(
					raw_schema.keywords,
					this,
					this.connection_pars.permit_degraded_mode,
					context,
				);
				if (context.timed_out) throw aborted;
				const [runtime_constants, disabledComponents] =
					await RuntimeConstants.initialize(raw_schema, this, timeout, context);
				if (context.timed_out) throw aborted;
				const schema = V2.annotate(
					raw_schema,
					disabledComponents,
					(constant_name) =>
						this.do_get_runtime_constant(
							raw_schema.constants,
							constant_name,
							runtime_constants,
						),
				);
				(this.state = {
					stage: 2,
					build_info: build_info,
					current_towel: { state: 2, value: "" },
					expecting_close_because: null,
					expected_message_type: 0,
					module_registry: module_registry,
					next_marker_id: 0,
					pending_markers: new Map(),
					protocol_features: protocol_features,
					reconnect_on_close: !0,
					runtime_constants: runtime_constants,
					schema: schema,
					thumbnails: null,
					webserver_build_info: this.state.webserver_build_info,
					websocket: this.state.websocket,
				}),
					this.register_kw_listener(
						{
							listener_id: "_towel_watch",
							kwl: "system.usrinfo",
							kw: "towel",
							ensure_initial_read: !0,
							execution_strategy: 0,
							cache: !0,
							listener_type: 0,
						},
						(new_towel) => {
							2 === this.state.stage &&
								(this.state.current_towel = {
									value: new_towel,
									state:
										new_towel === this.connection_pars.towel
											? 1
											: 0 === new_towel.length
												? 2
												: 0,
								});
						},
					),
					this.send([
						{ op: "subscribe", kwl: "system.usrinfo", kw: ["towel"] },
					]);
				for (const [kwl, cp] of this.listener_registry.kwl_listeners)
					if (Object.keys(cp.complete).length > 0)
						this.subscribe_to({ kwl: kwl, ensure_initial_read: !0 });
					else
						for (const kw in cp.partial)
							this.subscribe_to({ kwl: kwl, kw: kw, ensure_initial_read: !0 });
				did_open_at_least_once = !0;
			} catch (e) {
				if (
					(did_open_at_least_once ? arm_reconnect_timer() : await this.close(),
					typeof e != typeof aborted)
				)
					throw e;
			}
			2 === this.state.stage &&
				this.connection_pars.event_handler({
					event_type: "connection-reopened",
				});
		});
	}
	async close() {
		try {
			2 === this.state.stage &&
				this.state.current_towel.value.length > 0 &&
				this.state.current_towel.value === this.connection_pars.towel &&
				(await this.write({ kwl: "system.usrinfo", kw: "towel" }, ""));
		} catch (_) {
			this.write_unchecked({ kwl: "system.usrinfo", kw: "towel" }, "");
		} finally {
			0 !== this.state.stage && this.state.websocket.close(),
				(this.state = { stage: 0, reconnect_interval_ms: null }),
				this.listener_registry.close();
		}
	}
	async recovery(pars) {
		const timeout = VSocket.scaledTimeout(
				pars?.timeout ??
					Duration.max(this.connection_pars.timeout, new Duration(3, "min")),
			),
			interval = new Duration(100, "ms");
		for (
			const t0 = new Date().valueOf();
			new Date().valueOf() < t0 + timeout.ms();
		) {
			if (this.is_ready())
				return void (await this.read({
					kwl: "system.usrinfo",
					kw: "towel",
				}).catch(() => {}));
			await pause(interval);
		}
		throw new Error(
			`${this.connection_pars.ip} failed to recover within ${Math.round(timeout.s())} s; maybe the machine had to swap partitions?`,
		);
	}
	register_kw_listener(pars, handler) {
		if (!this.listener_registry.have_permanent_listener(pars))
			switch (pars.listener_type) {
				case 1:
					this.send([{ op: "readAll", kwl: pars.kwl, kw: [pars.kw] }]);
					break;
				case 0:
					if (0 === pars.execution_strategy)
						try {
							this.send([
								pars.kw
									? { kwl: pars.kwl, kw: [pars.kw], op: "subscribe" }
									: { kwl: pars.kwl, op: "subscribe" },
							]);
						} catch (_) {
							this.subscribe_to(pars);
						}
					else this.subscribe_to(pars);
			}
		return this.listener_registry.register_kw_listener(pars, handler);
	}
	register_kwl_listener(pars, handler) {
		if (!this.listener_registry.have_permanent_listener(pars))
			if (0 === pars.execution_strategy)
				try {
					this.send([{ op: "subscribe", kwl: pars.kwl }]);
				} catch (_) {
					this.subscribe_to(pars);
				}
			else this.subscribe_to(pars);
		return this.listener_registry.register_kwl_listener(pars, handler);
	}
	register_global_listener(handler, pars) {
		return this.listener_registry.register_global_listener(handler, pars);
	}
	subscribe_to(pars) {
		if (
			this.state &&
			0 !== this.state.stage &&
			this.m_pending_subscriptions.burstcount++ < this.max_messages_per_burst
		)
			this.send([
				pars.kw
					? { kwl: pars.kwl, kw: [pars.kw], op: "subscribe" }
					: { kwl: pars.kwl, op: "subscribe" },
			]);
		else {
			const entry = this.m_pending_subscriptions.per_kwl.get(pars.kwl);
			void 0 === entry
				? this.m_pending_subscriptions.per_kwl.set(pars.kwl, {
						ensure_initial_read: pars.ensure_initial_read ?? !0,
						kws: void 0 === pars.kw ? void 0 : [pars.kw],
					})
				: ((entry.ensure_initial_read ||= pars.ensure_initial_read ?? !0),
					void 0 === pars.kw || void 0 === entry.kws
						? (entry.kws = void 0)
						: entry.kws.find((other_kw) => other_kw === pars.kw) ||
							entry.kws.push(pars.kw)),
				null === this.m_pending_subscriptions.pending_flush &&
					(this.m_pending_subscriptions.pending_flush = (async () => {
						for (; 0 !== this.m_pending_subscriptions.per_kwl.size; ) {
							for (; !this.state || 0 === this.state.stage; )
								await pause(new Duration(100, "ms"));
							let n = 0;
							const outbound = [];
							for (const [kwl, entry] of this.m_pending_subscriptions.per_kwl)
								if (
									(null === entry.kws
										? this.send([{ op: "subscribe", kwl: kwl }])
										: this.send([{ op: "subscribe", kwl: kwl, kw: entry.kws }]),
									(n += 1),
									entry.ensure_initial_read &&
										(this.send([{ op: "readAll", kwl: kwl }]), (n += 1)),
									outbound.push([kwl, entry]),
									this.m_pending_subscriptions.per_kwl.delete(kwl),
									n > this.max_messages_per_burst)
								)
									break;
							try {
								await this.marker({ flush_subscriptions: !1 });
							} catch (e) {
								for (const [kwl, entry] of outbound)
									this.m_pending_subscriptions.per_kwl.set(kwl, entry);
								break;
							} finally {
								0 === this.m_pending_subscriptions.per_kwl.size &&
									(this.m_pending_subscriptions.pending_flush = null);
							}
						}
					})());
		}
	}
	register_thumbnail_listener(handler, pars) {
		const state = this.get_state(),
			kwl = "monitoring.live_view";
		null === state.thumbnails &&
			(this.send([
				{ op: "subscribe", kwl: kwl, kw: ["thumbnail_header", "thumbnail"] },
			]),
			(state.thumbnails = {
				header_data: void 0,
				prev_data: void 0,
				handlers: new Map(),
			})),
			state.thumbnails;
		const id = StringIDPool.generate_or_use_id(
			state.thumbnails.handlers,
			pars?.listener_id,
		);
		state.thumbnails.handlers.set(id, handler);
		return (
			this.listener_registry.register_kw_listener(
				{
					kwl: kwl,
					kw: "thumbnail",
					listener_id: "_bogus_listener_thumbnails",
					cache: !1,
					listener_type: 0,
					execution_strategy: 1,
					ensure_initial_read: !1,
				},
				() => {},
			),
			new ThumbnailListener(id, () => {
				state.thumbnails &&
					(state.thumbnails.handlers.delete(id),
					0 === state.thumbnails.handlers.size &&
						((state.thumbnails = null),
						this.listener_registry.safely_delete_handler({
							kwl: kwl,
							kw: "thumbnail",
							id: "_bogus_listener_thumbnails",
						}),
						this.send([
							{
								op: "unsubscribe",
								kwl: kwl,
								kw: ["thumbnail_header", "thumbnail"],
							},
						])));
			})
		);
	}
	handle_incoming(msg) {
		const state = this.state;
		switch (state.expected_message_type) {
			case 0:
				{
					const data = JSON.parse(msg.data),
						len = data.length;
					for (let idx = 0; idx < len; idx++) {
						const obj = data[idx];
						if (Object.prototype.hasOwnProperty.call(obj, "kwl")) {
							if (!Object.prototype.hasOwnProperty.call(obj, "kw")) {
								this.handle_error(
									new Error("Encountered malformed message:" + obj),
								);
								continue;
							}
							this.connection_pars.cache_everything &&
								this.m_kwl_cache.register(obj);
							for (const gl_entry of this.listener_registry.global_listeners)
								try {
									gl_entry[1](obj);
								} catch (e) {
									this.handle_error(e);
								}
							let unsubscribe_from_unlistened =
								this.connection_pars.remove_unhandled_subscriptions;
							const kwl_listeners = this.listener_registry.kwl_listeners.get(
								obj.kwl,
							);
							if (void 0 === kwl_listeners) {
								unsubscribe_from_unlistened &&
									this.send([{ op: "unsubscribe", kwl: obj.kwl }]);
								continue;
							}
							for (const id in kwl_listeners.complete) {
								unsubscribe_from_unlistened = !1;
								const entry = kwl_listeners.complete[id];
								if (0 === entry.execution_strategy)
									try {
										entry.handler();
									} catch (e) {
										this.handle_error(e);
									}
								else
									this.listener_registry.pending_lazy_updates.kwls.set(
										id,
										entry.handler,
									);
							}
							for (const kw_name in obj.kw) {
								const payload = obj.kw[kw_name],
									kw_listeners = kwl_listeners.partial[kw_name];
								if (void 0 === kw_listeners)
									unsubscribe_from_unlistened &&
										this.send([
											{ op: "unsubscribe", kwl: obj.kwl, kw: [kw_name] },
										]);
								else
									for (const id in kw_listeners) {
										const entry = kw_listeners[id];
										if (0 === entry.execution_strategy)
											try {
												entry.handler(payload);
											} catch (e) {
												this.handle_error(e);
											}
										else
											this.listener_registry.pending_lazy_updates.kws.set(id, [
												entry.handler,
												payload,
											]);
									}
							}
						} else if (Object.prototype.hasOwnProperty.call(obj, "file"))
							state.expected_message_type =
								"thumbnail_header" === obj.file ? 1 : 3;
						else if (Object.prototype.hasOwnProperty.call(obj, "thumbnail"))
							state.expected_message_type = 2;
						else if (Object.prototype.hasOwnProperty.call(obj, "marker")) {
							const id = 32767 & obj.marker;
							state.pending_markers.has(id)
								? (state.pending_markers.get(id).resolve(),
									state.pending_markers.delete(id))
								: this.handle_error(
										new Error(`Received unexpected marker #${id}`),
									);
						} else
							("login" in obj && 1 === Object.keys(obj).length) ||
								this.handle_error(
									new Error("Received unexpected message:" + obj),
								);
					}
				}
				return;
			case 3:
				break;
			case 1:
				{
					enforce_eq(this.state.stage, 2);
					const state = this.state;
					state.thumbnails && (state.thumbnails.header_data = msg.data);
				}
				break;
			case 2: {
				enforce_eq(this.state.stage, 2);
				const state = this.state;
				if (state.thumbnails) {
					state.thumbnails.prev_data = binary_concat(
						state.thumbnails.header_data,
						msg.data,
					);
					for (const [_, handler] of state.thumbnails.handlers)
						try {
							handler(state.thumbnails.prev_data);
						} catch (e) {
							this.handle_error(e);
						}
				}
			}
		}
		state.expected_message_type = 0;
	}
	async read(full_path, opts) {
		if (
			(this.do_check_readonly(full_path.kwl, opts),
			opts?.use_cache_if_present ?? 1)
		) {
			const maybe_cached = this.query_cache(full_path);
			if (maybe_cached !== MissingData) return maybe_cached[1];
		}
		return await this.read_unchecked(full_path, opts?.timeout);
	}
	async wait_until_offline(pars) {
		const timeout =
				pars?.timeout ??
				Duration.max(this.connection_pars.timeout, new Duration(2, "min")),
			deadline = new Date().valueOf() + timeout.ms();
		for (; this.is_ready(); )
			if (
				(await pause(new Duration(100, "ms")), new Date().valueOf() > deadline)
			)
				throw new Error(
					`Timed out while waiting for ${this.identify()} to go offline`,
				);
	}
	async reboot(pars) {
		await this.do_check_write("system", {
			check_component_liveness: !0,
			check_towel: !0,
		}),
			(this.get_state().expecting_close_because = "reboot"),
			this.write_unchecked(
				{ kwl: "system", kw: "reboot" },
				pars?.command ?? "reboot",
			),
			await this.wait_until_offline(),
			await this.recovery({ ...pars });
	}
	async reset(pars) {
		await this.do_check_write("system", {
			check_component_liveness: !0,
			check_towel: !0,
		}),
			(this.get_state().expecting_close_because = "reset"),
			this.write_unchecked(
				{ kwl: "system", kw: "reset" },
				pars?.partition ?? "reset",
			),
			await this.wait_until_offline(),
			await this.recovery({ ...pars });
	}
	async table_create_row(opts) {
		await this.subscription_flush(),
			await this.do_check_write(opts.table_kwl, opts);
		const with_index = void 0 !== opts?.index,
			with_name = void 0 !== opts?.name,
			request_id = NumericIDPool.generate_id(),
			supports_request_id = !0 === this.protocol_features.create_row_request_id,
			payload = (() => {
				if (
					with_index ||
					with_name ||
					void 0 !== opts?.allow_reuse_row ||
					supports_request_id
				) {
					const result = {};
					return (
						with_index && (result.index = opts.index),
						with_name && (result.name = opts.name),
						void 0 !== opts?.allow_reuse_row &&
							(result.allow_reuse_row = opts.allow_reuse_row),
						supports_request_id && (result.request_id = request_id),
						result
					);
				}
				return null;
			})(),
			response = await new Promise((resolve) => {
				const timeout = VSocket.scaledTimeout(opts?.timeout);
				let watchdog = null;
				const listener = this.register_kw_listener(
					{
						kwl: opts.table_kwl,
						kw: "created_row",
						listener_type: 0,
						execution_strategy: 0,
					},
					(r) => {
						(!supports_request_id || r[2] === request_id) &&
							(null === r[0]
								? (clearTimeout(watchdog), resolve(null))
								: (with_index && opts?.index !== r[0]) ||
									(with_name && opts?.name !== r[1]) ||
									(clearTimeout(watchdog), resolve([r[0], r[1]])));
					},
				);
				(watchdog = setTimeout(() => {
					listener.unregister(), resolve(null);
				}, timeout.ms())),
					this.write_unchecked(
						{ kwl: opts.table_kwl, kw: "create_row" },
						payload,
					);
			});
		if (null === response) {
			const failure = { type: "create-row", kwl: opts.table_kwl, opts: opts };
			throw (
				(this.connection_pars.failure_handler(failure),
				new Error(stringify_failure(failure, this)))
			);
		}
		return response;
	}
	async table_rename_row(opts) {
		await this.subscription_flush();
		try {
			await this.write(
				{ kwl: opts.row_kwl, kw: "row_name_command" },
				opts.desired_name,
				{ ...opts, check_keyword_type: !1 },
			);
		} catch (_) {
			this.connection_pars.failure_handler({
				type: "rename-row",
				kwl: opts.row_kwl,
				name: opts.desired_name,
			});
		}
	}
	async table_indices(opts) {
		await this.subscription_flush(),
			this.do_check_readonly(opts.table_kwl, opts);
		const rowmask = await this.read(
			{ kwl: opts.table_kwl, kw: "rowMask" },
			opts,
		);
		return indices_of_rowmask(rowmask);
	}
	async table_has_row(opts) {
		await this.subscription_flush();
		return (await this.table_indices(opts)).indexOf(opts.index) >= 0;
	}
	async table_delete_row(opts) {
		await this.subscription_flush();
		const listener = this.register_kw_listener(
			{
				kwl: opts.table_kwl,
				kw: "rowMask",
				cache: !0,
				listener_type: 0,
				execution_strategy: 0,
			},
			() => {},
		);
		try {
			await this.write(
				{ kwl: `${opts.table_kwl}[${opts.index}]`, kw: "row_cmd" },
				"DELETE_ROW",
				{
					...opts,
					retry_until: {
						criterion: "custom",
						validator: async () => !(await this.table_has_row(opts)),
					},
					check_keyword_type: !1,
				},
			);
		} catch (_) {
			this.connection_pars.failure_handler({
				type: "delete-rows",
				kwls: [`${opts.table_kwl}[${opts.index}]`],
			});
		} finally {
			listener.unregister();
		}
	}
	async table_delete_all_rows(opts) {
		await this.subscription_flush();
		const listener = this.register_kw_listener(
			{
				kwl: opts.table_kwl,
				kw: "rowMask",
				cache: !0,
				listener_type: 0,
				execution_strategy: 0,
			},
			() => {},
		);
		try {
			await this.write(
				{ kwl: opts.table_kwl, kw: "table_cmd" },
				"DELETE_ALL_ROWS",
				{
					...opts,
					retry_until: {
						criterion: "custom",
						validator: async () =>
							0 === (await this.table_indices(opts)).length,
					},
					check_keyword_type: !1,
				},
			);
		} catch (e) {
			this.connection_pars.failure_handler({
				type: "delete-rows",
				kwls: (await this.table_indices(opts)).map(
					(i) => `${opts.table_kwl}[${i}]`,
				),
			});
		} finally {
			listener.unregister();
		}
	}
	async write(full_path, payload, opts) {
		let can_read = !0,
			desc = null;
		if (opts?.check_keyword_type ?? 1) {
			if (
				((desc = V2.find_keyword(
					this.schema,
					path_to_branch(full_path.kwl),
					full_path.kw,
				)),
				"status" === desc.kw_type || "driver-owned event" === desc.kw_type)
			)
				return void this.connection_pars.failure_handler({
					type: "write",
					kwl: full_path.kwl,
					kw: full_path.kw,
					payload: payload,
					reason: `this keyword is of type '${desc.kw_type}' and hence cannot be written to`,
				});
			can_read = !(
				"event" === desc.kw_type ||
				("immediate-tracking" === desc.kw_priority &&
					!this.protocol_features.readable_command_tracking)
			);
		}
		await this.do_check_write(full_path.kwl, opts);
		const status_kw = full_path.kw.endsWith("_command")
				? full_path.kw.replace(/_command$/, "_status")
				: full_path.kw,
			timeout = VSocket.scaledTimeout(opts?.timeout),
			retry_interval_ms = opts?.retry_interval?.ms() ?? 250,
			indices =
				"object" != typeof payload || null === payload || Array.isArray(payload)
					? void 0
					: Object.keys(payload).map((s) => parseInt(s)),
			wv =
				opts?.retry_until ??
				(void 0 === indices
					? status_is(payload, desc)
					: status_sparsely_matches(payload));
		try {
			switch (wv.criterion) {
				case "return-immediately":
					this.write_unchecked(full_path, payload);
					break;
				case "custom": {
					const deadline = new Date().valueOf() + timeout.ms();
					let wait_interval_ms = opts?.first_write_interval?.ms() ?? 5;
					do {
						if (
							(this.write_unchecked(full_path, payload),
							await pause(new Duration(wait_interval_ms, "ms")),
							await wv.validator())
						)
							return;
						wait_interval_ms = Math.min(
							retry_interval_ms,
							Math.ceil(1.2 * wait_interval_ms),
						);
					} while (new Date().valueOf() <= deadline);
					throw void 0;
				}
				case "status":
					if ((this.write_unchecked(full_path, payload), can_read)) {
						const interval = setInterval(() => {
							this.write_unchecked(full_path, payload);
						}, retry_interval_ms);
						await this.wait_until(
							{ ...full_path, kw: status_kw },
							wv.validator,
							{ check_component_liveness: !1, timeout: timeout },
						).finally(() => clearInterval(interval));
					}
			}
		} catch (_) {
			this.connection_pars.failure_handler({
				type: "write",
				kwl: full_path.kwl,
				kw: full_path.kw,
				payload: payload,
			});
		}
	}
	async wait_until(full_path, criterion, opts) {
		await this.subscription_flush(),
			this.do_check_readonly(full_path.kwl, opts);
		const timeout = VSocket.scaledTimeout(opts?.timeout),
			err = new Error(
				`wait_until(${full_path.kwl}, ${full_path.kw})@${this.identify()}: failed to satisfy the given criterion within ${Math.round(timeout.s())} s`,
			),
			do_wait = new Promise((resolve, reject) => {
				let timer;
				const listener = this.register_kw_listener(
					{ ...full_path, execution_strategy: 0, listener_type: 0 },
					(payload) => {
						!0 === criterion(payload) &&
							((did_resolve = !0),
							listener.unregister(),
							timer && clearTimeout(timer),
							resolve(payload));
					},
				);
				let did_resolve = !1;
				(timer = setTimeout(() => {
					did_resolve || (listener.unregister(), reject(err));
				}, timeout.ms())),
					opts?.skip_initial_read || this.read(full_path, opts);
			});
		return await do_wait;
	}
	watch_unchecked(full_path, handler, opts) {
		const listener = this.register_kw_listener(
			{ ...full_path, ...opts, execution_strategy: 0, listener_type: 0 },
			handler,
		);
		return new Watcher(full_path, () => {
			listener.unregister();
		});
	}
	async watch(full_path, handler, opts) {
		return (
			await this.subscription_flush(),
			this.do_check_readonly(full_path.kwl, opts),
			this.watch_unchecked(full_path, handler, opts)
		);
	}
	do_check_readonly(full_kwl, opts) {
		(opts?.check_component_liveness ?? this.check_component_liveness) &&
			!this.module_registry.is_online(component_of_kwl(full_kwl)) &&
			this.connection_pars.failure_handler({
				type: "access-denied",
				kwl: full_kwl,
				reason: `component ${component_of_kwl(full_kwl)} appears to be offline.`,
			});
	}
}
function to_view_object(backing_store, kwl, desc) {
	switch (desc.container_type) {
		case 2:
			return new SubtreeArray(backing_store, kwl, desc);
		case 1:
			return desc.named_tables
				? new SubtreeNamedTable(backing_store, kwl, desc)
				: new SubtreeTable(backing_store, kwl, desc);
		case 0:
		case 5:
			return new Subtree(backing_store, kwl, desc);
	}
}
export class SubtreeArray {
	backing_store;
	kwl_basename;
	description;
	constructor(backing_store, kwl_basename, description) {
		(this.backing_store = backing_store),
			(this.kwl_basename = kwl_basename),
			(this.description = description);
	}
	row(index) {
		if (index >= 0 && index < this.description.capacity)
			return new Subtree(
				this.backing_store,
				`${this.kwl_basename}[${index}]`,
				this.description.contents,
			);
		throw new Error(
			`Requested array element ${this.kwl_basename}[${index}] exceeds capacity of ${this.description.capacity}`,
		);
	}
	[Symbol.iterator]() {
		const N = this.description.capacity,
			self = this;
		return (function* () {
			for (let i = 0; i < N; ++i) yield self.row(i);
		})();
	}
}
export class Subtree {
	backing_store;
	kwl;
	description;
	constructor(backing_store, kwl, description) {
		(this.backing_store = backing_store),
			(this.kwl = kwl),
			(this.description = description);
	}
	get parent() {
		if (5 === this.description.container_type) return this.backing_store.root;
		{
			const parent_kwl = this.kwl.endsWith("]")
				? path_strip_trailing_index(this.kwl)
				: path_hd(this.kwl);
			return to_view_object(
				this.backing_store,
				parent_kwl,
				this.description.parent,
			);
		}
	}
	get children() {
		return this.st_desc().children;
	}
	toString() {
		return `${this.kwl}@${this.backing_store.identify()}`;
	}
	to_full_kwl(kwl_relative) {
		return `${this.kwl}.${kwl_relative}`;
	}
	to_full_path(path_relative) {
		return {
			kwl:
				void 0 === path_relative.kwl
					? this.kwl
					: this.to_full_kwl(path_relative.kwl),
			kw: path_relative.kw,
		};
	}
	st_desc() {
		return 5 === this.description.container_type
			? this.description.contents
			: this.description;
	}
	find(relative_kwl) {
		const maybe_desc = V2.find_subtree_within(this.st_desc(), relative_kwl);
		if (void 0 !== maybe_desc)
			return to_view_object(
				this.backing_store,
				this.to_full_kwl(relative_kwl),
				maybe_desc,
			);
	}
	query_cache(path) {
		return this.backing_store.query_cache(this.to_full_path(path));
	}
	query_cache_kwl(kwl) {
		return this.backing_store.query_cache_kwl(this.to_full_kwl(kwl));
	}
	write_unchecked(path, payload) {
		this.backing_store.write_unchecked(this.to_full_path(path), payload);
	}
	async read(path, opts) {
		return await this.backing_store.read(this.to_full_path(path), opts);
	}
	read_unchecked(path, timeout) {
		return this.backing_store.read_unchecked(this.to_full_path(path), timeout);
	}
	write_tree_unchecked(kwl, payload) {
		let payload_wrapped = payload;
		const segments = this.to_full_kwl(kwl).split(".");
		for (let i = segments.length - 1; i >= 0; i--)
			payload_wrapped = { kwl: { [segments[i]]: payload_wrapped } };
		this.backing_store.send([{ op: "tree", ...payload_wrapped }]);
	}
	async write(path, payload, opts) {
		await this.backing_store.write(this.to_full_path(path), payload, opts);
	}
	async wait_until(path, criterion, opts) {
		return await this.backing_store.wait_until(
			this.to_full_path(path),
			criterion,
			opts,
		);
	}
	watch_unchecked(path, handler, opts) {
		return this.backing_store.watch_unchecked(
			this.to_full_path(path),
			handler,
			opts,
		);
	}
	async watch(path, handler, opts) {
		return await this.backing_store.watch(
			this.to_full_path(path),
			handler,
			opts,
		);
	}
}
export async function deref(backing_store, path, _opts) {
	backing_store instanceof VSocket &&
		(await backing_store.subscription_flush());
	const target = await backing_store.read(path);
	if (!target) return null;
	const maybe_view_object = backing_store.root.find(target);
	if (
		void 0 === maybe_view_object ||
		maybe_view_object instanceof SubtreeArray ||
		maybe_view_object instanceof SubtreeTable
	)
		throw new Error(
			`Unable to dereference ${path.kwl}.${path.kw} -- doesn't point at a referenceable object`,
		);
	return maybe_view_object;
}
export class Root {
	backing_store;
	constructor(backing_store) {
		this.backing_store = backing_store;
	}
	get components() {
		return this.backing_store.schema.keywords;
	}
	find(kwl) {
		const i = kwl.indexOf("."),
			hd = -1 === i ? kwl : kwl.substring(0, i);
		for (const component of this.components)
			if (component.contents.sys_name === hd) {
				if (-1 === i) return new Subtree(this.backing_store, hd, component);
				{
					const rest = kwl.substring(i + 1),
						maybe_desc = V2.find_subtree_within(component.contents, rest);
					if (void 0 === maybe_desc) return;
					return to_view_object(this.backing_store, kwl, maybe_desc);
				}
			}
	}
}
export class SubtreeTable {
	backing_store;
	kwl;
	description;
	constructor(backing_store, kwl, description) {
		(this.backing_store = backing_store),
			(this.kwl = kwl),
			(this.description = description);
	}
	get parent() {
		return new Subtree(
			this.backing_store,
			path_hd(this.kwl),
			this.description.parent,
		);
	}
	async allocated_indices(opts) {
		return await this.backing_store.table_indices({
			...opts,
			table_kwl: this.kwl,
		});
	}
	async is_allocated(index, opts) {
		return await this.backing_store.table_has_row({
			...(opts ?? {}),
			index: index,
			table_kwl: this.kwl,
		});
	}
	async rows(opts) {
		return (await this.allocated_indices(opts)).map((i) =>
			this.row_unchecked(i),
		);
	}
	row(index) {
		return this.row_unchecked(index);
	}
	row_unchecked(index) {
		return new SubtreeTableRow(
			this.backing_store,
			`${this.kwl}[${index}]`,
			this.description.contents,
		);
	}
	async *[Symbol.asyncIterator]() {
		const indices = await this.allocated_indices(),
			self = this;
		return (function* () {
			for (const i of indices) yield self.row_unchecked(i);
		})();
	}
}
export class SubtreeNamedTable extends SubtreeTable {
	constructor(backing_store, kwl, description) {
		super(backing_store, kwl, description);
	}
	async create_row(opts) {
		const index = (
			await this.backing_store.table_create_row({
				...opts,
				table_kwl: this.kwl,
			})
		)[0];
		return this.row_unchecked(index);
	}
	async rows(opts) {
		return (await this.allocated_indices(opts)).map((i) =>
			this.row_unchecked(i),
		);
	}
	row(index, _) {
		return this.row_unchecked(index);
	}
	row_unchecked(index) {
		return new SubtreeNamedTableRow(
			this.backing_store,
			`${this.kwl}[${index}]`,
			this.description.contents,
		);
	}
	async delete_all(opts) {
		await this.backing_store.table_delete_all_rows({
			table_kwl: this.kwl,
			...(opts ?? {}),
		});
	}
}
export class SubtreeTableRow extends Subtree {
	constructor(backing_store, kwl, description) {
		super(backing_store, kwl, description), path_index(this.kwl);
	}
	get index() {
		return path_index(this.kwl);
	}
}
export class SubtreeNamedTableRow extends SubtreeTableRow {
	constructor(backing_store, kwl, description) {
		super(backing_store, kwl, description);
	}
	async rename(desired_name, opts) {
		await this.backing_store.table_rename_row({
			...opts,
			row_kwl: this.kwl,
			desired_name: desired_name,
		});
	}
	async row_name(opts) {
		return await this.read({ kw: "row_name_status" }, opts);
	}
	async delete(opts) {
		await this.backing_store.table_delete_row({
			...opts,
			table_kwl: path_strip_trailing_index(this.kwl),
			index: this.index,
		});
	}
}
export var ComponentState;
!(function (ComponentState) {
	(ComponentState[(ComponentState.Unknown = 0)] = "Unknown"),
		(ComponentState[(ComponentState.Disabled = 1)] = "Disabled"),
		(ComponentState[(ComponentState.Uninitialized = 2)] = "Uninitialized"),
		(ComponentState[(ComponentState.Running = 3)] = "Running"),
		(ComponentState[(ComponentState.Crashed = 4)] = "Crashed");
})(ComponentState || (ComponentState = {}));
export class ModuleRegistry {
	static DISABLED = -2;
	data = {};
	sys_names = {};
	change_handlers = {};
	constructor(components, data) {
		for (const component of components)
			this.sys_names[component.ua_name] = component.sys_name;
		this.data = data;
	}
	on_change(f) {
		const id = StringIDPool.generate_id();
		return (
			(this.change_handlers[id] = f),
			new EventHandler(3, id, () => {
				delete this.change_handlers[id];
			})
		);
	}
	static async initialize(components, socket, degraded_mode, context) {
		const data = {};
		for (const component of components)
			data[component.sys_name] = {
				crashed: !1,
				pid: -1,
				owning_module: component.owning_module,
				ua_name: component.ua_name,
			};
		const mr = new ModuleRegistry(components, data),
			mr_kwl = "arkona_internal_module_registry",
			derive_state = (pid, crashed) =>
				pid === ModuleRegistry.DISABLED
					? ComponentState.Disabled
					: pid < 0
						? crashed
							? ComponentState.Crashed
							: ComponentState.Uninitialized
						: crashed
							? ComponentState.Crashed
							: ComponentState.Running;
		for (const component of components) {
			const pid_kw = `${component.sys_name}_pid`;
			socket.register_kw_listener(
				{
					listener_id: `ModuleRegistry:${pid_kw}`,
					kwl: mr_kwl,
					kw: pid_kw,
					listener_type: 0,
					execution_strategy: 0,
				},
				(pid) => {
					data[component.sys_name].pid = pid;
					for (const id in mr.change_handlers)
						mr.change_handlers[id](
							component.sys_name,
							derive_state(
								data[component.sys_name].pid,
								data[component.sys_name].crashed,
							),
						);
				},
			);
			const crashed_kw = `${component.sys_name}_crashed`;
			socket.register_kw_listener(
				{
					listener_id: `ModuleRegistry:${crashed_kw}`,
					kwl: mr_kwl,
					kw: crashed_kw,
					listener_type: 0,
					execution_strategy: 0,
				},
				(crashed) => {
					data[component.sys_name].crashed = crashed;
					for (const id in mr.change_handlers)
						mr.change_handlers[id](
							component.sys_name,
							derive_state(
								data[component.sys_name].pid,
								data[component.sys_name].crashed,
							),
						);
				},
			);
		}
		return (
			await new Promise((resolve, reject) => {
				let rejected = !1;
				context.register_timeout_handler(() => {
					(rejected = !0), reject();
				});
				const event_handler = mr.on_change(() => {
					const missing_modules = new Set();
					for (const component_sys_name in data) {
						if ("software_update" === component_sys_name) continue;
						const state = derive_state(
							data[component_sys_name].pid,
							data[component_sys_name].crashed,
						);
						if (
							state !== ComponentState.Running &&
							state !== ComponentState.Disabled &&
							(state !== ComponentState.Crashed || !degraded_mode)
						) {
							missing_modules.add(data[component_sys_name].owning_module),
								(context.desc = `while waiting for ${socket.identify()} to initialize the following modules: ${[...missing_modules].join(", ")}`);
							break;
						}
					}
					(0 === missing_modules.size || (degraded_mode && !rejected)) &&
						(event_handler.unregister(),
						context.unregister_timeout_handler(reject),
						resolve());
				});
				try {
					socket.send([{ op: "readAll", kwl: mr_kwl }]);
				} catch (e) {
					(rejected = !0), reject(e);
				}
			}),
			mr
		);
	}
	is_online(component_sys_name) {
		return (
			this.data[component_sys_name].pid > 0 &&
			!1 === this.data[component_sys_name].crashed
		);
	}
	is_disabled(component_sys_name) {
		return this.data[component_sys_name].pid === ModuleRegistry.DISABLED;
	}
	get_status(component_sys_name) {
		return {
			pid: this.data[component_sys_name].pid,
			crashed: this.data[component_sys_name].crashed,
		};
	}
}
export class RuntimeConstants {
	data = {};
	constructor(data) {
		this.data = data;
	}
	get_constant(component_sys_name, constant_name) {
		if (
			!Object.prototype.hasOwnProperty.call(this.data, component_sys_name) ||
			!Object.prototype.hasOwnProperty.call(
				this.data[component_sys_name],
				constant_name,
			)
		)
			throw new Error(
				`Unknown constant specifier {${component_sys_name}, ${constant_name}}`,
			);
		return this.data[component_sys_name][constant_name];
	}
	equal(other) {
		const components_a = Object.keys(this.data).sort(),
			components_b = Object.keys(other.data).sort();
		if (components_a.length !== components_b.length) return !1;
		for (let i = 0; i < components_a.length; ++i) {
			if (components_a[i] !== components_b[i]) return !1;
			const component = components_a[i],
				const_names_a = Object.keys(this.data[component]).sort(),
				const_names_b = Object.keys(other.data[component]).sort();
			if (const_names_a.length !== const_names_b.length) return !1;
			for (let j = 0; j < const_names_a.length; ++j) {
				if (const_names_a[j] !== const_names_b[j]) return !1;
				const const_name = const_names_a[j];
				if (
					this.data[component][const_name] !== other.data[component][const_name]
				)
					return !1;
			}
		}
		return !0;
	}
	static async initialize(schema, socket, timeout, context) {
		context.desc = "while waiting for runtime constants to initialize";
		const disabledComponentSysNames = new Set(),
			split = (orig_constant_specifier) => {
				const parts = orig_constant_specifier.split("::"),
					sysname_parts = schema.constants[parts[0]][parts[1]].path.split(".");
				return {
					component_sys_name: sysname_parts[0],
					constant_name: sysname_parts[1],
				};
			};
		if (!0 === socket.protocol_features.runtime_constants_json) {
			const raw_data = await download_json({
				protocol: "ws" === socket.protocol ? "http" : "https",
				reject_unauthorized: !1,
				path: "/runtime_constants.json",
				ip: socket.ip,
				timeout: timeout,
			});
			for (const component of schema.keywords)
				for (const constant of component.bound_to ?? []) {
					const {
						component_sys_name: component_sys_name,
						constant_name: constant_name,
					} = split(constant);
					if (!raw_data[component_sys_name][constant_name]) {
						disabledComponentSysNames.add(component.sys_name);
						break;
					}
				}
			return [new RuntimeConstants(raw_data), disabledComponentSysNames];
		}
		const data = {},
			todo = {},
			find_missing_constants = (component_sys_name) => {
				for (const c of schema.keywords)
					if (c.sys_name === component_sys_name)
						return (c.bound_to ?? []).map(split);
				throw new Error(`Unknown component ${component_sys_name}`);
			};
		for (const component_ua_name in schema.constants) {
			if (
				void 0 ===
				schema.keywords.find((comp) => comp.ua_name === component_ua_name)
			)
				continue;
			const component_sys_name = (() => {
				for (const c in schema.constants[component_ua_name])
					return schema.constants[component_ua_name][c].path.split(".")[0];
				unreachable();
			})();
			data[component_sys_name] = {};
			const constants = Object.keys(schema.constants[component_ua_name]);
			todo[component_sys_name] = {
				missing_constants: find_missing_constants(component_sys_name),
				own_constants: constants,
			};
		}
		for (; Object.keys(todo).length > 0; ) {
			const work_queue = [],
				old_keys = Object.keys(todo);
			for (const component_sys_name of old_keys) {
				let verdict = "read";
				for (const mc of todo[component_sys_name].missing_constants) {
					const cval = data?.[mc.component_sys_name]?.[mc.constant_name];
					if (void 0 === cval) {
						verdict = "wait";
						break;
					}
					if (!1 === cval) {
						verdict = "discard";
						break;
					}
				}
				switch (verdict) {
					case "wait":
						break;
					case "discard":
						disabledComponentSysNames.add(component_sys_name),
							delete todo[component_sys_name];
						break;
					case "read": {
						const kwl = `${component_sys_name}.runtime_constants`;
						for (const constant_name of todo[component_sys_name].own_constants)
							work_queue.push(
								socket
									.read_unchecked({ kwl: kwl, kw: constant_name }, timeout)
									.then((val) => {
										data[component_sys_name][constant_name] = val;
									}),
							);
						delete todo[component_sys_name];
					}
				}
			}
			await Promise.all(work_queue);
		}
		return [new RuntimeConstants(data), disabledComponentSysNames];
	}
}
function is_stored_named_table(x) {
	return "named-rows" in x;
}
export class VSettings {
	schema;
	build_info;
	m_identifier;
	m_data;
	listener_registry = new ListenerRegistry((err) =>
		setTimeout(() => {
			throw err;
		}, 0),
	);
	open_handlers = new Map();
	constructor(schema, build_info, json, m_identifier) {
		(this.schema = schema),
			(this.build_info = build_info),
			(this.m_identifier = m_identifier);
		const bail = (msg) => {
				throw new Error(
					`The provided settings object does not conform to the expected file format${msg ? `: ${msg}` : ""}`,
				);
			},
			is_record = (y) => null !== y && "object" == typeof y;
		is_record(json) || bail("should be a JSON dictionary"),
			Object.hasOwnProperty.call(json, "format") ||
				bail("should have a `format` field ");
		"V__matrix settings.json v1" !== json.format &&
			bail(
				`unknown format ${json.format} (expected to find V__matrix settings.json v1)`,
			),
			(Object.hasOwnProperty.call(json, "header") && is_record(json.header)) ||
				bail("should have a dictionary-valued `header` field"),
			(Object.hasOwnProperty.call(json.header, "fpga") &&
				"string" == typeof json.header.fpga) ||
				bail('data["header"] should have a string-valued `fpga` field'),
			(Object.hasOwnProperty.call(json.header, "version") &&
				Array.isArray(json.header.version) &&
				2 === json.header.version.length &&
				"string" == typeof json.header.version[0] &&
				"string" == typeof json.header.version[1]) ||
				bail(
					'data["header"] should have a `version` field containing a two-string array [versionstring, build_date]',
				),
			(Object.hasOwnProperty.call(json.header, "flags") &&
				Array.isArray(json.header.flags) &&
				-1 === json.header.flags.findIndex((x) => "string" != typeof x)) ||
				bail(
					'data["header"] should have a (possibly empty) string array called `flags`',
				),
			Object.hasOwnProperty.call(json, "components") ||
				bail(
					"there should be a `components` dictionary holding the saved state",
				);
		const validate_keyword = (path, kw) => {
				(Object.hasOwnProperty.call(kw, "isDefault") &&
					"boolean" == typeof kw.isDefault) ||
					bail(
						`components.${path} should contain a boolean-valued isDefault field`,
					),
					Object.hasOwnProperty.call(kw, "data") ||
						bail(`components.${path} should contain a field named data`);
			},
			validate_named_table = (kwl_so_far, nt) => {
				const p = (i) => `components.${kwl_so_far}[${i}]`;
				for (let i = 0; i < nt["named-rows"].length; ++i) {
					const row = nt["named-rows"][i];
					"number" != typeof row.idx &&
						bail(`${p(i)} should have a numeric idx field`),
						"string" != typeof row.id &&
							bail(`${p(i)} should have a string-valued id field`),
						validate_subtree(`${kwl_so_far}[${i}]`, row);
				}
			},
			validate_subtree = (kwl_so_far, st) => {
				for (const kw in st.kw ?? {})
					validate_keyword(`${kwl_so_far}.${kw}`, st.kw[kw]);
				for (const kwl in st.kwl ?? {})
					is_stored_named_table(st.kwl[kwl])
						? validate_named_table(`${kwl_so_far}.${kwl}`, st.kwl[kwl])
						: validate_subtree(`${kwl_so_far}.${kwl}`, st.kwl[kwl]);
			};
		for (const sys_name in json.components)
			validate_subtree(sys_name, json.components[sys_name]);
		this.m_data = json;
	}
	identify() {
		const flag_suffix =
			0 === this.m_data.header.flags.length
				? ""
				: `?${this.m_data.header.flags.join("&")}`;
		return this.m_identifier
			? `${this.m_identifier} [settings.json${flag_suffix}, ${this.m_data.header.version[0]}/${this.m_data.header.fpga}]`
			: `settings.json${flag_suffix}, ${this.m_data.header.version[0]}/${this.m_data.header.fpga}`;
	}
	get root() {
		return new Root(this);
	}
	is_ready() {
		return !0;
	}
	marker() {
		return new Promise((resolve) => resolve());
	}
	find_subtree(kwl) {
		const segments = kwl.split(".");
		let cur_level = this.m_data.components[segments[0]];
		if (void 0 !== cur_level) {
			for (let i = 1; i < segments.length; ++i) {
				const segment = segments[i];
				if (!Object.hasOwnProperty.call(cur_level.kwl, segment)) {
					if (
						segment.endsWith("]") &&
						Object.hasOwnProperty.call(
							cur_level.kwl,
							path_strip_trailing_index(segment),
						)
					) {
						const nt = cur_level.kwl[path_strip_trailing_index(segment)],
							idx = path_index(segment),
							maybe_row = nt["named-rows"].find((row) => row.idx === idx);
						if (void 0 !== maybe_row) {
							cur_level = maybe_row;
							continue;
						}
					}
					return;
				}
				cur_level = cur_level.kwl[segment];
			}
			return cur_level;
		}
	}
	find_keyword(path) {
		const st = this.find_subtree(path.kwl);
		if (
			void 0 !== st &&
			!is_stored_named_table(st) &&
			Object.hasOwnProperty.call(st.kw, path.kw)
		)
			return st.kw[path.kw];
	}
	query_cache(path) {
		const maybe_stored_kw = this.find_keyword(path);
		return void 0 === maybe_stored_kw ? MissingData : [0, maybe_stored_kw.data];
	}
	query_cache_kwl(kwl) {
		const st = this.find_subtree(kwl);
		if (void 0 === st) return MissingData;
		const result = new Map();
		if (!is_stored_named_table(st))
			for (const kw in st.kw) result.set(kw, [0, st.kw[kw].data]);
		return [0, result];
	}
	get_runtime_constant(constant_string) {
		const parts = constant_string.split("::"),
			kwl = `${snake_case(parts[0])}.runtime_constants`,
			kw = this.find_keyword({ kwl: kwl, kw: parts[1] });
		if (void 0 === kw)
			throw new Error(`Unknown runtime constant '${constant_string}'`);
		return kw.data;
	}
	register_kw_listener(pars, handler) {
		return this.listener_registry.register_kw_listener(pars, handler);
	}
	register_kwl_listener(pars, handler) {
		return this.listener_registry.register_kwl_listener(pars, handler);
	}
	register_global_listener(handler, pars) {
		return this.listener_registry.register_global_listener(handler, pars);
	}
	on_open(f, suggested_id) {
		const id = StringIDPool.generate_or_use_id(
			this.open_handlers,
			suggested_id,
		);
		return (
			this.open_handlers.set(id, f),
			new EventHandler(0, id, () => {
				this.open_handlers.delete(id);
			})
		);
	}
	write_unchecked(path, payload) {
		this.send([{ op: "data", kwl: path.kwl, kw: { [path.kw]: payload } }]);
	}
	async write(path, payload, _) {
		this.write_unchecked(path, payload);
	}
	send(requests) {
		for (const request of requests)
			switch (request.op) {
				case "data":
					for (const kw in request.kw) {
						const x = this.find_keyword({ kwl: request.kwl, kw: kw });
						if (void 0 === x)
							throw new Error(
								`This settings file contains no keyword named ${request.kwl}.${kw}`,
							);
						x.data = request.kw[kw];
					}
					break;
				case "flags":
				case "marker":
				case "readAll":
				case "subscribe":
				case "unsubscribe":
					break;
				case "tree":
					throw new Error("TODO");
			}
	}
	async read_unchecked(path) {
		const maybe_data = this.query_cache(path);
		if (maybe_data === MissingData)
			throw new Error(
				`This settings file contains no data for ${path.kwl}.${path.kw}`,
			);
		return maybe_data[1];
	}
	async read(full_path, opts) {
		return await this.read_unchecked(full_path);
	}
	async table_create_row(opts) {
		throw new Error("TODO");
	}
	async table_rename_row(opts) {
		throw new Error("TODO");
	}
	async table_indices(opts) {
		throw new Error("TODO");
	}
	async table_has_row(opts) {
		throw new Error("TODO");
	}
	async table_delete_row(opts) {
		throw new Error("TODO");
	}
	async table_delete_all_rows(opts) {
		throw new Error("TODO");
	}
	async wait_until(full_path, criterion, opts) {
		throw new Error("TODO");
	}
	watch_unchecked(full_path, handler, opts) {
		const listener = this.register_kw_listener(
			{
				...full_path,
				execution_strategy: 0,
				listener_type: 0,
				ensure_initial_read: opts?.ensure_initial_read ?? !1,
			},
			handler,
		);
		return new Watcher(full_path, () => {
			listener.unregister();
		});
	}
	async watch(full_path, handler, opts) {
		return this.watch_unchecked(full_path, handler, opts);
	}
}
