import { StringIDPool } from "./id_pool.js";
import { GlobalListener, KeywordListener, KWLListener } from "./pervasives.js";
import { enforce_nonnull } from "./utilities.js";
export class ListenerRegistry {
	m_error_handler;
	global_listeners = new Map();
	kwl_listeners = new Map();
	lazy_handler_interval = null;
	pending_lazy_updates = { kwls: new Map(), kws: new Map() };
	constructor(m_error_handler) {
		(this.m_error_handler = m_error_handler),
			(this.lazy_handler_interval = setInterval(() => {
				this.execute_lazy_handlers();
			}, 98));
	}
	dump_stats() {
		let n_complete = 0,
			n_partial = 0;
		const multi = new Set();
		for (const [kwl, { complete: complete, partial: partial }] of this
			.kwl_listeners) {
			n_complete += Object.keys(complete).length;
			for (const kw_name in partial) {
				const n = Object.keys(partial[kw_name]).length;
				(n_partial += n),
					n > 1 && multi.size < 20 && multi.add(`${kwl}/${kw_name}`);
			}
		}
	}
	close() {
		clearInterval(this.lazy_handler_interval);
	}
	prune_listeners(keep_if) {
		this.execute_lazy_handlers();
		for (const [listener_id, _] of this.global_listeners)
			keep_if({ listener_id: listener_id }) ||
				this.global_listeners.delete(listener_id);
		for (const [kwl, { complete: complete, partial: partial }] of this
			.kwl_listeners) {
			for (const id in complete)
				keep_if({ kwl: kwl, listener_id: id }) || delete complete[id];
			for (const kw_name in partial) {
				for (const id in partial[kw_name])
					keep_if({ kwl: kwl, listener_id: id, kw: kw_name }) ||
						delete partial[kw_name][id];
				0 === Object.keys(partial[kw_name]).length && delete partial[kw_name];
			}
			0 === Object.keys(complete).length &&
				0 === Object.keys(partial).length &&
				this.kwl_listeners.delete(kwl);
		}
	}
	execute_lazy_handlers() {
		for (const [listener_id, handler] of this.pending_lazy_updates.kwls)
			try {
				handler();
			} catch (e) {
				this.m_error_handler(e);
			}
		for (const [listener_id, entry] of this.pending_lazy_updates.kws)
			try {
				entry[0](entry[1]);
			} catch (e) {
				this.m_error_handler(e);
			}
		this.pending_lazy_updates.kwls.clear(),
			this.pending_lazy_updates.kws.clear();
	}
	safely_delete_handler(pars) {
		pars.kwl.includes("live_view") &&
			(0 !== this.pending_lazy_updates.kwls.size ||
				this.pending_lazy_updates.kws.size);
		const entry = this.kwl_listeners.get(pars.kwl);
		entry &&
			(void 0 !== pars.kw
				? Object.prototype.hasOwnProperty.call(entry.partial, pars.kw) &&
					(delete entry.partial[pars.kw][pars.id],
					0 === Object.keys(entry.partial[pars.kw]).length &&
						delete entry.partial[pars.kw])
				: delete entry.complete[pars.id],
			0 === Object.keys(entry.complete).length &&
				0 === Object.keys(entry.partial).length &&
				this.kwl_listeners.delete(pars.kwl),
			this.pending_lazy_updates.kwls.has(pars.id) &&
				this.execute_lazy_handlers());
	}
	do_get_listeners(kwl) {
		return (
			this.kwl_listeners.has(kwl) ||
				this.kwl_listeners.set(kwl, {
					complete: Object.create(null),
					partial: Object.create(null),
				}),
			this.kwl_listeners.get(kwl)
		);
	}
	have_permanent_listener(pars) {
		const entry = this.kwl_listeners.get(pars.kwl);
		if (!entry) return !1;
		if (void 0 === pars.kw) {
			for (const id in entry.complete)
				if (entry.complete[id].permanent) return !0;
		} else
			for (const id in entry.partial[pars.kw])
				if (entry.partial[pars.kw][id].permanent) return !0;
		return !1;
	}
	register_kw_listener(pars, handler) {
		const partial_listeners = this.do_get_listeners(pars.kwl).partial;
		Object.prototype.hasOwnProperty.call(partial_listeners, pars.kw) ||
			(partial_listeners[pars.kw] = Object.create(null));
		const listeners = partial_listeners[pars.kw],
			id = StringIDPool.generate_or_use_id(listeners, pars.listener_id);
		return (
			(listeners[id] = {
				handler:
					0 === pars.listener_type
						? handler
						: (payload) => {
								handler(payload),
									this.safely_delete_handler({ ...pars, id: id });
							},
				execution_strategy: pars.execution_strategy,
				permanent: 0 === pars.listener_type,
			}),
			new KeywordListener(pars.kwl, pars.kw, id, new WeakRef(this))
		);
	}
	register_kwl_listener(pars, handler) {
		const listeners = this.do_get_listeners(pars.kwl).complete,
			id = StringIDPool.generate_or_use_id(listeners, pars.listener_id);
		return (
			(listeners[id] = {
				handler:
					0 === pars.listener_type
						? handler
						: () => {
								handler(), this.safely_delete_handler({ ...pars, id: id });
							},
				execution_strategy: pars.execution_strategy,
				permanent: 0 === pars.listener_type,
			}),
			new KWLListener(pars.kwl, id, new WeakRef(this))
		);
	}
	register_global_listener(handler, pars) {
		const id = StringIDPool.generate_or_use_id(
			this.global_listeners,
			pars?.listener_id,
		);
		this.global_listeners.set(id, handler);
		const ref_to_self = new WeakRef(this);
		return new GlobalListener(id, (self) => {
			ref_to_self.deref()?.global_listeners.delete(self.id);
		});
	}
}
