import { digest } from "./polyfills/polyfill_digest.js";
import { Subtree, SubtreeNamedTable } from "./data_views.js";
import {
	children_of_raw_atomic_subtree,
	children_of_raw_subtree,
	kernel_of_child,
} from "./schema_v2.js";
import { Duration } from "./time.js";
import {
	asyncIter,
	enforce,
	enforce_eq,
	enforce_nonnull,
	path_hd,
	unreachable,
	WorkQueue,
} from "./utilities.js";
export async function subtree_iter(pars) {
	const wq = pars.work_queue ?? new WorkQueue({ num_workers: 128 }),
		worker = async (kwl, desc) => {
			if (
				"recurse" ===
				(pars.on_subtree
					? await pars.on_subtree(new Subtree(pars.backing_store, kwl, desc))
					: "recurse")
			)
				for (const child of desc.children) {
					const kernel = kernel_of_child(child);
					if ("subtree" === kernel.data_type)
						switch (child.container_type) {
							case 4:
								unreachable();
							case 0:
								await worker(`${kwl}.${kernel.sys_name}`, kernel);
								break;
							case 2:
								enforce_eq(child.contents.container_type, 0);
								for (let i = 0; i < child.capacity; ++i)
									wq.push(async () => {
										await worker(`${kwl}.${kernel.sys_name}[${i}]`, kernel);
									});
								break;
							case 1:
								enforce_eq(child.contents.container_type, 0);
								{
									const table_kwl = `${kwl}.${kernel.sys_name}`;
									if (
										"do-not-recurse" ===
										(pars.on_named_table && child.named_tables
											? await pars.on_named_table(
													new SubtreeNamedTable(
														pars.backing_store,
														table_kwl,
														child,
													),
												)
											: "recurse")
									)
										break;
									if ("allocated" === (pars.include ?? "allocated")) {
										const allocated_indices =
											await pars.backing_store.table_indices({
												table_kwl: table_kwl,
												check_component_liveness: pars.check_component_liveness,
											});
										for (const i of allocated_indices)
											wq.push(async () => {
												await worker(`${table_kwl}[${i}]`, kernel);
											});
									} else
										for (let i = 0; i < child.capacity; ++i)
											wq.push(async () => {
												await worker(`${table_kwl}[${i}]`, kernel);
											});
								}
						}
				}
		};
	for (const component of pars.backing_store.schema.keywords)
		(component.enabled || !0 === pars.include_disabled) &&
			wq.push(async () => {
				await worker(component.contents.sys_name, component.contents);
			});
	await wq.drain();
}
export function subtree_iter_sync(pars) {
	const worker = (kwl, desc) => {
		if ("recurse" === pars.handler(desc, kwl))
			for (const child of desc.children) {
				const kernel = kernel_of_child(child);
				if ("subtree" === kernel.data_type)
					switch (child.container_type) {
						case 4:
							unreachable();
						case 0:
							worker(`${kwl}.${kernel.sys_name}`, kernel);
							break;
						case 2:
							enforce_eq(child.contents.container_type, 0);
							for (let i = 0; i < child.capacity; ++i)
								worker(`${kwl}.${kernel.sys_name}[${i}]`, kernel);
							break;
						case 1:
							enforce_eq(child.contents.container_type, 0);
							const table_kwl = `${kwl}.${kernel.sys_name}`;
							for (let i = 0; i < child.capacity; ++i)
								worker(`${table_kwl}[${i}]`, kernel);
					}
			}
	};
	for (const component of pars.backing_store.schema.keywords)
		(component.enabled || !0 === pars.include_disabled) &&
			worker(component.contents.sys_name, component.contents);
}
export async function subtree_iter_typewise(backing_store, pars) {
	await subtree_iter({
		...pars,
		backing_store: backing_store,
		include: pars.include ?? "allocated",
		on_subtree: async (st) => (
			st.description.type_identifier === pars.type_identifier &&
				(await pars.handler(pars.lift(st), st.kwl)),
			"recurse"
		),
	});
}
export async function keyword_iter(pars) {
	const include_rowname_keywords = pars.include_rowname_keywords ?? !1,
		wq = pars.work_queue ?? new WorkQueue({ num_workers: 32 });
	await subtree_iter({
		...pars,
		include: pars.include ?? "allocated",
		on_subtree: async (st) => {
			for (const child of st.children)
				4 === child.container_type &&
					(await pars.handler(pars.backing_store, child, st.kwl));
			if (include_rowname_keywords) {
				const maybe_rnd = maybe_rowname_desc(
					5 === st.description.container_type
						? st.description.contents
						: st.description,
				);
				maybe_rnd &&
					wq.push(() =>
						pars.handler(pars.backing_store, maybe_rnd, path_hd(st.kwl)),
					);
			}
			return "recurse";
		},
	}),
		await wq.drain();
}
function maybe_rowname_desc(st) {
	if (1 === st.parent.container_type && st.parent.named_tables) {
		const kw_desc = {
			container_type: 4,
			kw_type: "duplex",
			parent: st,
			persistent: !0,
			contents: {
				container_type: 0,
				data_type: "string",
				max_length: 32,
				optional: !1,
				sys_name: "row_name",
				parent: void 0,
			},
		};
		return (kw_desc.contents.parent = kw_desc), kw_desc;
	}
	return null;
}
export function keyword_iter_sync(pars) {
	const include_rowname_keywords = pars.include_rowname_keywords ?? !1;
	subtree_iter_sync({
		...pars,
		include: pars.include ?? "everything",
		handler: (x, kwl) =>
			((st, kwl) => {
				for (const child of st.children)
					4 === child.container_type &&
						pars.handler(pars.backing_store, child, kwl);
				if (include_rowname_keywords) {
					const maybe_rnd = maybe_rowname_desc(st);
					maybe_rnd &&
						pars.handler(pars.backing_store, maybe_rnd, path_hd(kwl));
				}
				return "recurse";
			})(5 === x.container_type ? x.contents : x, kwl),
	});
}
export async function branch_iter(pars) {
	const branchWorker = async (branch, desc) => {
		if ("recurse" === (await pars.handler(branch, desc))) {
			const kernel = kernel_of_child(desc);
			"subtree" !== kernel.data_type && unreachable();
			for (const child of kernel.children) {
				const kernel = kernel_of_child(child);
				"subtree" === kernel.data_type &&
					(await branchWorker(`${branch}.${kernel.sys_name}`, child));
			}
		}
	};
	for (const component of pars.backing_store.schema.keywords)
		(component.enabled || !0 === pars.include_disabled) &&
			(await branchWorker(component.contents.sys_name, component.contents));
}
export function branch_iter_sync(pars) {
	const branchWorker = (branch, desc) => {
		if ("recurse" === pars.handler(branch, desc)) {
			const kernel = kernel_of_child(desc);
			"subtree" !== kernel.data_type && unreachable();
			for (const child of kernel.children) {
				const kernel = kernel_of_child(child);
				"subtree" === kernel.data_type &&
					branchWorker(`${branch}.${kernel.sys_name}`, child);
			}
		}
	};
	for (const component of pars.backing_store.schema.keywords)
		(component.enabled || !0 === pars.include_disabled) &&
			branchWorker(component.contents.sys_name, component.contents);
}
function do_reset_subtree_to_defaults(
	backing_store,
	kwl,
	st,
	wq,
	named_tables,
	pars,
) {
	const recursive = pars?.recursive ?? !0,
		get_default = (desc) => {
			if (desc.optional) return null;
			switch (desc.container_type) {
				case 3:
					return [];
				case 0:
					switch (desc.data_type) {
						case "device_tree_node":
							throw new Error("dtnode keywords are read-only");
						case "atomic subtree":
							return desc.children.map((child) => get_default(child));
						case "alerts":
							return 0;
						default:
							return desc.default_value ?? null;
					}
				case 2: {
					const per_element = get_default(desc.contents);
					return Array.from({ length: desc.capacity }, () => per_element);
				}
			}
		},
		unroll = (kwl, cst) => {
			switch (cst.container_type) {
				case 0:
					do_reset_subtree_to_defaults(
						backing_store,
						kwl,
						cst,
						wq,
						named_tables,
						pars,
					);
					break;
				case 2:
					for (let i = 0; i < cst.capacity; ++i)
						unroll(`${kwl}[${i}]`, cst.contents);
					break;
				case 1:
					cst.named_tables && cst.capacity > 0 && named_tables.push(kwl),
						wq.push(async () => {
							const indices = await backing_store.table_indices({
								table_kwl: kwl,
							});
							for (const i of indices) unroll(`${kwl}[${i}]`, cst.contents);
						});
			}
		};
	for (const child of st.children) {
		const kernel = kernel_of_child(child);
		if (4 === child.container_type) {
			let maybe_kw = null;
			switch (child.kw_type) {
				case "command":
					maybe_kw = kernel.sys_name;
					break;
				case "duplex":
					maybe_kw = kernel.sys_name + "_command";
			}
			if (null !== maybe_kw) {
				const kw = maybe_kw,
					payload = get_default(child.contents);
				wq.push(async () => {
					await backing_store
						.write({ kwl: kwl, kw: kw }, payload, {
							timeout: new Duration(3, "s"),
							retry_interval: new Duration(50, "ms"),
						})
						.catch((_) => {});
				});
			}
		} else recursive && unroll(`${kwl}.${kernel.sys_name}`, child);
		recursive &&
			"subtree" === kernel.data_type &&
			unroll(`${kwl}.${kernel.sys_name}`, child);
	}
}
export async function reset_subtree_to_defaults(backing_store, kwl, st, pars) {
	const recursive = pars?.recursive ?? !0,
		wq = pars?.wq ?? new WorkQueue({ num_workers: 32 }),
		named_tables = [];
	do_reset_subtree_to_defaults(backing_store, kwl, st, wq, named_tables, {
		recursive: recursive,
	}),
		(pars?.drain_wq ?? 1) && (await wq.drain()),
		await asyncIter(named_tables, async (kwl) => {
			await backing_store.write(
				{ kwl: kwl, kw: "table_cmd" },
				"DELETE_ALL_ROWS",
				{
					check_component_liveness: !1,
					check_keyword_type: !1,
					retry_until: {
						criterion: "custom",
						validator: async () =>
							0 ===
							(await backing_store.table_indices({ table_kwl: kwl })).length,
					},
				},
			);
		});
}
function variants_of_raw_variant(schema, desc) {
	if (void 0 !== desc.variants) return desc.variants;
	return schema.typedefs[enforce_nonnull(desc.type_identifier)].variants;
}
export async function compute_persistence_hash(schema) {
	let s = "";
	const add = (property_name, value) => {
		(s += property_name),
			(s += "string" == typeof value ? value : JSON.stringify(value));
	};
	for (const {
		enclosing_subtree_properties: enclosing_subtree_properties,
		keyword_properties: keyword_properties,
	} of (function* (schema) {
		function* add_subtree(
			branch_name,
			bound_to,
			upstream_multiplicities,
			desc,
		) {
			const enclosing_subtree_multiplicities =
				"array_size" in desc
					? [...upstream_multiplicities, { array_size: desc.array_size }]
					: "table_size" in desc
						? [
								...upstream_multiplicities,
								{ table_size: desc.table_size, named_rows: desc.named_rows },
							]
						: upstream_multiplicities;
			for (const child of children_of_raw_subtree(schema, desc))
				if ("subtree" === child.data_type)
					for (const x of add_subtree(
						branch_name + "." + desc.sys_name,
						bound_to,
						enclosing_subtree_multiplicities,
						child,
					))
						yield x;
				else {
					const keyword_properties = child.persistent
							? { kw_type: child.kw_type, persistent: !0, atomic: child }
							: {
									kw_type: child.kw_type,
									persistent: !1,
									sys_name: child.sys_name,
								},
						enclosing_subtree_properties = {
							bound_to: bound_to,
							branch_name: branch_name,
							total_multiplicity: enclosing_subtree_multiplicities,
						};
					yield {
						enclosing_subtree_properties: enclosing_subtree_properties,
						keyword_properties: keyword_properties,
					};
				}
		}
		for (const component of schema.keywords)
			for (const x of add_subtree(
				component.sys_name,
				component.bound_to ?? [],
				[],
				component,
			))
				yield x;
	})(schema))
		if (
			(add("bound_to", enclosing_subtree_properties.bound_to),
			add(
				"enclosing_subtree_multiplicities",
				enclosing_subtree_properties.total_multiplicity,
			),
			add("kw_type", keyword_properties.kw_type),
			add("persistent", keyword_properties.persistent),
			keyword_properties.persistent)
		) {
			const field_worker = (field) => {
				switch (
					(add("atomic_multiplicity", field.atomic_multiplicity),
					add("data_type", field.data_type),
					add("sys_name", field.sys_name),
					field.data_type)
				) {
					case "alerts":
						add("alerts", field.alerts);
						break;
					case "enum":
						add("enums", field.enum_values);
						break;
					case "float duration":
					case "float":
					case "int duration":
					case "int":
					case "timestamp":
						add("min", field.min), add("max", field.max);
						break;
					case "string":
						add("maxlength", field.max_length);
						break;
					case "ref":
						add("ref_properties", {
							ref_perm: field.ref_perm,
							target_type: field.target_type,
							target_type_identifier: field.target_type_identifier,
						});
						break;
					case "atomic subtree":
						for (const child of children_of_raw_atomic_subtree(schema, field))
							field_worker(child);
						break;
					case "variant":
						for (const [constructor_name, variant] of variants_of_raw_variant(
							schema,
							field,
						))
							add("constructor_name", constructor_name), field_worker(variant);
						break;
					case "bool":
					case "ipaddress":
						break;
					default:
						unreachable();
				}
			};
			field_worker(keyword_properties.atomic);
		} else add("sys_name", keyword_properties.sys_name);
	return await digest(s);
}
function is_record(x) {
	return "object" == typeof x && null !== x;
}
function equal(x, y) {
	const aprops_seen = new Set(),
		worker = (a, b) => {
			if (is_record(a)) {
				if (!is_record(b)) return !1;
				const rec_a = a,
					rec_b = b;
				for (const k in rec_a) {
					if (a === x) {
						if (aprops_seen.has(k)) continue;
						aprops_seen.add(k);
					}
					if (void 0 === rec_b[k]) return !1;
					if (!worker(rec_a[k], rec_b[k])) return !1;
				}
				for (const k in rec_b) if (void 0 === rec_a[k]) return !1;
				return !0;
			}
			if (Array.isArray(a)) {
				if (!Array.isArray(b)) return !1;
				const arr_a = a,
					arr_b = a;
				if (arr_a.length !== arr_b.length) return !1;
				for (let i = 0; i < arr_a.length; ++i)
					if (!worker(arr_a[i], arr_b[i])) return !1;
				return !0;
			}
			return JSON.stringify(a) === JSON.stringify(b);
		};
	return worker(x, y);
}
function diff_recursively(a, b, pars) {
	const result = {};
	for (const key in a)
		if (!pars?.ignored_properties?.has(key) || "children" === pars?.cur_key)
			if (void 0 === b[key]) result[key] = { kind: "removed", was: a[key] };
			else if (
				is_record(a[key]) &&
				is_record(b[key]) &&
				!pars?.dontrecurse?.has(key)
			) {
				const as_record = (x) => {
						if ("children" === key) {
							const result = {};
							for (const child of x) result[child.sys_name] = child;
							return result;
						}
						return x;
					},
					changes = diff_recursively(as_record(a[key]), as_record(b[key]), {
						...pars,
						cur_key: key,
					});
				Object.keys(changes).length > 0 &&
					(result[key] = { kind: "record-changed", changes: changes });
			} else
				equal(a[key], b[key]) ||
					(result[key] = { kind: "scalar-changed", was: a[key], is: b[key] });
	for (const key in b)
		void 0 !== a[key] ||
			pars?.ignored_properties?.has(key) ||
			(result[key] = { kind: "added", is: b[key] });
	return result;
}
export function extract_persistent_parts(schema) {
	const subtree_worker = (st) => {
			const remaining_children = [];
			for (const child of children_of_raw_subtree(schema, st))
				if ("subtree" === child.data_type) {
					const maybe_st = subtree_worker(child);
					maybe_st && remaining_children.push(maybe_st);
				} else child.persistent && remaining_children.push(child);
			return 0 === remaining_children.length
				? null
				: { ...st, children: remaining_children };
		},
		remaining_components = [];
	for (const component of schema.keywords) {
		const maybe_rest = subtree_worker(component);
		maybe_rest && remaining_components.push(maybe_rest);
	}
	const referenced_typedefs = new Set(),
		add_typedef = (td_identifier) => {
			if (!referenced_typedefs.has(td_identifier)) {
				const td = schema.typedefs[td_identifier];
				if (!td) return;
				switch ((referenced_typedefs.add(td_identifier), td.data_type)) {
					case "subtree":
					case "atomic subtree":
						include_typedefs(td);
				}
			}
		},
		include_typedefs = (st) => {
			st.type_identifier && add_typedef(st.type_identifier);
			for (const child of "subtree" === st.data_type
				? children_of_raw_subtree(schema, st)
				: children_of_raw_atomic_subtree(schema, st))
				switch (child.data_type) {
					case "subtree":
					case "atomic subtree":
						include_typedefs(child);
						break;
					default:
						child.type_identifier && add_typedef(child.type_identifier);
				}
		};
	remaining_components.forEach((comp) => include_typedefs(comp));
	const typedefs = {};
	for (const td of referenced_typedefs) typedefs[td] = schema.typedefs[td];
	return { ...schema, typedefs: typedefs, keywords: remaining_components };
}
export function reify_typedefs(schema) {
	const st_worker = (st) => {
		const all_children =
			"subtree" === st.data_type
				? children_of_raw_subtree(schema, st)
				: children_of_raw_atomic_subtree(schema, st);
		return {
			...st,
			children: all_children.map((child) => {
				switch (child.data_type) {
					case "subtree":
					case "atomic subtree":
						return st_worker(child);
					default:
						return child;
				}
			}),
		};
	};
	return { ...schema, keywords: schema.keywords.map(st_worker) };
}
export function schema_diff(a, b, pars) {
	const kw_dict = (schema) => {
		const result = {};
		for (const component of schema.keywords)
			result[component.sys_name] = component;
		return result;
	};
	return {
		constants: diff_recursively(a.constants, b.constants, pars),
		typedefs: diff_recursively(a.typedefs, b.typedefs, pars),
		keywords: diff_recursively(kw_dict(a), kw_dict(b), pars),
	};
}
