import {
	enforce,
	enforce_nonnull,
	path_index,
	path_strip_trailing_index,
	unreachable,
} from "./utilities.js";
export var ContainerType;
!(function (ContainerType) {
	(ContainerType[(ContainerType.None = 0)] = "None"),
		(ContainerType[(ContainerType.Table = 1)] = "Table"),
		(ContainerType[(ContainerType.Array = 2)] = "Array"),
		(ContainerType[(ContainerType.Vector = 3)] = "Vector"),
		(ContainerType[(ContainerType.Keyword = 4)] = "Keyword"),
		(ContainerType[(ContainerType.Component = 5)] = "Component");
})(ContainerType || (ContainerType = {}));
export function children_of_raw_subtree(schema, st) {
	if ("children" in st) return st.children;
	{
		const td = schema.typedefs[st.type_identifier];
		if ("subtree" === td.data_type) return td.children ?? [];
		unreachable();
	}
}
export function children_of_raw_atomic_subtree(schema, st) {
	if ("children" in st) return st.children;
	{
		const td = schema.typedefs[st.type_identifier];
		if ("atomic subtree" === td.data_type) return td.children ?? [];
		unreachable();
	}
}
function annotate_atomic_field(
	child,
	schema,
	units,
	constants_resolver,
	field_parent,
) {
	const clear_atomic_multiplicity = (c) => ({ ...c, atomic_multiplicity: [] }),
		first_kernel = ((kernel_parent) => {
			switch (child.data_type) {
				case "atomic subtree": {
					const raw_children = children_of_raw_atomic_subtree(schema, child),
						result = {
							...clear_atomic_multiplicity(child),
							optional: !1,
							container_type: 0,
							children: [],
							parent: kernel_parent,
						};
					return (
						(result.children = raw_children.map((child) =>
							annotate_atomic_field(
								child,
								schema,
								units,
								constants_resolver,
								result,
							),
						)),
						result
					);
				}
				case "variant": {
					const raw_variants =
							child.variants ??
							schema.typedefs[child.type_identifier].variants ??
							[],
						result = {
							...clear_atomic_multiplicity(child),
							optional: !1,
							container_type: 0,
							variants: [],
							parent: kernel_parent,
						};
					for (const [constructor_name, variant_desc] of raw_variants)
						result.variants.push([
							constructor_name,
							annotate_atomic_field(
								variant_desc,
								schema,
								units,
								constants_resolver,
								result,
							),
						]);
					return result;
				}
				case "int":
				case "float":
					return {
						...clear_atomic_multiplicity(child),
						container_type: 0,
						optional: !1,
						parent: kernel_parent,
						unit:
							void 0 === child.other_properties?.unit
								? null
								: units[child.other_properties.unit],
					};
				default:
					return {
						...clear_atomic_multiplicity(child),
						container_type: 0,
						optional: !1,
						parent: kernel_parent,
					};
			}
		})(field_parent);
	let result = first_kernel;
	const add_container_layer = (container_type, capacity) => {
		const container = {
			container_type: container_type,
			capacity:
				"string" == typeof capacity ? constants_resolver(capacity) : capacity,
			contents: result,
			parent: field_parent,
			optional: !1,
		};
		(result.parent = container), (result = container);
	};
	for (let i = child.atomic_multiplicity.length - 1; i >= 0; --i) {
		const level = child.atomic_multiplicity[i];
		switch (level.type) {
			case "optional":
				if (!1 !== result.optional)
					throw new Error(
						"Invalid atomic multiplicity: multiple consecutive levels of optionality are not supported",
					);
				result.optional = !0;
				break;
			case "array":
				add_container_layer(2, level.capacity);
				break;
			case "vector":
				add_container_layer(3, level.capacity);
		}
	}
	return result;
}
function populate_raw_subtree(
	parent,
	st_raw,
	schema,
	units,
	numeric_constants_resolver,
) {
	for (const raw_child of children_of_raw_subtree(schema, st_raw))
		if ("subtree" !== raw_child.data_type) {
			const temporary_bogus_parent = null;
			if (0 === parent.container_type) {
				if ("virtual" !== raw_child.kw_type) {
					const c = {
						container_type: 4,
						kw_type: raw_child.kw_type,
						persistent: raw_child.persistent,
						contents: annotate_atomic_field(
							raw_child,
							schema,
							units,
							numeric_constants_resolver,
							temporary_bogus_parent,
						),
						parent: parent,
						...(void 0 === raw_child.kw_priority
							? {}
							: { kw_priority: raw_child.kw_priority }),
					};
					(c.contents.parent = c), parent.children.push(c);
				}
			} else unreachable();
		} else {
			const kernel = {
					data_type: "subtree",
					other_properties: raw_child.other_properties,
					sys_name: raw_child.sys_name,
					brief: raw_child.brief,
					desc: raw_child.desc,
					type_identifier: raw_child.type_identifier,
					parent: parent,
					container_type: 0,
					children: [],
				},
				get_capacity = (maybe_constant) =>
					"string" == typeof maybe_constant
						? numeric_constants_resolver(maybe_constant)
						: maybe_constant;
			if (0 !== parent.container_type) unreachable();
			else if ("array_size" in raw_child) {
				const array_container = {
					parent: parent,
					capacity: get_capacity(raw_child.array_size),
					container_type: 2,
					contents: kernel,
				};
				(kernel.parent = array_container),
					parent.children.push(array_container);
			} else if ("table_size" in raw_child) {
				const table_container = {
					parent: parent,
					capacity: get_capacity(raw_child.table_size),
					container_type: 1,
					named_tables: raw_child.named_rows,
					contents: kernel,
				};
				(kernel.parent = table_container),
					parent.children.push(table_container);
			} else parent.children.push(kernel);
			populate_raw_subtree(
				kernel,
				raw_child,
				schema,
				units,
				numeric_constants_resolver,
			);
		}
}
export function annotate(schema, disabledComponents, constants_resolver) {
	const keywords = [],
		numeric_constants_resolver = (constant_name) =>
			constants_resolver(constant_name),
		dummy_parent = {
			container_type: 5,
			bound_to: [],
			enabled: !1,
			owning_module: "",
			ua_name: "",
			contents: {},
		},
		units = Object.fromEntries(
			Object.entries(schema.units ?? {}).map(([name, udef]) => {
				const exceptions_map = new Map();
				for (const [value, s] of udef.exceptions) exceptions_map.set(value, s);
				return [name, { ...udef, exceptions_map: exceptions_map }];
			}),
		);
	for (const raw_component of schema.keywords) {
		if (disabledComponents.has(raw_component.sys_name)) continue;
		const component = {
			container_type: 5,
			bound_to: raw_component.bound_to ?? [],
			enabled:
				-1 ===
				(raw_component.bound_to ?? []).findIndex(
					(constantName) => !1 === constants_resolver(constantName),
				),
			owning_module: raw_component.owning_module,
			ua_name: raw_component.ua_name,
			contents: {
				parent: dummy_parent,
				children: [],
				container_type: 0,
				data_type: "subtree",
				other_properties: raw_component.other_properties,
				sys_name: raw_component.sys_name,
				brief: raw_component.brief,
				desc: raw_component.desc,
				type_identifier: raw_component.type_identifier,
			},
		};
		(component.contents.parent = component),
			populate_raw_subtree(
				component.contents,
				raw_component,
				schema,
				units,
				numeric_constants_resolver,
			),
			keywords.push(component);
	}
	const typedefs = {};
	for (const typedefName of Object.keys(schema.typedefs)) {
		const td = schema.typedefs[typedefName];
		if ("subtree" === td.data_type) {
			const dummy_st = {
				container_type: 0,
				children: [],
				data_type: "subtree",
				sys_name: "",
				other_properties: {},
				parent: null,
			};
			populate_raw_subtree(
				dummy_st,
				{
					data_type: "subtree",
					children: [td],
					sys_name: "",
					other_properties: {},
				},
				schema,
				units,
				numeric_constants_resolver,
			);
			const c = dummy_st.children[0];
			typedefs[typedefName] = { ...c, common: td.common };
		} else
			typedefs[typedefName] = {
				...annotate_atomic_field(
					td,
					schema,
					units,
					numeric_constants_resolver,
					void 0,
				),
				common: td.common,
			};
	}
	return { ...schema, typedefs: typedefs, keywords: keywords, units: units };
}
export function kernel_of_child(c) {
	let result = c;
	for (; "contents" in result; ) result = result.contents;
	return result;
}
export function kernel_of_field(f) {
	let result = f;
	for (; "contents" in result; ) result = result.contents;
	return result;
}
export function find_branch(annotated_schema, branch_name) {
	const segments = branch_name.split(".");
	for (const component of annotated_schema.keywords)
		if (component.contents.sys_name === segments[0]) {
			let cur_st = component.contents;
			if (1 === segments.length) return cur_st;
			for (let i = 1; i < segments.length - 1; ++i)
				for (const child of cur_st.children) {
					const kernel = kernel_of_child(child);
					if (kernel.sys_name === segments[i]) {
						if ("subtree" === kernel.data_type) {
							cur_st = kernel;
							break;
						}
						throw new Error(`Unable to look up ${branch_name}`);
					}
				}
			const base_name = segments[segments.length - 1].replace(
				/_(command|status)$/,
				"",
			);
			for (const child of cur_st.children) {
				const kernel = kernel_of_child(child);
				if (
					(kernel.sys_name === segments[segments.length - 1] &&
						"subtree" === kernel.data_type) ||
					(4 === child.container_type &&
						(("duplex" === child.kw_type && kernel.sys_name === base_name) ||
							("duplex" !== child.kw_type &&
								kernel.sys_name === segments[segments.length - 1])))
				)
					return child;
			}
		}
	throw new Error(`Unknown branch ${branch_name}`);
}
export function find_keyword(annotated_schema, branch_name, kw_name) {
	const child = find_branch(annotated_schema, `${branch_name}.${kw_name}`);
	if (4 !== child.container_type)
		throw new Error(
			`Mistyped result (${branch_name} should have been a keyword, but is a subtree)`,
		);
	return child;
}
export function find_subtree_within(desc, relative_kwl) {
	let cur_desc = desc;
	for (const top of relative_kwl.split(".")) {
		const sys_name = path_strip_trailing_index(top),
			kernel = kernel_of_child(cur_desc);
		if ("subtree" !== kernel.data_type) return;
		const child_st = (() => {
			for (const child of kernel.children) {
				const child_kernel = kernel_of_child(child);
				if (child_kernel.sys_name === sys_name)
					return "subtree" === child_kernel.data_type ? child : void 0;
			}
		})();
		if (void 0 === child_st) return;
		cur_desc = child_st;
		const maybe_index = path_index(top);
		switch (child_st.container_type) {
			case 2:
			case 1:
				if (void 0 !== maybe_index) {
					if (maybe_index >= child_st.capacity) return;
					cur_desc = cur_desc.contents;
				}
				break;
			case 0:
				if (void 0 !== maybe_index) return;
		}
	}
	return cur_desc;
}
