import { enforce, enforce_nonnull } from "./utilities.js";
export class Duration {
	unit;
	value;
	static abs(x) {
		return new Duration(Math.abs(x.value), x.unit);
	}
	constructor(x, unit) {
		switch (unit) {
			case "h":
				(this.value = 3600 * x), (this.unit = "s");
				break;
			case "min":
				(this.value = 60 * x), (this.unit = "s");
				break;
			default:
				(this.value = x), (this.unit = "us" === unit ? "µs" : unit);
		}
	}
	static from_string(raw_input) {
		const m = raw_input.trim().match(/^(.*[0-9.]+)\s*(.*)$/);
		if (null === m) return;
		const num_str = m[1],
			num = parseFloat(num_str);
		if (isNaN(num)) return;
		const unit_str = m[2];
		switch (unit_str) {
			case "ns":
			case "µs":
			case "ms":
			case "s":
				return new Duration(num, unit_str);
			case "us":
				return new Duration(num, "µs");
			case "min":
				return new Duration(num / 60, "s");
			case "h":
				return new Duration(num / 3600, "s");
			default:
				return;
		}
	}
	plus(other) {
		return other.unit === this.unit
			? new Duration(other.value + this.value, this.unit)
			: new Duration(other.s() + this.s(), "s");
	}
	minus(other) {
		return other.unit === this.unit
			? new Duration(this.value - other.value, this.unit)
			: new Duration(this.s() - other.s(), "s");
	}
	divide_by(x) {
		return "number" == typeof x
			? new Duration(this.value / x, this.unit)
			: this.s() / x.s();
	}
	times(x) {
		return new Duration(this.value * x, this.unit);
	}
	ns_rounded() {
		return Math.round(this.ns());
	}
	ns() {
		switch (this.unit) {
			case "ns":
				return this.value;
			case "µs":
				return 1e3 * this.value;
			case "ms":
				return 1e6 * this.value;
			case "s":
				return 1e9 * this.value;
		}
	}
	us() {
		switch (this.unit) {
			case "ns":
				return 0.001 * this.value;
			case "µs":
				return this.value;
			case "ms":
				return 1e3 * this.value;
			case "s":
				return 1e6 * this.value;
		}
	}
	ms() {
		switch (this.unit) {
			case "ns":
				return 1e-6 * this.value;
			case "µs":
				return 0.001 * this.value;
			case "ms":
				return this.value;
			case "s":
				return 1e3 * this.value;
		}
	}
	s() {
		switch (this.unit) {
			case "ns":
				return 1e-9 * this.value;
			case "µs":
				return 1e-6 * this.value;
			case "ms":
				return 0.001 * this.value;
			case "s":
				return this.value;
		}
	}
	toString(mode) {
		if (void 0 === mode || "precise" === mode) {
			let v = this.value,
				u = this.unit;
			for (; "s" !== u && v % 1e3 == 0; )
				switch (((v /= 1e3), u)) {
					case "ns":
						u = "µs";
						break;
					case "µs":
						u = "ms";
						break;
					case "ms":
						u = "s";
				}
			return `${v} ${u}`;
		}
		{
			const ns = this.ns(),
				abs_ns = Math.abs(ns),
				rough_time = (raw_seconds) => {
					const seconds = Math.round(raw_seconds),
						hours = Math.floor(seconds / 3600),
						minutes = Math.floor((seconds - 3600 * hours) / 60),
						rem = Math.round(seconds - 3600 * hours - 60 * minutes),
						pad = (num) => ("0" + num).slice(-2);
					return `${pad(hours)}:${pad(minutes)}:${pad(rem)}`;
				},
				to_fixed = (x, digits) => x.toFixed(digits).replace(/\.0+$/, "");
			if (abs_ns < 1e3) return `${to_fixed(ns, 1)} ns`;
			if (abs_ns < 1e6) return `${to_fixed(ns / 1e3, 1)} µs`;
			if (abs_ns < 1e9) return `${to_fixed(ns / 1e6, 1)} ms`;
			if (abs_ns < 6e10) return `${to_fixed(ns / 1e9, 3)} s`;
			if (abs_ns < 864e11)
				return `${ns < 0 ? "-" : ""}${rough_time(1e-9 * abs_ns)}`;
			{
				const days = Math.floor(abs_ns / 864e11),
					prefix = `${ns < 0 ? "-" : ""}${days}d`;
				return 1e-9 * abs_ns - 86400 * days >= 0.5
					? `${prefix} ${rough_time(1e-9 * abs_ns - 86400 * days)}`
					: prefix;
			}
		}
	}
	equal(other) {
		const abs_diff_s = Math.abs(this.s() - other.s());
		return (
			abs_diff_s < 1e-9 ||
			abs_diff_s < 1e-16 * Math.max(Math.abs(this.s()), Math.abs(other.s()))
		);
	}
	static max(x, ...ys) {
		let greatest = x;
		for (const y of ys) y.s() > greatest.s() && (greatest = y);
		return greatest;
	}
	static min(x, ...ys) {
		let smallest = x;
		for (const y of ys) y.s() < smallest.s() && (smallest = y);
		return smallest;
	}
}
export class Timestamp {
	ns_total;
	constructor(x) {
		switch (typeof x) {
			case "bigint":
				this.ns_total = x;
				break;
			case "number":
				this.ns_total = BigInt(Math.round(1e9 * x));
				break;
			case "string": {
				const parts = x.split(".");
				switch (parts.length) {
					case 1:
						this.ns_total = BigInt(parts[0]) * BigInt(1e9);
						break;
					case 2:
						if (
							((this.ns_total =
								BigInt(parts[0]) * BigInt(1e9) + BigInt(parts[1])),
							this.ns_total < 0)
						)
							throw new Error(`Unable to interpret '${x}' as a timestamp`);
						break;
					default:
						throw new Error(`Unable to interpret '${x}' as a timestamp`);
				}
			}
		}
	}
	nanoseconds() {
		return Number(this.ns_total % BigInt(1e9));
	}
	seconds() {
		return Number(this.ns_total / BigInt(1e9));
	}
	distance_to(other) {
		return new Duration(Number(other.ns_total - this.ns_total), "ns");
	}
	distance_from(other) {
		return other.distance_to(this);
	}
	plus(d) {
		return new Timestamp(this.ns_total + BigInt(d.ns_rounded()));
	}
	minus(d) {
		return this.plus(d.times(-1));
	}
	toString(mode) {
		const secs = this.ns_total / BigInt(1e9),
			ns = this.ns_total % BigInt(1e9);
		return "convenient" === mode && ns === BigInt(0)
			? secs.toString()
			: secs.toString() + "." + ns.toString().padStart(9, "0");
	}
	toNumber() {
		return 1e-9 * Number(this.ns_total);
	}
	toNSBigInt() {
		return this.ns_total;
	}
	toJSON() {
		return this.toString("backend-style");
	}
	equal(other) {
		return other.ns_total === this.ns_total;
	}
}
