import { Duration } from "./time.js";
import { enforce_nonnull, unreachable, enforce } from "./asserts.js";
export * from "./asserts.js";
export function pause(t) {
	return new Promise((resolve) => {
		setTimeout(() => {
			resolve();
		}, t.ms());
	});
}
export function path_hd(path) {
	const i = path.lastIndexOf(".");
	return -1 !== i ? path.substring(0, i) : "";
}
export function path_tl(path) {
	const i = path.lastIndexOf(".");
	return -1 !== i ? path.substring(i + 1) : path;
}
export function path_to_branch(path) {
	return path.replace(/\[[0-9]+\]/g, "");
}
export function path_strip_trailing_index(path) {
	return path.replace(/\[[0-9]+\]$/, "");
}
export function path_index(path) {
	if ("]" !== path.charAt(path.length - 1)) return;
	const i = path.lastIndexOf("[");
	return -1 !== i ? parseInt(path.substr(i + 1, path.length - i - 2)) : void 0;
}
export function path_compare(a, b) {
	const by_branch = path_to_branch(a).localeCompare(path_to_branch(b));
	if (0 !== by_branch) return by_branch;
	const num_re = /(\d+)/;
	let m_a = num_re.exec(a),
		m_b = num_re.exec(b);
	for (; m_a; ) {
		const diff = parseInt(m_a[1]) - parseInt(m_b[1]);
		if (0 !== diff) return diff;
		(m_a = num_re.exec(a.substring(m_a.index + m_a[1].length))),
			(m_b = num_re.exec(b.substring(m_b.index + m_b[1].length)));
	}
	return 0;
}
export function snake_case(s) {
	let result = "";
	for (const C of s) {
		const c = C.toLowerCase();
		C !== c && 0 !== result.length && (result += "_"), (result += c);
	}
	return result;
}
export class WorkQueue {
	num_workers;
	num_running = 0;
	todo = [];
	finalizers = [];
	thrown_exceptions = [];
	m_waiting = !0;
	constructor(pars) {
		(this.num_workers = pars.num_workers),
			(pars.start_immediately ?? 1) && this.start();
	}
	get size() {
		return this.todo.length + this.num_running;
	}
	get stats() {
		return { waiting: this.todo.length, running: this.num_running };
	}
	tick() {
		const n = Math.min(this.num_workers - this.num_running, this.todo.length),
			pop = this.todo.splice(0, n);
		this.num_running += n;
		for (const f of pop)
			f()
				.catch((e) => this.thrown_exceptions.push(e))
				.finally(() => {
					(this.num_running -= 1), this.m_waiting || this.tick();
				});
		if (this.idle()) {
			for (const [resolve, reject] of this.finalizers)
				0 === this.thrown_exceptions.length
					? resolve()
					: reject(this.thrown_exceptions);
			this.finalizers = [];
		}
	}
	start() {
		(this.m_waiting = !1), this.tick();
	}
	idle() {
		return 0 === this.num_running && 0 === this.todo.length;
	}
	push(f) {
		this.todo.push(f), this.m_waiting || this.tick();
	}
	clear() {
		(this.todo = []), this.m_waiting || this.tick();
	}
	drain() {
		return (
			(this.m_waiting = !1),
			new Promise((resolve, reject) => {
				this.finalizers.push([resolve, reject]), this.tick();
			})
		);
	}
}
export async function poll(criterion, pars) {
	const ms0 = new Date().valueOf();
	for (;;) {
		if (await criterion()) return;
		const timeout = pars?.timeout ?? new Duration(5, "s");
		if (new Date().valueOf() > ms0 + timeout.ms())
			throw new Error(`Failed to satisfy ${criterion} within ${timeout} s`);
		await pause(pars?.interval ?? new Duration(200, "ms"));
	}
}
export async function asyncFilter(xs, f, pars) {
	const wq = new WorkQueue({ num_workers: pars?.num_workers ?? 32 }),
		result = [];
	return (
		xs.forEach((x, i) => {
			wq.push(async () => {
				(await f(x)) && result.push([x, i]);
			});
		}),
		await wq.drain(),
		result.sort((a, b) => a[1] - b[1]),
		result.map((a) => a[0])
	);
}
export async function asyncFilterMap(xs, f, pars) {
	const wq = new WorkQueue({ num_workers: pars?.num_workers ?? 32 }),
		result = [];
	return (
		xs.forEach((x, i) => {
			wq.push(async () => {
				result.push([await f(x), i]);
			});
		}),
		await wq.drain(),
		result.sort((a, b) => a[1] - b[1]),
		result.filter((x) => void 0 !== x[0]).map((a) => a[0])
	);
}
export async function asyncMap(xs, f, pars) {
	const result = [],
		wq = new WorkQueue({ num_workers: pars?.num_workers ?? 32 });
	return (
		xs.forEach((x, i) => {
			wq.push(async () => {
				result.push([await f(x, i), i]);
			});
		}),
		await wq.drain(),
		result.sort((a, b) => a[1] - b[1]),
		result.map((a) => a[0])
	);
}
export async function poll_until(f, pars) {
	const ms0 = new Date().valueOf();
	for (;;) {
		const x = await f();
		if (!0 === x.satisfied) return x.result;
		if (
			(await pause(pars.pollInterval ?? new Duration(1, "s")),
			new Date().valueOf() > ms0 + pars.timeout.ms())
		)
			throw new Error(
				`Failed to satisfy criterion ${f} within ${pars.timeout} s`,
			);
	}
}
export async function asyncFind(xs, criterion) {
	for (const x of xs) if (await criterion(x)) return x;
}
export async function asyncGet(xs, criterion) {
	const maybeResult = await asyncFind(xs, criterion);
	if (void 0 === maybeResult)
		throw new Error(`Found no list element satisfying ${criterion}`);
	return maybeResult;
}
export async function asyncFindMap(xs, f) {
	for (const x of xs) {
		const y = await f(x);
		if (void 0 !== y) return y;
	}
}
export async function asyncGetMap(xs, f) {
	const maybeResult = await asyncFindMap(xs, f);
	if (void 0 === maybeResult)
		throw new Error(`Found no list element satisfying ${f}`);
	return maybeResult;
}
export async function asyncIter(things, f, opts) {
	const wq = new WorkQueue(opts ?? { num_workers: 32 });
	for (let i = 0; i < things.length; ++i) wq.push(() => f(things[i], i));
	await wq.drain();
}
export async function asyncZip(left_things, right_things, f, opts) {
	await asyncIter(
		Array.from(
			{ length: Math.min(left_things.length, right_things.length) },
			(_, i) => i,
		),
		(i) => f(left_things[i], right_things[i], i),
		opts,
	);
}
export function parse_machine_address(str) {
	const m = str.match(/(.*):\/\/(.*)/);
	if (!m) return { protocol: "ws", ip: str };
	switch (m[1].toLowerCase()) {
		case "https":
		case "wss":
			return { protocol: "wss", ip: m[2] };
		case "http":
		case "ws":
			return { protocol: "ws", ip: m[2] };
		default:
			unreachable();
	}
}
export function numstring_with_precision(payload, precision, force_precise) {
	return force_precise || Number.isInteger(payload)
		? payload.toString()
		: payload.toPrecision(precision).replace(/0+e/, "e");
}
export function scale_format(payload, scale, precision) {
	let i = 0;
	for (; i < scale.length - 1 && scale[i + 1][0] < Math.abs(payload); ) ++i;
	return `${numstring_with_precision(payload / scale[i][0], precision, !1)} ${scale[i][1]}`;
}
export function format_with_unit(payload, unit, precision) {
	return (
		unit.exceptions_map.get(payload) ??
		scale_format(payload, unit.scale, precision)
	);
}
export function addrs_equal_up_to_port(pars) {
	const command = pars.command ?? "",
		status = pars.status ?? "";
	return command.match(/:[0-9]+$/)
		? command.trim() === status.trim()
		: command.trim() === status.trim().replace(/:[0-9]+$/, "");
}
