import {
	Subtree,
	SubtreeArray,
	SubtreeNamedTable,
	SubtreeTable,
} from "./data_views.js";
import { Duration, Timestamp } from "./time.js";
import { kernel_of_child } from "./schema_v2.js";
import {
	enforce,
	enforce_nonnull,
	path_hd,
	path_index,
	path_strip_trailing_index,
	path_tl,
	pause,
} from "./utilities.js";
import {
	RefinedIndexedMutatingReference,
	RefinedIndexedNonmutatingReference,
	RefinedMutatingReference,
	RefinedNonmutatingReference,
} from "./voverlays.js";
export function get_subtree(socket, kwl) {
	return socket.root.find(kwl);
}
export function component_checked(socket, expected_sys_name) {
	const child_desc = socket.schema.keywords.find(
		(comp) => comp.contents.sys_name === expected_sys_name && comp.enabled,
	);
	return void 0 === child_desc
		? null
		: new Subtree(socket, expected_sys_name, child_desc.contents);
}
function child_not_found_errmsg(expected_sys_name, subtree) {
	const children_suffix =
			0 === subtree.description.container_type
				? ` (children available at the current level are: ${subtree.description.children.map((c) => kernel_of_child(c).sys_name).join(", ")})`
				: "",
		vsocket = subtree.backing_store;
	return `Unable to find child '${expected_sys_name}' ${children_suffix}. This might be caused by a version mismatch between the vapi module you're currently using and the software version ${vsocket.build_info.version}[model: ${vsocket.build_info.hardware_model}] running on the target machine ${vsocket.ip}`;
}
export function subtree_child_checked(subtree, child_index, expected_sys_name) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			0 === child_desc.container_type &&
			child_desc.sys_name === expected_sys_name
		)
			return new Subtree(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function subtree_table_child_checked(
	subtree,
	child_index,
	expected_sys_name,
) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			1 === child_desc.container_type &&
			child_desc.contents.sys_name === expected_sys_name
		)
			return new SubtreeTable(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function subtree_named_table_child_checked(
	subtree,
	child_index,
	expected_sys_name,
) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			1 === child_desc.container_type &&
			!0 === child_desc.named_tables &&
			child_desc.contents.sys_name === expected_sys_name
		)
			return new SubtreeNamedTable(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function subtree_array_child_checked(
	subtree,
	child_index,
	expected_sys_name,
) {
	for (let i = 0; i < subtree.children.length; ++i) {
		const child_desc =
			subtree.children[(child_index + i) % subtree.children.length];
		if (
			2 === child_desc.container_type &&
			child_desc.contents.sys_name === expected_sys_name
		)
			return new SubtreeArray(
				subtree.backing_store,
				subtree.kwl + "." + expected_sys_name,
				child_desc,
			);
	}
	throw new Error(child_not_found_errmsg(expected_sys_name, subtree));
}
export function liftOrLowerNullable(f) {
	return (maybe_payload, socket) =>
		null === maybe_payload ? null : f(maybe_payload, socket);
}
export function liftNanoseconds(payload, _) {
	return new Duration(payload, "ns");
}
export function liftSeconds(payload, _) {
	return new Duration(payload, "s");
}
export function liftTimestamp(payload, _) {
	return new Timestamp(payload);
}
export function lowerTimestamp(ts, _) {
	return ts.toString("backend-style");
}
export function lowerNanoseconds(payload, _) {
	return payload.ns_rounded();
}
export function lowerSeconds(payload, _) {
	return payload.s();
}
export function liftArray(f) {
	return (raw_elems, socket) => raw_elems.map((x) => f(x, socket));
}
export function mapMaybeSparseArray(raw_elems, f) {
	if (Array.isArray(raw_elems)) return raw_elems.map((x) => f(x));
	const result = {};
	for (const k in raw_elems) result[k] = f(raw_elems[k]);
	return result;
}
export function liftMaybeSparseArray(f) {
	return (raw_elems, socket) =>
		mapMaybeSparseArray(raw_elems, (x) => f(x, socket));
}
export function liftNullable(f) {
	return (maybe_x, socket) => (null === maybe_x ? null : f(maybe_x, socket));
}
export function lowerArray(f) {
	return (elems, socket) => elems.map((x) => f(x, socket));
}
export function lowerNullableArray(f) {
	return (raw_elems, socket) =>
		null === raw_elems ? null : raw_elems.map((x) => f(x, socket));
}
export function liftRefinedMutatingReference(f, member_name, type_converters) {
	return (path, backing_store) => {
		if (null === path) return null;
		const parent = f(path_hd(path), backing_store);
		return new RefinedMutatingReference(parent, member_name, type_converters);
	};
}
export function liftRefinedIndexedMutatingReference(
	f,
	member_name,
	type_converters,
) {
	return (path, backing_store) => {
		if (null === path) return null;
		const parent = f(path_hd(path), backing_store),
			index = path_index(path);
		return new RefinedIndexedMutatingReference(
			parent,
			member_name,
			index,
			type_converters,
		);
	};
}
export function liftRefinedNonmutatingReference(
	f,
	member_name,
	type_converters,
) {
	return (path, backing_store) => {
		if (null === path) return null;
		const parent = f(path_hd(path), backing_store);
		return new RefinedNonmutatingReference(
			parent,
			member_name,
			type_converters,
		);
	};
}
export function liftRefinedIndexedNonmutatingReference(
	f,
	member_name,
	type_converters,
) {
	return (path, backing_store) => {
		if (null === path) return null;
		const parent = f(path_hd(path), backing_store),
			index = path_index(path);
		return new RefinedIndexedNonmutatingReference(
			parent,
			member_name,
			index,
			type_converters,
		);
	};
}
export function lowerRefinedReference() {
	return (maybe_ref, _) =>
		null === maybe_ref
			? null
			: `${maybe_ref.enclosing_subtree.raw.kwl}.${maybe_ref.kw()}`;
}
export function lowerRefinedIndexedReference() {
	return (maybe_ref, _) =>
		null === maybe_ref
			? null
			: `${maybe_ref.enclosing_subtree.raw.kwl}.${maybe_ref.kw()}[${maybe_ref.index}]`;
}
export async function legacy_table_resize(pars) {
	const retry_interval = pars.retry_interval ?? new Duration(200, "ms");
	let indices = await pars.table.allocated_indices();
	for (; indices.length < pars.n; ) {
		for (let i = 0; i < pars.n - indices.length; ++i)
			await pars.add_row.write("Click");
		await pause(retry_interval),
			(indices = await pars.table.allocated_indices());
	}
	for (; indices.length > pars.n; ) {
		for (let i = pars.n; i < indices.length; ++i)
			await pars.delete_row(i).write("Click");
		await pause(retry_interval),
			(indices = await pars.table.allocated_indices());
	}
}
