import * as WAVE from "wav";
import * as http from "http";
import * as fs from "fs";
import { Duration, pause } from "vscript";
const total_page_size = 33554432;
const WORDS_PER_SAMPLE = 7;
const ADDR_ALIGNMENT = 16;
const ADDR_ALIGNMENT_END = WORDS_PER_SAMPLE * ADDR_ALIGNMENT;
let json_header = {
  Date: "dd/mm/yyyy",
  Time: "hh:mm:ss",
  SampleRate: 48000,
  BitDepth: 24,
  SamplesPerChannles: 0,
  Channels: 0,
  Scale_Channels: 0,
  MemTotal: 0,
  Pages: 0,
  Reader: 0,
  AddrSize: 0,
  Endianness: "",
};
class BUF_OBJ {
  shift;
  count;
  buf;
  constructor(size) {
    this.buf = Buffer.alloc(size);
    this.buf.fill(0);
    this.count = 0;
    this.shift = false;
  }
}
function calculateBlocks(mm) {
  const num_reader = Math.ceil(json_header.Scale_Channels / 16);
  let mm_single = mm / num_reader;
  let num_of_min_start_to_start = 1;
  const words_start_to_start = 1 << 14;
  while (mm_single) {
    const words = words_start_to_start * num_of_min_start_to_start;
    const usable_words = words - (words % 112);
    if (mm_single < usable_words) mm_single = 0;
    else num_of_min_start_to_start++;
  }
  let mm_sts = num_of_min_start_to_start * num_reader;
  let num_of_pages = 1;
  const num_start_in_page = 32;
  while (mm_sts) {
    const n_sip_all = num_start_in_page * num_of_pages;
    const n_sip = n_sip_all - (n_sip_all % num_reader);
    if (mm_sts < n_sip) mm_sts = 0;
    else num_of_pages++;
  }
  return [num_of_pages, num_reader];
}
const calculateSingleInputBuffer = (page) => {
  const num_reader = Math.ceil(json_header.Scale_Channels / 16);
  const words_start_to_start = 1 << 14;
  const words_in_page = total_page_size / 64;
  const total_words = words_in_page * page;
  const total_words_pro_wr = total_words / num_reader;
  const total_sts_pro_wr = Math.ceil(total_words_pro_wr / words_start_to_start);
  const total_mm_pro_wr = total_sts_pro_wr * words_start_to_start;
  const total_mm_pro_wr_align = total_mm_pro_wr - (total_mm_pro_wr % ADDR_ALIGNMENT_END);
  return total_mm_pro_wr_align * 64;
};
export function encode_biw(data) {
  const json_string = JSON.stringify(data.header);
  const json_extra = 8 - (json_string.length % 8);
  let data_length = 0;
  for (const buf_16_channels of data.t_buffer) {
    data_length += buf_16_channels.buf.length;
  }
  const data_buf = Buffer.alloc(json_string.length + json_extra + data_length);
  data_buf.fill(json_string, 0, json_string.length);
  for (let i = json_string.length; i < json_string.length + json_extra; i++)
    data_buf.writeInt8(0x00, i);
  data_length = 0;
  for (const buf_16_channels of data.t_buffer) {
    buf_16_channels.buf.copy(data_buf, json_string.length + json_extra + data_length);
    data_length += buf_16_channels.buf.length;
  }
  return data_buf;
}
export async function load_wave(source_file) {
  let done_file = false;
  const stat = fs.statSync(source_file);
  const file = fs.createReadStream(source_file);
  const reader = new WAVE.Reader();
  const result = {
    header: json_header,
    t_buffer: [new BUF_OBJ(0), new BUF_OBJ(0), new BUF_OBJ(0), new BUF_OBJ(0), new BUF_OBJ(0)],
  };
  reader.on("format", function (format) {
    const d = new Date();
    result.header.Date = `${d.getDate()}/${d.getMonth() + 1}/${d.getFullYear()}`;
    result.header.Time = `${d.getHours()}:${d.getMinutes()}:${d.getSeconds()}`;
    result.header.Endianness = format.endianness;
    result.header.Channels = format.channels;
    result.header.BitDepth = format.bitDepth;
    result.header.SampleRate = format.sampleRate;
    result.header.SamplesPerChannles = (stat.size - 44) / (format.channels * (format.bitDepth / 8));
    result.header.Scale_Channels =
      format.channels <= 16
        ? 16
        : format.channels <= 32
          ? 32
          : format.channels <= 48
            ? 48
            : format.channels <= 64
              ? 64
              : format.channels <= 80
                ? 80
                : 0;
    result.header.MemTotal =
      (result.header.SamplesPerChannles * result.header.Scale_Channels * 28) / 64;
    [result.header.Pages, result.header.Reader] = calculateBlocks(result.header.MemTotal);
    result.header.AddrSize = calculateSingleInputBuffer(result.header.Pages);
    const num_reader = Math.ceil(result.header.Scale_Channels / 16);
    for (let idx = 0; idx < num_reader; idx++) {
      const start_granularity = 1 << 14;
      const mm_byte =
        Math.ceil((json_header.SamplesPerChannles * 7) / start_granularity) * start_granularity * 8;
      result.t_buffer[idx] = new BUF_OBJ(mm_byte);
    }
  });
  reader.on("cart", function (format) {
    console.log("cart", format);
  });
  reader.on("bext", function (format) {
    console.log("bext", format);
  });
  let count_channel = 0;
  let count_block = 0;
  let frame_count = 0;
  const channel_status = new Array(24).fill(0);
  channel_status[0] = 0x85;
  channel_status[2] = 0x2c;
  channel_status[23] = 0x2b;
  let last_sample = {
    sample: 0,
    sample_byte: 0,
  };
  reader.on("end", function () {
    done_file = true;
  });
  reader.on("data", function (data) {
    const buf = data;
    for (let idx = 0; idx < buf.length; ) {
      const bytes = result.header.BitDepth / 8;
      for (let sample_byte = last_sample.sample_byte; sample_byte < bytes; sample_byte++) {
        if (json_header.Endianness == "LE") {
          last_sample.sample |= buf.readUInt8(idx) << ((sample_byte + (3 - bytes)) * 8);
        } else {
          last_sample.sample |= buf.readUInt8(idx) << ((bytes + (3 - bytes) - 1 - sample_byte) * 8);
        }
        idx++;
        if (idx >= buf.length) {
          last_sample.sample_byte = sample_byte + 1;
          return;
        }
      }
      let cuvz = 0;
      cuvz = !frame_count ? 1 : 0;
      cuvz |= channel_status[frame_count >> 3] & (1 << (frame_count & 7)) ? 1 << 3 : 0;
      last_sample.sample |= cuvz << 24;
      let sample = last_sample.sample;
      last_sample = { sample: 0, sample_byte: 0 };
      const t_buf = result.t_buffer[count_block];
      const write_normal = () => {
        t_buf.buf.writeUInt8(sample & 0xff, t_buf.count++);
        sample >>= 8;
        t_buf.buf.writeUInt8(sample & 0xff, t_buf.count++);
        sample >>= 8;
        t_buf.buf.writeUInt8(sample & 0xff, t_buf.count++);
        sample >>= 8;
        t_buf.buf.writeUInt8(sample & 0xf, t_buf.count);
      };
      const write_shift = () => {
        let val = t_buf.buf.readUInt8(t_buf.count);
        val |= (sample & 0xf) << 4;
        t_buf.buf.writeUInt8(val, t_buf.count++);
        sample >>= 4;
        t_buf.buf.writeUInt8(sample & 0xff, t_buf.count++);
        sample >>= 8;
        t_buf.buf.writeUInt8(sample & 0xff, t_buf.count++);
        sample >>= 8;
        t_buf.buf.writeUInt8(sample & 0xff, t_buf.count++);
      };
      t_buf.shift ? write_shift() : write_normal();
      t_buf.shift = !t_buf.shift;
      if (count_channel == json_header.Channels - 1) {
        if (frame_count == 191) frame_count = 0;
        else frame_count++;
        count_channel = 0;
      } else count_channel++;
      const block = count_channel >> 4;
      if (block != count_block || (!count_channel && json_header.Channels % 16)) {
        const mod = t_buf.count % 56;
        if (mod) {
          t_buf.count = t_buf.count + 56 - mod;
        }
        count_block = block;
      }
    }
  });
  file.pipe(reader);
  while (!done_file) {
    await pause(new Duration(1, "s"));
  }
  return result;
}
export function upload(url, data) {
  return new Promise((resolve, reject) => {
    const req = http.request(
      url,
      {
        method: "PUT",
        headers: {
          "Content-Type": "application/octet-stream",
          "Content-Length": data.length,
        },
      },
      (res) => {
        if (res.statusCode == 200) resolve(res.statusCode);
        else reject(new Error(`HTTP status code: ${res.statusCode}`));
      },
    );
    req.on("error", (e) => {
      reject(new Error(`HTTP error: ${e.message}`));
    });
    req.end(data);
  });
}
export function ll_to_wave(data, channels) {
  if (channels < 1 || channels > 16) throw Error("Invalid channel number");
  const samples = Math.floor(data.length / (7 * 8));
  const audio = Buffer.alloc(samples * channels * 3);
  for (let s = 0; s < samples; ++s) {
    const data_off = s * (7 * 8);
    const audio_off = s * channels * 3;
    const read_sample = (c) => {
      if ((c & 1) == 0) {
        const b0 = data.readUInt8(data_off + (c >> 1) * 7 + 0);
        const b1 = data.readUInt8(data_off + (c >> 1) * 7 + 1);
        const b2 = data.readUInt8(data_off + (c >> 1) * 7 + 2);
        return b0 | (b1 << 8) | (b2 << 16);
      } else {
        const b0 = data.readUInt8(data_off + (c >> 1) * 7 + 3);
        const b1 = data.readUInt8(data_off + (c >> 1) * 7 + 4);
        const b2 = data.readUInt8(data_off + (c >> 1) * 7 + 5);
        const b3 = data.readUInt8(data_off + (c >> 1) * 7 + 6);
        return (b0 >> 4) | (b1 << 4) | (b2 << 12) | ((b3 & 0x0f) << 20);
      }
    };
    for (let c = 0; c < channels; ++c) {
      const sample = read_sample(c);
      audio.writeUInt8(sample & 0xff, audio_off + c * 3 + 0);
      audio.writeUInt8((sample >> 8) & 0xff, audio_off + c * 3 + 1);
      audio.writeUInt8((sample >> 16) & 0xff, audio_off + c * 3 + 2);
    }
  }
  return audio;
}
