import { mkdtempSync, writeFileSync } from "fs";
import { join } from "path";
import { enforce, enforce_nonnull } from "vscript";
import { sh } from "./shell.js";
export function parse_diff(diff_txt) {
  const result = [];
  for (const line of diff_txt.split("\n")) {
    if (!line.startsWith("@@")) continue;
    const contents = line.substr(2).replace(/@@.*/, "").trim().split(" ");
    const split_hunkrange = (hunkrange) => {
      const parts = hunkrange.replace(/[-\+]/g, "").split(",");
      const start = parseInt(enforce_nonnull(parts[0]), 10);
      const end = parseInt(parts.length > 1 ? enforce_nonnull(parts[1]) : "1", 10) + start;
      return { start, end };
    };
    enforce(enforce_nonnull(contents[0]).startsWith("-"));
    enforce(enforce_nonnull(contents[1]).startsWith("+"));
    result.push({
      removed: split_hunkrange(enforce_nonnull(contents[0])),
      inserted: split_hunkrange(enforce_nonnull(contents[1])),
    });
  }
  return result;
}
export async function diff_files(pars) {
  const diff_txt = (
    await sh(`diff -u '${pars.left_filename}' '${pars.right_filename}'`, {
      fail_on_error: false,
      max_buffer_bytes: 65 * 1024 * 1024,
    })
  ).stdout;
  return { diff_txt, diff_hunks: parse_diff(diff_txt) };
}
export async function diff(pars) {
  const tmpdir = mkdtempSync("/tmp/");
  const left_filename = join(tmpdir, "left.txt");
  const right_filename = join(tmpdir, "right.txt");
  writeFileSync(left_filename, pars.left.join("\n"));
  writeFileSync(right_filename, pars.right.join("\n"));
  return await diff_files({ left_filename, right_filename });
}
export function matching_ranges(diff_hunks, length_left, length_right) {
  const result = [];
  let line_start_left = 1;
  let line_start_right = 1;
  for (const diff_hunk of diff_hunks) {
    const next_left = diff_hunk.removed.start;
    const next_right = diff_hunk.inserted.start;
    if (next_left !== line_start_left) {
      const num_lines = next_left - line_start_left;
      enforce(next_right === line_start_right + num_lines);
      result.push({ num_lines, line_start_left, line_start_right });
      line_start_left = diff_hunk.removed.end;
      line_start_right = diff_hunk.inserted.end;
    } else {
      line_start_left = diff_hunk.removed.end;
      line_start_right = diff_hunk.inserted.end;
    }
  }
  if (length_left > line_start_left) {
    const num_lines = length_left + 1 - line_start_left;
    enforce(length_right + 1 - line_start_right === num_lines);
    result.push({ num_lines, line_start_left, line_start_right });
  }
  return result;
}
