import { Duration, enforce_nonnull, parse_machine_address } from "vscript";
import { env_keys, getenv } from "./node/polyfills.js";
class MissingParameter {}
export class ParseFailure {}
function make_bool_parspec(default_value) {
  return {
    parser: parse_bool(default_value),
    description: () => "bool ('true' or 'false')",
  };
}
async function reify_default(machines, maybe_default) {
  if (maybe_default === undefined) return undefined;
  if (typeof maybe_default === "function") {
    const tmp = maybe_default(machines);
    if (typeof tmp.then === "function") {
      return await tmp;
    } else {
      return tmp;
    }
  }
  return maybe_default;
}
function make_number_parspec(num_type, pars) {
  return {
    parser: parse_number(num_type, pars),
    description: () => {
      const suffix = (() => {
        if (pars?.min !== undefined && pars?.max !== undefined) {
          return ` between ${pars.min} and ${pars.max} (inclusive)`;
        } else if (pars?.min !== undefined) {
          return ` >= ${pars.min}`;
        } else if (pars?.max !== undefined) {
          return ` <= ${pars.max}`;
        }
        return "";
      })();
      return `${num_type}${suffix}`;
    },
  };
}
function make_string_parspec(default_value) {
  return {
    parser: parse_string(default_value),
    description: () => "string",
  };
}
function make_enum_parspec(enumerators, default_value) {
  return {
    parser: parse_enum({ enumerators, default: default_value }),
    description: () => `one of the following: ${enumerators.join(", ")}`,
  };
}
function make_list_parspec(element_parspec, default_value) {
  return {
    parser: parse_list(element_parspec.parser, default_value),
    description: () => `a comma-separated list of [${element_parspec.description()}]`,
  };
}
function make_optional_parspec(inner_parspec) {
  return {
    description: () => inner_parspec.description() + " (optional)",
    parser: async (env_val, machines) => {
      if (env_val === undefined) return null;
      return await inner_parspec.parser(env_val, machines);
    },
  };
}
function wrap_missing(inner_parser, default_value) {
  return async (env_val, machines) => {
    if (env_val === undefined) {
      if (default_value !== undefined)
        return enforce_nonnull(await reify_default(machines, default_value));
      throw new MissingParameter();
    }
    return await inner_parser(env_val, machines);
  };
}
function parse_bool(default_value) {
  return wrap_missing(async (env_val) => {
    switch ((env_val ?? "").trim().toLowerCase()) {
      case "false":
        return false;
      case "true":
        return true;
    }
    throw new ParseFailure();
  }, default_value);
}
function parse_number_ll(env_val, num_type, pars) {
  const i = (num_type === "int" ? parseInt : parseFloat)(env_val.trim(), 10);
  if (
    isNaN(i) ||
    (pars?.min !== undefined && i < pars?.min) ||
    (pars?.max !== undefined && i > pars?.max)
  ) {
    throw new ParseFailure();
  }
  return i;
}
function parse_number(num_type, pars) {
  return wrap_missing(async (env_val) => {
    return parse_number_ll(env_val, num_type, pars);
  }, pars?.default);
}
function parse_enum_ll(env_val, enumerators) {
  const val_lower = env_val.toLowerCase();
  for (const choice of enumerators) {
    if (choice.trim().toLowerCase() === val_lower) return choice;
  }
  throw new ParseFailure();
}
function parse_enum(pars) {
  return wrap_missing(async (env_val) => {
    return parse_enum_ll(env_val, pars.enumerators);
  }, pars.default);
}
function parse_string(default_value) {
  return wrap_missing(async (env_val) => env_val, default_value);
}
function parse_list(element_parser, default_value) {
  return wrap_missing(async (env_val, machines) => {
    return await Promise.all(env_val.split(",").map((s) => element_parser(s, machines)));
  }, default_value);
}
function make_machine_address_parspec() {
  return {
    parser: wrap_missing(async (str) => {
      try {
        return parse_machine_address(str);
      } catch (_) {
        throw new ParseFailure();
      }
    }, undefined),
    description: () =>
      "machine address (examples are 172.16.1.23, ws://192.168.1.45 or wss://172.16.100.0)",
  };
}
function make_duration_parspec(pars) {
  return {
    parser: wrap_missing(async (env_val, _machines) => {
      const m = env_val.match(/([0-9]*(?:\.[0-9]*)?)(s|ms|us|µs|ns|h|min)/);
      if (!m) throw new ParseFailure();
      const value = parseFloat(enforce_nonnull(m[1]));
      if (isNaN(value)) throw new ParseFailure();
      let dur;
      switch (m[2]) {
        case "s":
        case "ms":
        case "us":
        case "µs":
        case "ns":
        case "h":
        case "min":
          dur = new Duration(value, m[2]);
          break;
        default:
          throw new ParseFailure();
      }
      if (
        (pars?.min !== undefined && dur.s() < pars?.min.s()) ||
        (pars?.max !== undefined && dur.s() > pars?.max.s())
      )
        throw new ParseFailure();
      return dur;
    }, pars?.default),
    description: () => {
      if (pars?.min !== undefined && pars?.max !== undefined) {
        return `Duration between ${pars.min} and ${pars.max} (inclusive)`;
      } else if (pars?.min !== undefined) {
        return `Duration >= ${pars.min}`;
      } else if (pars?.max !== undefined) {
        return `Duration <= ${pars.max}`;
      }
      return "Duration";
    },
  };
}
function make_first_of_parspec(a, b, maybe_default) {
  const description = () => `either (${a.description()}) or (${b.description()})`;
  return {
    description,
    parser: async (env_val, machines) => {
      if (env_val === undefined) {
        const default_value = await reify_default(machines, maybe_default);
        if (default_value === undefined)
          throw new Error(
            `Missing parameter: please specify ${description()} or provide a default value`,
          );
        return default_value;
      }
      let error_a = "";
      try {
        return await a.parser(env_val, machines);
      } catch (e) {
        error_a = `${e}`;
      }
      try {
        return await b.parser(env_val, machines);
      } catch (e) {
        throw new Error(
          `Unable to parse ${env_val} as ${description()}: trying to parse as ${a.description()} failed with '${error_a.trim()}', while trying to parse as ${b.description()} failed with '${`${e}`.trim()}'`,
        );
      }
    },
  };
}
export const ENV = {
  bool: make_bool_parspec,
  true: make_bool_parspec(true),
  false: make_bool_parspec(false),
  string: make_string_parspec,
  int: (pars) => make_number_parspec("int", pars),
  float: (pars) => make_number_parspec("float", pars),
  enum: make_enum_parspec,
  duration: make_duration_parspec,
  list_of: make_list_parspec,
  custom: (ps) => ps,
  machine_address: make_machine_address_parspec(),
  optional: make_optional_parspec,
  first_of: make_first_of_parspec,
};
export async function env(parspecs) {
  return derive_values({ parspecs, defaults: {}, machines: [] });
}
export async function derive_values({ parspecs, defaults, machines }) {
  const result = {};
  const extract_from_env = (parname, typedesc) => {
    for (const key2 of env_keys(parname)) {
      const maybe_value = getenv(key2, typedesc);
      if (maybe_value !== undefined) return maybe_value;
    }
    return undefined;
  };
  for (const k in parspecs) {
    result[k] = extract_from_env(k, parspecs[k].description());
    try {
      if (result[k] === undefined && defaults[k] !== undefined) {
        result[k] = await reify_default(machines, defaults[k]);
      } else {
        const parspec = enforce_nonnull(parspecs[k]);
        result[k] = await parspec.parser(result[k], machines);
      }
    } catch (e) {
      if (e instanceof MissingParameter) {
        throw new Error(
          `Missing parameter ${k}: please specify ${enforce_nonnull(parspecs[k]?.description())}\n`,
        );
      } else if (e instanceof ParseFailure) {
        throw new Error(
          `Unable to parse ${k}=${result[k]}: please specify ${enforce_nonnull(parspecs[k]?.description())}\n`,
        );
      }
      throw e;
    }
  }
  return result;
}
