import { exec, execFile } from "child_process";
import * as process from "process";
import { enforce_nonnull } from "vscript";
export function sh(command, opts) {
  const e_stack_capture = new Error();
  const cwd = process.cwd();
  return new Promise((resolve, reject) => {
    if (opts?.echo_command ?? false) process.stderr.write(command + "\n");
    const timeout = opts?.timeout?.ms();
    const child = exec(
      command,
      { maxBuffer: opts?.max_buffer_bytes ?? 1024 * 1024, ...(timeout ? { timeout } : {}) },
      (err, stdout, stderr) => {
        if (err && (opts?.fail_on_error ?? true)) {
          let err_msg = `Error trying to run '${command}' within '${cwd}'.`;
          if (err.code) err_msg += ` Exit code: ${err.code}.`;
          if (err.signal) err_msg += ` Signal: ${err.signal}.`;
          if (err.killed) err_msg += ` The process was killed.`;
          if (stderr.trim().length > 0) err_msg += `\n== [stderr] ==\n${stderr.trim()}`;
          if (stdout.trim().length > 0) err_msg += `\n== [stdout] ==\n${stdout.trim()}`;
          const e = new Error(err_msg);
          e.stack = e_stack_capture.stack;
          reject(e);
        } else {
          resolve({ stdout, stderr, err });
        }
      },
    );
    if (opts?.forward_io ?? false) {
      process.stdin.pipe(enforce_nonnull(child.stdin));
      enforce_nonnull(child.stdout).pipe(process.stdout);
      enforce_nonnull(child.stderr).pipe(process.stderr);
    }
  });
}
export function system(command, args, opts) {
  const e_stack_capture = new Error();
  const cwd = process.cwd();
  return new Promise((resolve, reject) => {
    if (opts?.echo_command ?? false)
      process.stderr.write([command, ...args.map((s) => `'${s}'`)].join(" ") + "\n");
    const timeout = opts?.timeout?.ms();
    const child = execFile(
      command,
      args,
      { maxBuffer: opts?.max_buffer_bytes ?? 1024 * 1024, ...(timeout ? { timeout } : {}) },
      (err, stdout, stderr) => {
        if (err && (opts?.fail_on_error ?? true)) {
          let err_msg = `Error trying to run '${command}' within '${cwd}'.`;
          if (err.code) err_msg += ` Exit code: ${err.code}.`;
          if (err.signal) err_msg += ` Signal: ${err.signal}.`;
          if (err.killed) err_msg += ` The process was killed.`;
          if (stderr.trim().length > 0) err_msg += `\n== [stderr] ==\n${stderr.trim()}`;
          if (stdout.trim().length > 0) err_msg += `\n== [stdout] ==\n${stdout.trim()}`;
          const e = new Error(err_msg);
          e.stack = e_stack_capture.stack;
          reject(e);
        } else {
          resolve({ stdout, stderr, err });
        }
      },
    );
    if (opts?.forward_io ?? false) {
      process.stdin.pipe(enforce_nonnull(child.stdin));
      enforce_nonnull(child.stdout).pipe(process.stdout);
      enforce_nonnull(child.stderr).pipe(process.stderr);
    }
  });
}
export async function executable_path(executable_name) {
  const result = await sh(`which '${executable_name}'`, { fail_on_error: false });
  if (!result.err && result.stdout.trim().length > 0) return result.stdout.trim();
  return null;
}
export async function has_executable(executable_name) {
  return (await executable_path(executable_name)) !== null;
}
