import { registerWindow, SVG } from "@svgdotjs/svg.js";
import { createHash } from "crypto";
import * as fs from "fs";
import { mkdtempSync, rmSync } from "fs";
import { default as Jimp, default as jimp_pkg } from "jimp";
import * as os from "os";
import * as path from "path";
import { finished } from "stream/promises";
import { createSVGWindow } from "svgdom";
import { enforce, enforce_nonnull, path_index } from "vscript";
import { download_to_buffer, upload_from_buffer } from "./http_async.js";
import { dissect_st291_packet, ST291Decoder } from "./metadata.js";
import { sh } from "./shell.js";
import { enumerate } from "./utils.js";
import { guess_standard, is_higher_than_3G, standard_info, vend, vstart } from "./video.js";
const { read: jimp_read } = jimp_pkg;
export class BIDFrame {
  pixels;
  meta;
  constructor(pars) {
    const pixels = pars.width * pars.height;
    function compute_c_pixels() {
      switch (pars.sampling) {
        case "422":
          if (pars.width & 1) throw Error("width must be a multiple of 2");
          return pixels / 2;
        case "444":
          return pixels;
        default:
          throw Error("Unknown sampling");
      }
    }
    const c_pixels = compute_c_pixels();
    this.pixels = {
      type: "YCbCr",
      width: pars.width,
      height: pars.height,
      sampling: pars.sampling,
      y: new Int16Array(pixels),
      cb: new Int16Array(c_pixels),
      cr: new Int16Array(c_pixels),
    };
    this.meta = pars.meta;
  }
  to_ascii() {
    let result = "Format: y1 y2 cb cr\n";
    const chunks_per_line = this.pixels.height / 2;
    enforce(this.pixels.cb.length === this.pixels.cr.length);
    enforce(this.pixels.cb.length * 2 === this.pixels.y.length);
    for (let i = 0; i < this.pixels.cb.length; ++i) {
      result += `${this.pixels.y[2 * i]} ${this.pixels.y[2 * i + 1]} ${this.pixels.cb[i]} ${this.pixels.cr[i]}${i % chunks_per_line === 0 ? "\n" : " "}`;
    }
    return result;
  }
  to_hash(algorithm, encoding = "hex") {
    const h = createHash(algorithm);
    h.update(this.pixels.cb);
    h.update(this.pixels.cr);
    h.update(this.pixels.y);
    return h.digest(encoding);
  }
  static from_image(image, opts) {
    const matrix = conversion_matrix(opts?.colorspace ?? "BT709");
    for (let i = 0; i < matrix.length; ++i) matrix[i] *= 1023.0 / 255.0;
    const rgb = image.bitmap.data;
    const frame = new BIDFrame({
      width: image.getWidth(),
      height: image.getHeight(),
      sampling: "444",
    });
    const num_pixels = image.getWidth() * image.getHeight();
    for (let i = 0; i < num_pixels; ++i) {
      const r = rgb[i * 4 + 0];
      const g = rgb[i * 4 + 1];
      const b = rgb[i * 4 + 2];
      frame.pixels.y[i] = round_clamp(matrix[0] * r + matrix[1] * g + matrix[2] * b, 0, 876);
      frame.pixels.cb[i] = round_clamp(matrix[3] * r + matrix[4] * g + matrix[5] * b, -448, 448);
      frame.pixels.cr[i] = round_clamp(matrix[6] * r + matrix[7] * g + matrix[8] * b, -448, 448);
    }
    return frame;
  }
  async to_image(opts) {
    const m = invert_3x3(conversion_matrix(opts?.colorspace ?? "BT709"));
    for (let i = 0; i < m.length; ++i) m[i] *= 255.0 / 1023.0;
    enforce(this.pixels.cb.length === this.pixels.cr.length);
    enforce(2 * this.pixels.cb.length === this.pixels.y.length);
    const image = await Jimp.create(this.pixels.width, this.pixels.height);
    for (let y = 0; y < this.pixels.height; ++y) {
      for (let x = 0; x < this.pixels.width / 2; ++x) {
        const i_sample_in = (y * this.pixels.width) / 2 + x;
        const y0 = this.pixels.y[2 * i_sample_in];
        const y1 = this.pixels.y[2 * i_sample_in + 1];
        const cb = this.pixels.cb[i_sample_in];
        const cr = this.pixels.cr[i_sample_in];
        const i_pixel_out = 2 * i_sample_in;
        const i_out = 4 * i_pixel_out;
        image.bitmap.data[i_out] = round_clamp(y0 * m[0] + cb * m[1] + cr * m[2], 0, 255);
        image.bitmap.data[i_out + 1] = round_clamp(y0 * m[3] + cb * m[4] + cr * m[5], 0, 255);
        image.bitmap.data[i_out + 2] = round_clamp(y0 * m[6] + cb * m[7] + cr * m[8], 0, 255);
        image.bitmap.data[i_out + 3] = 255;
        image.bitmap.data[i_out + 4] = round_clamp(y1 * m[0] + cb * m[1] + cr * m[2], 0, 255);
        image.bitmap.data[i_out + 5] = round_clamp(y1 * m[3] + cb * m[4] + cr * m[5], 0, 255);
        image.bitmap.data[i_out + 6] = round_clamp(y1 * m[6] + cb * m[7] + cr * m[8], 0, 255);
        image.bitmap.data[i_out + 7] = 255;
      }
    }
    if (opts?.timecode === "burn-in") {
      const font = await Jimp.loadFont(opts?.font ?? Jimp.FONT_SANS_32_BLACK);
      let y = image.getHeight() - 64;
      for (const substream of this.meta ?? []) {
        for (const pkt of substream) {
          const x = dissect_st291_packet(pkt);
          if (x.type === "Timecode") {
            const content = `${x.tc_type} ${x.hours.toString().padStart(2, "0")}:${x.minutes
              .toString()
              .padStart(2, "0")}:${x.seconds.toString().padStart(2, "0")}:${x.frames
              .toString()
              .padStart(2, "0")}`;
            const text = `${content} @ line ${pkt.linenr}, column ${pkt.column}`;
            image.print(font, image.getWidth() / 2, y, text);
            y -= 4 + Jimp.measureTextHeight(font, text, image.getWidth());
          }
        }
      }
    }
    return image;
  }
  to_yuv(fmt) {
    const num_pixels = this.pixels.width * this.pixels.height;
    let buf;
    switch (fmt) {
      case "YUV":
        function c2u8(c) {
          return round_clamp((c + 512) / 4, 0, 255);
        }
        function y2u8(y) {
          return round_clamp((y + 64) / 4, 0, 255);
        }
        switch (this.pixels.sampling) {
          case "422":
            buf = Buffer.alloc(num_pixels * 2);
            for (let idx = 0; idx < num_pixels; idx += 2) {
              buf.writeUInt8(c2u8(this.pixels.cb[idx / 2]), idx * 2 + 0);
              buf.writeUInt8(y2u8(this.pixels.y[idx]), idx * 2 + 1);
              buf.writeUInt8(c2u8(this.pixels.cr[idx / 2]), idx * 2 + 2);
              buf.writeUInt8(y2u8(this.pixels.y[idx + 1]), idx * 2 + 3);
            }
            break;
          case "444":
            buf = Buffer.alloc(num_pixels * 3);
            for (let idx = 0; idx < num_pixels; idx++) {
              buf.writeUInt8(c2u8(this.pixels.cb[idx]), idx * 3 + 0);
              buf.writeUInt8(c2u8(this.pixels.cr[idx]), idx * 3 + 1);
              buf.writeUInt8(y2u8(this.pixels.y[idx]), idx * 3 + 2);
            }
            break;
        }
        break;
      case "YUV10":
        switch (this.pixels.sampling) {
          case "422":
            buf = Buffer.alloc(Math.ceil((num_pixels * 2) / 3) * 4);
            let tmp = 0;
            let shift = 20;
            let pos = 0;
            function store10(n) {
              tmp |= round_clamp(n, 0, 1023) << shift;
              shift -= 10;
              if (shift < 0) {
                buf.writeUInt32BE(tmp, pos);
                shift = 20;
                tmp = 0;
                pos += 4;
              }
            }
            for (let idx = 0; idx < num_pixels; idx += 2) {
              store10(this.pixels.cb[idx / 2] + 512);
              store10(this.pixels.y[idx] + 64);
              store10(this.pixels.cr[idx / 2] + 512);
              store10(this.pixels.y[idx + 1] + 64);
            }
            store10(0);
            store10(0);
            break;
          case "444":
            buf = Buffer.alloc(num_pixels * 4);
            for (let idx = 0; idx < num_pixels; idx++) {
              const cb = round_clamp(this.pixels.cb[idx] + 512, 0, 1023);
              const cr = round_clamp(this.pixels.cr[idx] + 512, 0, 1023);
              const y = round_clamp(this.pixels.y[idx] + 64, 0, 1023);
              buf.writeUInt32BE((cb << 20) | (cr << 10) | y, idx * 4);
            }
            break;
        }
    }
    return buf;
  }
  mse(other) {
    const a = this.pixels;
    const b = other.pixels;
    if (a.width != b.width || a.height != b.height) throw Error("Dimensions must be equal");
    function sqr_error_sum(a, b, cnt) {
      if (a.length < cnt || b.length < cnt) throw Error("Arrays too short");
      let sum = 0.0;
      for (let i = 0; i < cnt; ++i) {
        let diff = a[i] - b[i];
        sum += diff * diff;
      }
      return sum;
    }
    function sqr_error_sum2(a, b, cnt) {
      if (a.length < cnt || b.length * 2 < cnt) throw Error("Arrays too short");
      let sum = 0.0;
      for (let i = 0; i < cnt; ++i) {
        let diff = a[i] - b[i * 2];
        sum += diff * diff;
      }
      return sum;
    }
    const pixels = a.width * a.height;
    if (a.sampling == "422" && b.sampling == "422") {
      const sum =
        sqr_error_sum(a.y, b.y, pixels) +
        sqr_error_sum(a.cb, b.cb, pixels / 2) +
        sqr_error_sum(a.cr, b.cr, pixels / 2);
      return sum / (pixels * 2);
    }
    if (a.sampling == "422" && b.sampling == "444") {
      const sum =
        sqr_error_sum(a.y, b.y, pixels) +
        sqr_error_sum2(a.cb, b.cb, pixels / 2) +
        sqr_error_sum2(a.cr, b.cr, pixels / 2);
      return sum / (pixels * 2);
    }
    if (a.sampling == "444" && b.sampling == "422") {
      const sum =
        sqr_error_sum(a.y, b.y, pixels) +
        sqr_error_sum2(b.cb, a.cb, pixels / 2) +
        sqr_error_sum2(b.cr, a.cr, pixels / 2);
      return sum / (pixels * 2);
    }
    if (a.sampling == "444" && b.sampling == "444") {
      const sum =
        sqr_error_sum(a.y, b.y, pixels) +
        sqr_error_sum(a.cb, b.cb, pixels) +
        sqr_error_sum(a.cr, b.cr, pixels);
      return sum / (pixels * 3);
    }
    throw Error("Unsupported sampling combination");
  }
  diff(other) {
    const a = this.pixels;
    const b = other.pixels;
    if (a.width != b.width || a.height != b.height) throw Error("Dimensions must be equal");
    if (a.sampling != b.sampling) throw Error("Sampling must be equal");
    function diff_component(a, b, ret) {
      const cnt = Math.min(a.length, b.length, ret.length);
      for (let i = 0; i < cnt; ++i) ret[i] = a[i] - b[i];
    }
    const ret = new BIDFrame({
      width: a.width,
      height: a.height,
      sampling: a.sampling,
    });
    diff_component(a.y, b.y, ret.pixels.y);
    diff_component(a.cb, b.cb, ret.pixels.cb);
    diff_component(a.cr, b.cr, ret.pixels.cr);
    return ret;
  }
  minmax() {
    function minmax_component(a) {
      const ret = {
        min: a[0],
        max: a[0],
      };
      for (let i = 1; i < a.length; ++i) {
        const v = a[i];
        if (ret.min > v) ret.min = v;
        if (ret.max < v) ret.max = v;
      }
      return ret;
    }
    return {
      y: minmax_component(this.pixels.y),
      cb: minmax_component(this.pixels.cb),
      cr: minmax_component(this.pixels.cr),
    };
  }
  apply(par) {
    function apply_component(a, ret, par) {
      const offset = par.offset ?? 0;
      const scale = par.scale ?? 1;
      const cnt = Math.min(a.length, ret.length);
      for (let i = 0; i < cnt; ++i) ret[i] = a[i] * scale + offset;
    }
    const ret = new BIDFrame({
      width: this.pixels.width,
      height: this.pixels.height,
      sampling: this.pixels.sampling,
    });
    apply_component(this.pixels.y, ret.pixels.y, par.y);
    apply_component(this.pixels.cb, ret.pixels.cb, par.cb);
    apply_component(this.pixels.cr, ret.pixels.cr, par.cr);
    return ret;
  }
  subarea(hoff, voff, width, height) {
    if (hoff < 0 || hoff >= this.pixels.width) throw new Error("hoff out of range");
    if (voff < 0 || voff >= this.pixels.height) throw new Error("voff out of range");
    if (width < 1 || width > this.pixels.width - hoff) throw new Error("width out of range");
    if (height < 1 || height > this.pixels.height - voff) throw new Error("height out of range");
    const ret = new BIDFrame({ width, height, sampling: this.pixels.sampling });
    switch (ret.pixels.sampling) {
      case "422":
        if ((hoff & 1) != 0) throw new Error("hoff must be even for 4:2:2");
        if ((width & 1) != 0) throw new Error("width must be even for 4:2:2");
        for (let y = 0; y < height; y++) {
          const dst_off = y * width;
          const src_off = (y + voff) * this.pixels.width + hoff;
          ret.pixels.y.set(this.pixels.y.subarray(src_off, src_off + width), dst_off);
          ret.pixels.cb.set(
            this.pixels.cb.subarray(src_off / 2, src_off / 2 + width / 2),
            dst_off / 2,
          );
          ret.pixels.cr.set(
            this.pixels.cr.subarray(src_off / 2, src_off / 2 + width / 2),
            dst_off / 2,
          );
        }
        break;
      case "444":
        for (let y = 0; y < height; y++) {
          const dst_off = y * width;
          const src_off = (y + voff) * this.pixels.width + hoff;
          ret.pixels.y.set(this.pixels.y.subarray(src_off, src_off + width), dst_off);
          ret.pixels.cb.set(this.pixels.cb.subarray(src_off, src_off + width), dst_off);
          ret.pixels.cr.set(this.pixels.cr.subarray(src_off, src_off + width), dst_off);
        }
        break;
    }
    return ret;
  }
}
export async function extract_header(path) {
  const stream = fs.createReadStream(path, "utf8");
  let finished = false;
  let header = Buffer.alloc(0);
  for await (const chunk of stream) {
    if (finished) break;
    const bytes = Buffer.from(chunk);
    let len = 0;
    for (const byte of bytes) {
      if (byte == 0x00) finished = true;
      if (finished) break;
      len++;
    }
    header = Buffer.concat([header, bytes.subarray(0, len)]);
  }
  return header;
}
export async function parse_header(path) {
  return JSON.parse((await extract_header(path)).toString("utf8"));
}
export async function concat_bids_on_disk(pars) {
  const header = await parse_header(pars.infiles[0]);
  const p = (n, fieldname) => `(${pars.infiles[n]}).${fieldname}`;
  for (let i = 1; i < pars.infiles.length; ++i) {
    const next_header = await parse_header(pars.infiles[i]);
    for (const key of ["Blanking", "Interlace", "HActive", "VActive", "Standard"]) {
      if (header[key] !== next_header[key])
        throw new Error(
          `${p(0, key)} = ${header[key]} is incompatible with ${p(i, key)} = ${next_header[key]}`,
        );
    }
    header.Frames += next_header.Frames;
  }
  const writer = fs.createWriteStream(pars.output_filename, { encoding: "binary" });
  const header_txt = JSON.stringify(header) + "\0";
  const padlen = (8 - (header_txt.length % 8)) % 8;
  writer.write(header_txt + "\0".repeat(padlen));
  const roundup = (l) => (8 - (l % 8)) % 8;
  for (const infile of pars.infiles) {
    const headerlen = roundup((await extract_header(infile)).length + 1);
    const reader = fs.createReadStream(infile, { encoding: "binary" });
    await reader.read(headerlen);
    reader.pipe(writer, { end: false });
    await finished(reader);
  }
  writer.close();
}
export class BID {
  buffer;
  hdr;
  hdr_end;
  constructor(buffer) {
    this.buffer = buffer;
    this.hdr_end = buffer.findIndex((v) => v === 0) + 1;
    this.hdr = JSON.parse(buffer.subarray(0, this.hdr_end - 1).toString("utf8"));
    this.hdr_end += (8 - (this.hdr_end % 8)) % 8;
    enforce(this.hdr_end % 8 === 0);
  }
  raw_data() {
    return this.buffer.subarray(this.hdr_end);
  }
  bid_geometry() {
    const [width, height] = this.hdr.Blanking
      ? [this.hdr.HTotal, this.hdr.VTotal]
      : [this.hdr.HActive, this.hdr.VActive];
    const lineLen = divRoundUp(width * 20, 8);
    const lineStride = (lineLen + 63) & -64;
    const frameStride = lineStride * height;
    const eavSkip = this.hdr.Blanking
      ? ((hactive) => {
          if (hactive > 2048) return 8;
          if (hactive < 1024) return 2;
          return 4;
        })(this.hdr.HActive)
      : 0;
    return { width, height, lineLen, lineStride, frameStride, eavSkip };
  }
  frame(framenumber, opts) {
    if (framenumber >= this.hdr.Frames)
      throw new Error(`Out of bounds: this bid file only contains ${this.hdr.Frames} frames`);
    const { lineStride, frameStride, eavSkip } = this.bid_geometry();
    const start = this.hdr_end + frameStride * framenumber;
    const vpos = new VPositions(this.hdr);
    const is_4k = is_higher_than_3G(this.hdr.Standard);
    const interlaced = this.hdr.Interlace;
    let meta;
    if (this.hdr.Blanking) {
      meta = [...Array(is_4k ? 8 : 2)].map(() => []);
      for (let y = 0; y < this.hdr.VTotal; ++y) {
        const bwidth = vpos.in_vblanking(y) ? this.hdr.HTotal : this.hdr.HTotal - this.hdr.HActive;
        const srcoff = start + y * lineStride;
        if (is_4k) {
          const dec = [0, 1, 2, 3].map(
            (_, i) => new ST291Decoder((y >> 1) + 1, 8, meta[i + (y & 1) * 4]),
          );
          for (let x = 0; x < bwidth >> 2; ++x) {
            {
              const b0 = this.buffer.readUInt8(srcoff + x * 10 + 0);
              const b1 = this.buffer.readUInt8(srcoff + x * 10 + 1);
              const b2 = this.buffer.readUInt8(srcoff + x * 10 + 2);
              const b3 = this.buffer.readUInt8(srcoff + x * 10 + 3);
              const b4 = this.buffer.readUInt8(srcoff + x * 10 + 4);
              dec[0].push((b1 * 256 + b0) & 0x3ff);
              dec[1].push(((b2 * 256 + b1) >> 2) & 0x3ff);
              dec[0].push(((b3 * 256 + b2) >> 4) & 0x3ff);
              dec[1].push((b4 * 256 + b3) >> 6);
            }
            {
              const b0 = this.buffer.readUInt8(srcoff + x * 10 + 5);
              const b1 = this.buffer.readUInt8(srcoff + x * 10 + 6);
              const b2 = this.buffer.readUInt8(srcoff + x * 10 + 7);
              const b3 = this.buffer.readUInt8(srcoff + x * 10 + 8);
              const b4 = this.buffer.readUInt8(srcoff + x * 10 + 9);
              dec[2].push((b1 * 256 + b0) & 0x3ff);
              dec[3].push(((b2 * 256 + b1) >> 2) & 0x3ff);
              dec[2].push(((b3 * 256 + b2) >> 4) & 0x3ff);
              dec[3].push((b4 * 256 + b3) >> 6);
            }
          }
        } else {
          const dec = [0, 1].map((_, i) => new ST291Decoder(y + 1, 4, meta[i]));
          for (let x = 0; x < bwidth >> 1; ++x) {
            const b0 = this.buffer.readUInt8(srcoff + x * 5 + 0);
            const b1 = this.buffer.readUInt8(srcoff + x * 5 + 1);
            const b2 = this.buffer.readUInt8(srcoff + x * 5 + 2);
            const b3 = this.buffer.readUInt8(srcoff + x * 5 + 3);
            const b4 = this.buffer.readUInt8(srcoff + x * 5 + 4);
            dec[0].push((b1 * 256 + b0) & 0x3ff);
            dec[1].push(((b2 * 256 + b1) >> 2) & 0x3ff);
            dec[0].push(((b3 * 256 + b2) >> 4) & 0x3ff);
            dec[1].push((b4 * 256 + b3) >> 6);
          }
        }
      }
    }
    const img_with_blanking = this.hdr.Blanking && !(opts?.strip_blanking ?? true);
    const img_hstart =
      img_with_blanking || !this.hdr.Blanking ? 0 : this.hdr.HTotal - this.hdr.HActive - eavSkip;
    const frame = new BIDFrame({
      width: img_with_blanking ? this.hdr.HTotal - eavSkip : this.hdr.HActive,
      height: img_with_blanking ? this.hdr.VTotal : this.hdr.VActive,
      sampling: "422",
      meta,
    });
    for (let y = 0; y < frame.pixels.height; ++y) {
      const y_src = img_with_blanking
        ? y
        : interlaced
          ? vpos.vstart[y & 1] + (y >> 1)
          : vpos.vstart[0] + y;
      const srcoff = start + y_src * lineStride + (img_hstart >> 1) * 5;
      const dstoff = frame.pixels.width * y;
      for (let x = 0; x < frame.pixels.width / 2; ++x) {
        const b0 = this.buffer.readUInt8(srcoff + x * 5 + 0);
        const b1 = this.buffer.readUInt8(srcoff + x * 5 + 1);
        const b2 = this.buffer.readUInt8(srcoff + x * 5 + 2);
        const b3 = this.buffer.readUInt8(srcoff + x * 5 + 3);
        const b4 = this.buffer.readUInt8(srcoff + x * 5 + 4);
        frame.pixels.cb[dstoff / 2 + x] = ((b1 * 256 + b0) & 0x3ff) - 512;
        frame.pixels.y[dstoff + x * 2 + 0] = (((b2 * 256 + b1) >> 2) & 0x3ff) - 64;
        frame.pixels.cr[dstoff / 2 + x] = (((b3 * 256 + b2) >> 4) & 0x3ff) - 512;
        frame.pixels.y[dstoff + x * 2 + 1] = ((b4 * 256 + b3) >> 6) - 64;
      }
    }
    return frame;
  }
  *frames(opts) {
    for (let fn = 0; fn < this.hdr.Frames; ++fn) yield this.frame(fn, opts);
  }
  concat(other) {
    if (
      this.hdr.Blanking !== other.hdr.Blanking ||
      this.hdr.HActive !== other.hdr.HActive ||
      this.hdr.HTotal !== other.hdr.HTotal ||
      this.hdr.Interlace !== other.hdr.Interlace ||
      this.hdr.Standard !== other.hdr.Standard ||
      this.hdr.VActive !== other.hdr.VActive ||
      this.hdr.VTotal !== other.hdr.VTotal
    )
      throw new Error(
        `Unable to concatenate incompatible BID objects: this object's header is ${JSON.stringify(this.hdr)}, the other's is ${JSON.stringify(other.hdr)}`,
      );
    const patched_hdr = Buffer.from(
      JSON.stringify({ ...this.hdr, Frames: this.hdr.Frames + other.hdr.Frames }) + "\0",
      "utf8",
    );
    const padding = Buffer.alloc((8 - (patched_hdr.length % 8)) % 8).fill(0);
    enforce(
      this.raw_data().length % this.bid_geometry().frameStride === 0,
      `raw_data().length is ${this.raw_data().length}, frameStride is ${this.bid_geometry().frameStride}`,
    );
    return new BID(Buffer.concat([patched_hdr, padding, this.raw_data(), other.raw_data()]));
  }
  static from_frame(frame, opts) {
    const standard =
      opts?.standard ?? guess_standard({ width: frame.pixels.width, height: frame.pixels.height });
    if (!standard) throw new Error(`Unable to guess standard, please specify one explicitly`);
    const d = opts?.date ?? new Date();
    const hdr = {
      Date: `${d.getDate()}/${d.getMonth() + 1}/${d.getFullYear()}`,
      Time: `${d.getHours()}:${d.getMinutes()}:${d.getSeconds()}`,
      Interlace: standard_info(standard).interlaced,
      Blanking: false,
      Hostname: "e2etests",
      Frames: 1,
      HTotal: standard_info(standard).width,
      VTotal: standard_info(standard).height,
      HActive: frame.pixels.width,
      VActive: frame.pixels.height,
      Standard: standard,
    };
    const json_string = JSON.stringify(hdr);
    const json_terminator_padded = 8 - (json_string.length % 8);
    const line_len = (frame.pixels.width / 2) * 5;
    const line_stride = (line_len + 63) & -64;
    const total_height = frame.pixels.height;
    enforce(json_terminator_padded > 0 && (json_terminator_padded + json_string.length) % 8 === 0);
    const buf = Buffer.alloc(
      json_string.length + json_terminator_padded + line_stride * total_height,
    );
    buf.fill(json_string, 0, json_string.length);
    buf.fill(0, json_string.length, json_string.length + json_terminator_padded);
    const cbcrshift = frame.pixels.sampling === "422" ? 1 : 0;
    for (let y = 0; y < total_height; ++y) {
      const src_off = y * frame.pixels.width;
      const y_dst = hdr.Interlace ? (y & 1) * (frame.pixels.height >> 1) + (y >> 1) : y;
      let dst_off = json_string.length + json_terminator_padded + y_dst * line_stride;
      for (let x = 0; x < frame.pixels.width; x += 2) {
        const y0 = frame.pixels.y[src_off + x] + 64;
        const y1 = frame.pixels.y[src_off + x + 1] + 64;
        const i_cbcr = (src_off + x) >>> cbcrshift;
        const cb = frame.pixels.cb[i_cbcr] + 512;
        const cr = frame.pixels.cr[i_cbcr] + 512;
        buf.writeUInt8(cb & 0xff, dst_off);
        buf.writeUInt8(((cb >> 8) & 0x03) | ((y0 << 2) & 0xfc), dst_off + 1);
        buf.writeUInt8(((y0 >> 6) & 0x0f) | ((cr << 4) & 0xf0), dst_off + 2);
        buf.writeUInt8(((cr >> 4) & 0x3f) | ((y1 << 6) & 0xc0), dst_off + 3);
        buf.writeUInt8((y1 >> 2) & 0xff, dst_off + 4);
        dst_off += 5;
      }
    }
    return new BID(buf);
  }
  static from_images(images, opts) {
    if (images.length === 0) throw new Error(`Please provide a nonempty list of images`);
    const [width, height] = [images[0].bitmap.width, images[0].bitmap.height];
    for (let i = 1; i < images.length; ++i) {
      if (images[i].bitmap.width !== width || images[i].bitmap.height !== height) {
        throw new Error(
          `Image #${i} has a different size (${images[i].bitmap.width}x${images[i].bitmap.height}) than image #0 (${width}x${height}). Please convert all images to a common size and try again`,
        );
      }
    }
    const standard = opts?.standard ?? guess_standard({ width, height });
    const si = standard_info(standard);
    if (width !== si.width || height !== si.height) {
      throw new Error(
        `Your supplied images' dimensions of ${width}x${height} pixels do not match the configured standard ${standard}`,
      );
    }
    let fill = null;
    let alpha = null;
    const is_sd = standard == "PAL" || standard == "NTSC";
    const colorspace = opts?.colorspace ?? (is_sd ? "BT601" : "BT709");
    for (const image of images) {
      if (opts?.context?.aborted) break;
      const next_fill = BID.from_frame(BIDFrame.from_image(image, { colorspace }), opts);
      if (!fill) fill = next_fill;
      else fill = fill.concat(next_fill);
      const alpha_img = new Jimp(image.getWidth(), image.getHeight());
      for (let i_pixel = 0; i_pixel < image.getWidth() * image.getHeight(); ++i_pixel) {
        const a = image.bitmap.data[4 * i_pixel + 3];
        alpha_img.bitmap.data[4 * i_pixel] = a;
        alpha_img.bitmap.data[4 * i_pixel + 1] = a;
        alpha_img.bitmap.data[4 * i_pixel + 2] = a;
      }
      const next_alpha = BID.from_frame(BIDFrame.from_image(alpha_img), opts);
      if (!alpha) alpha = next_alpha;
      else alpha = alpha.concat(next_alpha);
    }
    return { fill: enforce_nonnull(fill), alpha: enforce_nonnull(alpha) };
  }
  static async download_from(delay, opts) {
    const index = enforce_nonnull(path_index(delay.raw.kwl));
    const store =
      opts?.frame_index !== undefined
        ? `frame&frame_index=${opts?.frame_index}`
        : "clip_single_file";
    await delay.dma.http.setup_handler.write("Stop");
    const result = new BID(
      await download_to_buffer(
        `https://${delay.raw.backing_store.ip}/replay/video?action=read&handler=${index}&store=${store}`,
      ),
    );
    await delay.dma.http.setup_handler.write("Restart");
    return result;
  }
  async upload_to(player) {
    await player.output.control.stop.write("Click", {
      retry_until: {
        criterion: "custom",
        validator: async () => (await player.output.control.motion_status.read()) === "Stop",
      },
    });
    await player.capabilities.command.write({
      input_caliber: {
        add_blanking: this.hdr.Blanking,
        constraints: { variant: "Standard", value: { standard: this.hdr.Standard } },
      },
      capacity: { variant: "Frames", value: { frames: this.hdr.Frames } },
    });
    const index = enforce_nonnull(path_index(player.raw.kwl));
    const vsocket = player.raw.backing_store;
    const url = `http${vsocket.protocol === "wss" ? "s" : ""}://${vsocket.ip}/replay/video?action=write&handler=${index}&store=clip_single_file`;
    const res = await upload_from_buffer(url, this.buffer);
    if (res !== this.buffer.length) throw new Error(`Upload to ${url} failed`);
    await player.restart.write("Click", {
      retry_until: {
        criterion: "custom",
        validator: async () => await player.output.active.read(),
      },
    });
  }
  static async from_inkscape(pars) {
    if ((await sh(`which inkscape`, { fail_on_error: false })).err) {
      throw new Error(
        `Apparently inkscape has not been installed, or is not to be found within your PATH. Please install it and try again`,
      );
    }
    const images = [];
    let tmp_dir;
    try {
      tmp_dir = mkdtempSync(path.join(pars.tmp_dir ?? os.tmpdir(), "vutil-graphics"));
      let i = 0;
      for (const svg_filename of pars.svg_filenames) {
        const img_filename = path.join(tmp_dir, `${i++}.png`);
        await sh(
          `inkscape --actions="export-filename: ${img_filename}; export-do;" ${svg_filename}`,
        );
        if (pars.context?.aborted) break;
        images.push(await jimp_read(img_filename));
      }
    } finally {
      if (tmp_dir) rmSync(tmp_dir, { recursive: true });
    }
    return BID.from_images(images, pars);
  }
  static async draw_with_inkscape(pars) {
    let tmp_dir;
    let bid;
    try {
      tmp_dir = fs.mkdtempSync(path.join(os.tmpdir(), "vutil"));
      const filenames = [];
      const std_info = standard_info(pars.standard);
      const [width, height] = [std_info.width, std_info.height];
      for (let frame_index = 0; frame_index < pars.num_frames; ++frame_index) {
        const pseudowindow = createSVGWindow();
        registerWindow(pseudowindow, pseudowindow.document);
        const svg = SVG().size(width, height);
        await pars.f({ frame_index, num_frames: pars.num_frames, svg, width, height });
        const filename = `frame_${frame_index}.svg`;
        fs.writeFileSync(filename, svg.svg());
        filenames.push(filename);
      }
      bid = await BID.from_inkscape({
        svg_filenames: filenames,
        standard: pars.standard,
      });
    } finally {
      if (tmp_dir) {
        fs.rmSync(tmp_dir, { recursive: true });
      }
    }
    return enforce_nonnull(bid);
  }
}
class VPositions {
  vstart;
  vlen;
  constructor(hdr) {
    if (hdr.Blanking) {
      if (hdr.Interlace) {
        const vstart0 = vstart(hdr.Standard, "first");
        const vstart1 = vstart(hdr.Standard, "second");
        const vend0 = vend(hdr.Standard, "first");
        const vlen0 = vend0 - vstart0;
        const vblank0 = vstart1 - vend0;
        this.vlen = [vlen0, hdr.VActive - vlen0];
        this.vstart = [hdr.VTotal - (vblank0 + hdr.VActive), hdr.VTotal - this.vlen[1]];
      } else {
        this.vlen = [hdr.VActive, 0];
        this.vstart = [hdr.VTotal - this.vlen[0], 0];
      }
    } else if (hdr.Interlace) {
      const f0len = (hdr.VActive + 1) >> 1;
      this.vstart = [0, f0len];
      this.vlen = [f0len, hdr.VActive - f0len];
    } else {
      this.vstart = [0, 0];
      this.vlen = [hdr.VActive, 0];
    }
  }
  in_vblanking(y) {
    for (let i = 0; i < 2; ++i) {
      if (y >= this.vstart[i] && y - this.vstart[i] < this.vlen[i]) return true;
    }
    return false;
  }
}
function divRoundUp(n, d) {
  return Math.ceil(n / d);
}
export function round_clamp(v, min, max) {
  const r = Math.round(v);
  if (r < min) return min;
  if (r > max) return max;
  return r;
}
export function invert_3x3(matrix) {
  const m = (i) => enforce_nonnull(matrix[i]);
  const det =
    -m(2) * m(4) * m(6) +
    m(1) * m(5) * m(6) +
    m(2) * m(3) * m(7) -
    m(0) * m(5) * m(7) -
    m(1) * m(3) * m(8) +
    m(0) * m(4) * m(8);
  return [
    m(4) * m(8) - m(5) * m(7),
    m(2) * m(7) - m(1) * m(8),
    -m(2) * m(4) + m(1) * m(5),
    m(5) * m(6) - m(3) * m(8),
    -m(2) * m(6) + m(0) * m(8),
    m(2) * m(3) - m(0) * m(5),
    -m(4) * m(6) + m(3) * m(7),
    m(1) * m(6) - m(0) * m(7),
    -m(1) * m(3) + m(0) * m(4),
  ].map((x) => x / det);
}
export function conversion_matrix(cs) {
  switch (cs) {
    case "BT601":
      return [
        +0.255785137, +0.502160192, +0.097523436, -0.147643909, -0.289856106, +0.4375, +0.43749997,
        -0.366351664, -0.071148358,
      ];
    case "BT709":
      return [
        +0.181787118, +0.612002313, +0.061679296, -0.100192644, -0.337307364, +0.4375, +0.4375,
        -0.397444427, -0.040055554,
      ];
    case "BT2020":
    case "BT2100":
      return [
        +0.224732, +0.58001, +0.0507278, -0.122176, -0.315325, +0.4375, +0.4375, -0.402314,
        -0.0351865,
      ];
    default:
      throw Error("Unsupported color space");
  }
}
export class UnpackedBID {
  hdr;
  frames;
  height;
  frameStride;
  lineLen;
  lineStride;
  samplesPerLine;
  constructor(bid) {
    const { height, frameStride, lineStride, lineLen } = bid.bid_geometry();
    this.hdr = bid.hdr;
    this.height = height;
    this.frameStride = frameStride;
    this.lineStride = lineStride;
    this.lineLen = lineLen;
    if (lineLen % 5 !== 0) throw new Error(`Unsupported line length: must be a multiple of 5`);
    this.samplesPerLine = (lineLen * 4) / 5;
    this.frames = [];
    const raw = bid.raw_data();
    for (let frame_index = 0; frame_index < bid.hdr.Frames; ++frame_index) {
      const buf = new Uint16Array((lineLen * height * 4) / 5);
      const start = frameStride * frame_index;
      let i = 0;
      for (let y = 0; y < height; ++y) {
        const srcoff = start + y * lineStride;
        for (let x = 0; x < lineLen / 5; ++x) {
          const i0 = srcoff + 5 * x;
          const b0 = raw[i0 + 0];
          const b1 = raw[i0 + 1];
          const b2 = raw[i0 + 2];
          const b3 = raw[i0 + 3];
          const b4 = raw[i0 + 4];
          buf[i++] = (b1 * 256 + b0) & 0x3ff;
          buf[i++] = ((b2 * 256 + b1) >> 2) & 0x3ff;
          buf[i++] = ((b3 * 256 + b2) >> 4) & 0x3ff;
          buf[i++] = (b4 * 256 + b3) >> 6;
        }
      }
      this.frames.push(buf);
    }
  }
  static from_bid(bid) {
    return new UnpackedBID(bid);
  }
  enforce_blanking() {
    if (!this.hdr.Blanking)
      throw new Error(`This UnpackedBID object does not include blanking information`);
  }
  scrub_blanking() {
    this.enforce_blanking();
    const vpos = new VPositions(this.hdr);
    for (const frame of this.frames) {
      for (let y = 0; y < this.height; ++y) {
        const hend = vpos.in_vblanking(y)
          ? (y + 1) * this.samplesPerLine
          : y * this.samplesPerLine + this.hdr.HTotal - this.hdr.HActive;
        const blanking = frame.subarray(y * this.samplesPerLine, hend);
        enforce(blanking.length % 2 === 0);
        for (let i = 0; i < blanking.length / 2; ++i) {
          blanking[2 * i] = 0x0200;
          blanking[2 * i + 1] = 0x0040;
        }
      }
    }
    this.regenerate_edh();
  }
  sabotage_edh() {
    this.enforce_blanking();
  }
  regenerate_edh() {
    this.enforce_blanking();
    enforce(false, "not implemented");
  }
  to_bid() {
    const hdr_string = JSON.stringify(this.hdr) + "\0";
    const hdr_end = hdr_string.length + (8 - (hdr_string.length % 8));
    const buf = Buffer.alloc(hdr_end + this.frameStride * this.frames.length);
    buf.write(hdr_string);
    const raw = buf.subarray(hdr_end);
    enforce(this.lineLen % 5 === 0);
    for (const [frame_index, frame] of enumerate(this.frames)) {
      enforce(frame.length % 4 === 0);
      const start = this.frameStride * frame_index;
      let dst_offset = start;
      let src_offset = 0;
      for (let y = 0; y < this.height; ++y) {
        dst_offset = start + this.lineStride * y;
        for (let x = 0; x < this.lineLen / 5; ++x) {
          const s0 = frame[src_offset];
          const s1 = frame[src_offset + 1];
          const s2 = frame[src_offset + 2];
          const s3 = frame[src_offset + 3];
          src_offset += 4;
          raw[dst_offset++] = s0 & 0xff;
          raw[dst_offset++] = (s0 >> 8) | ((s1 & 0x3f) << 2);
          raw[dst_offset++] = (s1 >> 6) | ((s2 & 0xf) << 4);
          raw[dst_offset++] = (s2 >> 4) | ((s3 & 0x3) << 6);
          raw[dst_offset++] = s3 >> 2;
        }
      }
    }
    return new BID(buf);
  }
}
