import * as fs from "fs";
import * as http from "http";
import * as https from "https";
import * as path from "path";
import { Readable } from "stream";
import { finished } from "stream/promises";
import { Buffer } from "buffer";
export function download_to_buffer(url) {
  return new Promise((resolve, reject) => {
    const cb_get = (res) => {
      if (res.statusCode == 200) {
        let data = [];
        let data_len = 0;
        res
          .on("data", (chunk) => {
            data.push(chunk);
            data_len += chunk.length;
          })
          .on("error", (err) => {
            console.log(`Error downloading from ${url} after ${data_len} bytes. ${err.message}`);
            reject(err);
          })
          .on("end", () => {
            const data_result = Buffer.concat(data);
            data = [];
            resolve(data_result);
          });
      } else {
        reject(new Error(`HTTP status code: ${res.statusCode}`));
      }
    };
    if (url.startsWith("http://")) {
      http.get(url, cb_get);
    } else if (url.startsWith("https://")) {
      https.get(url, { rejectUnauthorized: false }, cb_get);
    } else throw new Error("Unknown protocol. Supported protocols: http, https.");
  });
}
export function download_to_file(url, path_file) {
  if (fs.existsSync(path_file)) {
    throw new Error(`File: <${path_file}> exists.`);
  }
  const dir_name = path.dirname(path_file);
  if (!fs.existsSync(dir_name)) {
    fs.mkdirSync(dir_name, { recursive: true });
  }
  const write_stream = fs.createWriteStream(path_file);
  write_stream.on("error", function (err) {
    throw err;
  });
  return new Promise((resolve, reject) => {
    const cb_get = (res) => {
      if (res.statusCode == 200) {
        res.pipe(write_stream);
        res
          .on("end", function () {
            write_stream.close();
            resolve(path_file);
          })
          .on("error", (err) => {
            reject(err);
          });
      } else {
        reject(new Error(`HTTP status code: ${res.statusCode}`));
      }
    };
    if (url.startsWith("http://")) {
      http.get(url, cb_get);
    } else if (url.startsWith("https://")) {
      https.get(url, { rejectUnauthorized: false }, cb_get);
    } else throw new Error("Unknown protocol. Supported protocols: http, https.");
  });
}
export function upload_from_file(url, path_file) {
  const stat_file = fs.statSync(path_file);
  if (!stat_file.isFile()) {
    throw new Error(`The path <${path_file} doesn't point to a file.`);
  }
  return new Promise((resolve, reject) => {
    const req = upload_handler(url, stat_file.size, resolve, reject);
    const readable = fs.createReadStream(path_file);
    readable
      .on("open", () => {
        readable.pipe(req);
      })
      .on("error", (err) => {
        reject(err);
      })
      .on("end", () => {
        req.end();
      });
  });
}
export function upload_from_buffer(url, data) {
  return new Promise((resolve, reject) => {
    upload_handler(url, data.length, resolve, reject).end(data);
  });
}
function upload_handler(url, data_size, resolve, reject) {
  const http_options = {
    method: "PUT",
    headers: {
      "Content-Type": "application/octet-stream",
      "Content-Length": data_size,
    },
  };
  const cb_put = (resp) => {
    if (resp.statusCode == 200) {
      resolve(data_size);
    } else {
      reject(new Error(`HTTP status code: ${resp.statusCode}`));
    }
  };
  let req;
  if (url.startsWith("http://")) {
    req = http.request(url, http_options, cb_put);
  } else if (url.startsWith("https://")) {
    req = https.request(url, { ...http_options, rejectUnauthorized: false }, cb_put);
  } else throw new Error("Unknown protocol. Supported protocols: http, https.");
  req.on("error", (err) => reject(err));
  return req;
}
export async function fetch_file(pars) {
  const rsp = await fetch(pars.url);
  const dir_name = path.dirname(pars.filename);
  fs.mkdirSync(dir_name, { recursive: true });
  if (!rsp.body) throw new Error(`Unable to download ${pars.url}`);
  const stream = rsp.body;
  switch (pars.if_file_exists) {
    case "overwrite":
      fs.rmSync(pars.filename, { force: true });
      break;
    case "fail":
      if (fs.existsSync(pars.filename))
        throw new Error(
          `Download failed, because ${pars.filename} already exists. Please delete it and try again, or set if_file_exists to "overwrite"`,
        );
      break;
  }
  const write_stream = fs.createWriteStream(pars.filename, { flags: "wx" });
  write_stream.on("error", function (err) {
    throw err;
  });
  await finished(Readable.fromWeb(stream).pipe(write_stream));
}
