import * as dgram from "dgram";
import { networkInterfaces } from "os";
import { enforce, enforce_nonnull } from "vscript";
export class Timestamper {
  mWatcher;
  mPrevData;
  constructor(master) {
    this.initialize(master);
  }
  close() {
    if (this.mWatcher) {
      this.mWatcher.unwatch();
      this.mWatcher = undefined;
    }
    this.mPrevData = undefined;
  }
  estimate_timestamp() {
    if (!this.mPrevData) {
      return null;
    }
    const msSince = new Date().valueOf() - this.mPrevData[0];
    return (Math.round(this.mPrevData[1].seconds() + msSince * 1e-3) * 90000) % Math.pow(2, 32);
  }
  get_time() {
    if (!this.mPrevData) {
      return null;
    }
    return this.mPrevData[1].seconds() + (new Date().valueOf() - this.mPrevData[0]) * 1e-3;
  }
  async initialize(master) {
    this.mWatcher = await master.p_t_p_clock.micro_epochs.watch((microEpochs) => {
      this.mPrevData = [new Date().valueOf(), microEpochs.current.reference_time];
    });
  }
}
export class MetadataSender {
  socket;
  dstAddress;
  frameRate;
  payload;
  mContents;
  port;
  timestamper;
  timestampingMode;
  timestampOffset;
  seqnr;
  srcAddress;
  interlaced;
  static s_sockets = new Map();
  static get_socket(address, port) {
    if (!MetadataSender.s_sockets.has(port)) {
      MetadataSender.s_sockets.set(port, new Map());
    }
    const port_map = MetadataSender.s_sockets.get(port);
    if (!port_map.has(address)) {
      port_map.set(
        address,
        new Promise((resolve, reject) => {
          const socket = dgram.createSocket("udp4");
          const sentinel = setTimeout(
            () =>
              reject(
                `Failed to open Metadata sender from src address ${address}, on UDP port ${port}`,
              ),
            1000,
          );
          socket.bind({ address, port }, () => {
            socket.setMulticastTTL(128);
            clearTimeout(sentinel);
            resolve(socket);
          });
        }),
      );
    }
    return port_map.get(address);
  }
  static async open({
    dstAddress,
    port,
    frameRate,
    interlaced,
    payload,
    contents,
    timestampingMode,
    timestamper,
    timestampOffset,
    seqnr,
    srcAddress,
  }) {
    const find_src_address = () => {
      const ifcs = networkInterfaces();
      for (const ifname of Object.keys(ifcs)) {
        for (const virtIfc of ifcs[ifname]) {
          if (virtIfc.family !== "IPv4" || virtIfc.internal === true) {
            continue;
          }
          return virtIfc.address;
        }
      }
      return undefined;
    };
    const address = enforce_nonnull(srcAddress ?? find_src_address());
    const socket = await this.get_socket(address, port);
    return new MetadataSender(
      socket,
      dstAddress,
      frameRate,
      payload,
      contents,
      port,
      timestamper,
      timestampingMode,
      Math.round((timestampOffset ?? 0) * 90000) % Math.pow(2, 32),
      seqnr ?? 0,
      address,
      interlaced,
    );
  }
  static popcount(u32) {
    enforce((u32 >> 31) >> 1 === 0);
    u32 = u32 - ((u32 >> 1) & 0x55555555);
    u32 = (u32 & 0x33333333) + ((u32 >> 2) & 0x33333333);
    return (((u32 + (u32 >> 4)) & 0xf0f0f0f) * 0x1010101) >> 24;
  }
  static u8to10(u8) {
    enforce(u8 >> 8 === 0);
    if (MetadataSender.popcount(u8) % 2 === 0) {
      return 0x200 | u8;
    } else {
      return 0x100 | u8;
    }
  }
  static insert(buf, bitsOffset, bitWidth, x) {
    enforce(bitWidth <= 16);
    const lastBit = bitsOffset + bitWidth - 1;
    const i = lastBit >> 3;
    enforce(buf.length > i);
    let y = (buf[i - 2] << 16) | (buf[i - 1] << 8) | buf[i];
    const leftOffset = bitsOffset - 8 * (i - 2);
    const mask = ((1 << (24 - leftOffset)) - 1) ^ ((1 << (24 - leftOffset - bitWidth)) - 1);
    y = (y & ~mask) | (x << (24 - (leftOffset + bitWidth)));
    buf[i - 2] = y >> 16;
    buf[i - 1] = (y >> 8) & 0xff;
    buf[i] = y & 0xff;
  }
  static fillMarkerBuffer(buf, pars) {
    buf[0] = 0x80;
    enforce(pars.payload <= 127);
    buf[1] = 0x80 | pars.payload;
    enforce(pars.extendedSeqnr >> 16 === 0);
    buf[12] = pars.extendedSeqnr >> 8;
    buf[13] = pars.extendedSeqnr & 0xff;
    if (pars.field !== undefined) {
      buf[17] = (pars.field + 2) << 6;
    }
  }
  static fillAFDBuffer(buf, pars) {
    buf[0] = 0x80;
    buf[1] = pars.payload;
    buf[12] = pars.extendedSeqnr >> 8;
    buf[13] = pars.extendedSeqnr & 0xff;
    const LENGTH = 20;
    const ANC_COUNT = 1;
    buf[15] = LENGTH;
    buf[16] = ANC_COUNT;
    if (pars.field !== undefined) {
      buf[17] = (pars.field + 2) << 6;
    }
    let offset = 161;
    const doInsert = (bitWidth, x) => {
      MetadataSender.insert(buf, offset, bitWidth, x);
      offset += bitWidth;
    };
    doInsert(11, pars.lineNumber);
    doInsert(12, pars.horizontalOffset);
    const STREAM_NUM = null;
    if (STREAM_NUM === null) {
      doInsert(8, 0);
    } else {
      doInsert(8, 0x80 | STREAM_NUM);
    }
    let checksum = 0;
    const doInsertWithChecksum = (u10) => {
      checksum += u10;
      doInsert(10, u10);
    };
    const DATA_COUNT = 8;
    doInsertWithChecksum(MetadataSender.u8to10(pars.did));
    doInsertWithChecksum(MetadataSender.u8to10(pars.sdid));
    doInsertWithChecksum(MetadataSender.u8to10(DATA_COUNT));
    doInsertWithChecksum(
      MetadataSender.u8to10((pars.code << 3) | (pars.ar === "AR_16_9" ? 0b100 : 0b000)),
    );
    doInsertWithChecksum(MetadataSender.u8to10(0));
    doInsertWithChecksum(MetadataSender.u8to10(0));
    const presenceFlag = (maybeValue, position) => {
      if (maybeValue === undefined) {
        return 0;
      }
      return 1 << position;
    };
    const formatLineOrPixelNumber = (lineOrPixelNumber) => {
      if (lineOrPixelNumber === undefined) {
        return 0;
      }
      return (3 << 14) | lineOrPixelNumber;
    };
    let flags = 0;
    let word0 = 0;
    let word1 = 0;
    if ("left" in pars || "right" in pars) {
      flags = presenceFlag(pars.right, 4) | presenceFlag(pars.left, 5);
      word0 = formatLineOrPixelNumber(pars.left);
      word1 = formatLineOrPixelNumber(pars.right);
    } else if ("top" in pars || "bottom" in pars) {
      flags = presenceFlag(pars.bottom, 6) | presenceFlag(pars.top, 7);
      word0 = formatLineOrPixelNumber(pars.top);
      word1 = formatLineOrPixelNumber(pars.bottom);
    }
    doInsertWithChecksum(MetadataSender.u8to10(flags));
    doInsertWithChecksum(MetadataSender.u8to10(word0 >> 8));
    doInsertWithChecksum(MetadataSender.u8to10(word0 & 0xff));
    doInsertWithChecksum(MetadataSender.u8to10(word1 >> 8));
    doInsertWithChecksum(MetadataSender.u8to10(word1 & 0xff));
    checksum &= 0x1ff;
    checksum |= (~(checksum >> 8) & 1) << 9;
    doInsert(10, checksum);
  }
  nominalFramesPerSecond;
  t0 = null;
  packetsSent = 0;
  interval;
  markerData = new Uint8Array(28);
  contentData = new Uint8Array(40);
  field = 0;
  mPrevTimestamp = null;
  mRTPCountsPerFieldOrFrame;
  constructor(
    socket,
    dstAddress,
    frameRate,
    payload,
    mContents,
    port,
    timestamper,
    timestampingMode,
    timestampOffset,
    seqnr,
    srcAddress,
    interlaced,
  ) {
    this.socket = socket;
    this.dstAddress = dstAddress;
    this.frameRate = frameRate;
    this.payload = payload;
    this.mContents = mContents;
    this.port = port;
    this.timestamper = timestamper;
    this.timestampingMode = timestampingMode;
    this.timestampOffset = timestampOffset;
    this.seqnr = seqnr;
    this.srcAddress = srcAddress;
    this.interlaced = interlaced;
    switch (frameRate) {
      case "f50":
        this.nominalFramesPerSecond = [50, 1];
        break;
      case "f24":
        this.nominalFramesPerSecond = [24, 1];
        break;
      case "f25":
        this.nominalFramesPerSecond = [25, 1];
        break;
      case "f30":
        this.nominalFramesPerSecond = [30, 1];
        break;
      case "f60":
        this.nominalFramesPerSecond = [60, 1];
        break;
      case "f23_98":
        this.nominalFramesPerSecond = [24 * 1000, 1001];
        break;
      case "f29_97":
        this.nominalFramesPerSecond = [30 * 1000, 1001];
        break;
      case "f59_94":
        this.nominalFramesPerSecond = [60 * 1000, 1001];
        break;
    }
    this.mRTPCountsPerFieldOrFrame =
      (90000 * this.nominalFramesPerSecond[1] * (this.interlaced ? 2 : 1)) /
      this.nominalFramesPerSecond[0];
  }
  close() {
    this.socket.close();
    if (this.interval) {
      clearInterval(this.interval);
      this.interval = undefined;
    }
  }
  media_description(mid) {
    return `m=video ${this.port} RTP/AVP ${this.payload}
c=IN IP4 ${this.dstAddress}/64
a=source-filter:incl IN IP4 ${this.dstAddress} ${this.srcAddress}
a=rtpmap:${this.payload} smpte291/90000
a=mid:${mid}`;
  }
  activate() {
    enforce(this.t0 === null);
    this.interval = setInterval(() => {
      this.tick();
    }, 10);
  }
  deactivate() {
    if (this.interval) {
      clearInterval(this.interval);
      this.interval = null;
      this.t0 = null;
      this.packetsSent = 0;
    } else {
      enforce(this.t0 === null);
    }
  }
  shiftSeqnrBy(delta) {
    enforce(Math.round(delta) === delta);
    this.seqnr = (this.seqnr + Math.round(delta)) & 0xffff;
  }
  setContents(contents) {
    this.mContents = contents;
  }
  afdData() {
    return { ...this.mContents };
  }
  async sendPacket(data) {
    return new Promise((resolve, reject) =>
      this.socket.send(data, this.port, this.dstAddress, (maybeError) => {
        if (maybeError) {
          reject(maybeError);
        }
        resolve();
      }),
    );
  }
  updateBuffer(buffer, timestamp) {
    buffer[2] = this.seqnr >> 8;
    buffer[3] = this.seqnr & 0xff;
    buffer[4] = (timestamp >> 24) & 0xff;
    buffer[5] = (timestamp >> 16) & 0xff;
    buffer[6] = (timestamp >> 8) & 0xff;
    buffer[7] = timestamp & 0xff;
  }
  async sendFrame(timestamp) {
    this.field = this.field === 0 ? 1 : 0;
    const fieldInfo = this.interlaced ? { field: this.field } : {};
    const extendedSeqnr = 0x0000;
    MetadataSender.fillMarkerBuffer(this.markerData, {
      extendedSeqnr,
      payload: this.payload,
      ...fieldInfo,
    });
    if ("left" in this.mContents) {
      this.mContents.left = (this.mContents.left + (this.mContents.leftStep ?? 0)) & 0x3fff;
    }
    if ("right" in this.mContents) {
      this.mContents.right = (this.mContents.right + (this.mContents.rightStep ?? 0)) & 0x3fff;
    }
    if ("top" in this.mContents) {
      this.mContents.top = (this.mContents.top + (this.mContents.topStep ?? 0)) & 0x3fff;
    }
    if ("bottom" in this.mContents) {
      this.mContents.bottom = (this.mContents.bottom + (this.mContents.bottomStep ?? 0)) & 0x3fff;
    }
    MetadataSender.fillAFDBuffer(this.contentData, {
      did: 0x41,
      extendedSeqnr,
      horizontalOffset: 0,
      lineNumber: 9,
      payload: this.payload,
      sdid: 0x05,
      ...fieldInfo,
      ...this.mContents,
    });
    this.updateBuffer(this.contentData, timestamp);
    this.seqnr = (this.seqnr + 1) % 65536;
    this.updateBuffer(this.markerData, timestamp);
    this.seqnr = (this.seqnr + 1) % 65536;
    await this.sendPacket(this.contentData);
    await this.sendPacket(this.markerData);
  }
  async tick() {
    if (!this.t0) {
      this.t0 = this.timestamper.get_time();
      if (this.t0 === null) {
        return;
      }
    }
    const t1 = this.timestamper.get_time();
    if (t1 === null) {
      return;
    }
    const nominalMPacketsPerSecond =
      (this.nominalFramesPerSecond[0] / this.nominalFramesPerSecond[1]) * (this.interlaced ? 2 : 1);
    const packetsExpected = Math.round((t1 - this.t0) * nominalMPacketsPerSecond);
    if (packetsExpected <= this.packetsSent) {
      return;
    }
    const timestamp =
      this.mPrevTimestamp === null || this.timestampingMode === "estimate-continuously"
        ? this.timestamper.estimate_timestamp()
        : (this.mPrevTimestamp + this.mRTPCountsPerFieldOrFrame) % Math.pow(2, 32);
    this.mPrevTimestamp = timestamp;
    if (timestamp === null) {
      return;
    }
    this.sendFrame((timestamp + this.timestampOffset) % Math.pow(2, 32));
    this.packetsSent += 1;
    if (packetsExpected > this.packetsSent + 4) {
      this.t0 = null;
      this.packetsSent = 0;
    }
  }
}
export class ST291Decoder {
  linenr;
  column;
  sr;
  cnt;
  pkt;
  res;
  constructor(linenr, column, res) {
    this.linenr = linenr;
    this.column = column;
    this.sr = 0xaaaaaaaa;
    this.cnt = 0;
    this.res = res;
  }
  push(v) {
    if (v < 0 || v > 0x3ff) throw Error("Value out of range");
    this.column++;
    this.sr = ((this.sr & 0xfffff) << 10) | v;
    if (this.sr == 0xfffff) this.cnt = 1;
    else if (this.cnt > 0) {
      if (this.cnt == 3)
        this.pkt = {
          linenr: this.linenr,
          column: this.column - 5,
          did: (this.sr >> 20) & 0x3ff,
          sdid: (this.sr >> 10) & 0x3ff,
          data: new Uint16Array(v & 0xff),
          cs: 0,
        };
      if (this.cnt > 3) {
        if (this.cnt == 4 + this.pkt.data.length) {
          this.pkt.cs = v;
          this.res.push(this.pkt);
          this.pkt = undefined;
          this.cnt = 0;
          return;
        } else this.pkt.data[this.cnt - 4] = v;
      }
      this.cnt++;
    }
  }
}
export class ST291Encoder {
  static serialize_tc(pkt) {
    const udw = new Uint8Array();
    udw[0] = (pkt.frames & 0xf) << 4;
    udw[1] = (pkt.binary_groups[0] & 0xf) << 4;
    udw[2] = ((pkt.frames / 10) & 3) << 4;
    udw[3] = (pkt.binary_groups[1] & 0xf) << 4;
    udw[4] = (pkt.seconds & 0xf) << 4;
    udw[5] = (pkt.binary_groups[2] & 0xf) << 4;
    udw[6] = ((pkt.seconds / 10) & 7) << 4;
    udw[7] = (pkt.binary_groups[3] & 0xf) << 4;
    udw[8] = (pkt.minutes & 0xf) << 4;
    udw[9] = (pkt.binary_groups[4] & 0xf) << 4;
    udw[10] = ((pkt.minutes / 10) & 7) << 4;
    udw[11] = (pkt.binary_groups[5] & 0xf) << 4;
    udw[12] = (pkt.hours & 0xf) << 4;
    udw[13] = (pkt.binary_groups[6] & 0xf) << 4;
    udw[14] = ((pkt.hours / 10) & 3) << 4;
    udw[15] = (pkt.binary_groups[7] & 0xf) << 4;
    const set_bit = (i_udw, i_bit, b) => {
      enforce(typeof b === "boolean" || b === 0 || b === 1);
      const mask = 1 << i_bit;
      const old = udw[i_udw];
      udw[i_udw] = (old & ~mask) | (b ? mask : 0);
    };
    set_bit(14, 3, pkt.tc_validity);
    set_bit(15, 3, pkt.process_bit);
    switch (pkt.tc_type) {
      case "LTC":
        break;
      case "VITC1":
        set_bit(0, 3, 1);
        break;
      case "VITC2":
        set_bit(1, 3, 1);
        break;
    }
    if (pkt.tc_type !== "LTC") {
      for (let i = 0; i < 5; ++i) {
        set_bit(8 + i, 3, ((pkt.line_select ?? 0) >> i) & 1);
      }
      set_bit(13, 3, pkt.line_duplication_flag);
    }
    return {
      did: 0x60,
      sdid: 0x60,
      udw,
    };
  }
  static encode(pars) {
    if (pars.udw.length > 255)
      throw new Error(
        `The provided udw array extends the max anc packet size of 255 user data words`,
      );
    const n_samples_total = 3 + 3 + 1 + pars.udw.length;
    const result = new Uint16Array(n_samples_total);
    result[0] = 0;
    result[1] = 0x3ff;
    result[2] = 0x3ff;
    if (pars.did >= 256) throw new Error(`Got invalid DID ${pars.did}`);
    if (pars.sdid >= 256) throw new Error(`Got invalid DID ${pars.sdid}`);
    let cs = 0;
    const to_291 = (x) => {
      enforce(x < 256);
      let parity = 0;
      let x2 = x;
      while (x2 !== 0) {
        parity ^= x2 & 1;
        x2 >>= 1;
      }
      cs += (parity << 8) | x;
      return (~parity << 9) | (parity << 8) | x;
    };
    result[3] = to_291(pars.did);
    result[4] = to_291(pars.sdid);
    for (let i = 0; i < pars.udw.length; ++i) {
      result[4 + i] = to_291(pars.udw[i]);
    }
    cs &= 0x1ff;
    cs |= (~cs & 0x100) << 1;
    result[4 + pars.udw.length] = cs;
    return result;
  }
}
export function dissect_st291_packet(pkt_291) {
  const udw = (n_byte) => pkt_291.data[n_byte] & 0xff;
  const did = pkt_291.did & 0xff;
  const sdid = pkt_291.sdid & 0xff;
  const bit = (n) => {
    return (x) => 1 & ((x & ((1 << (n + 1)) - 1)) >> n);
  };
  if (did === 0x60 && sdid === 0x60) {
    const dbb1 = bit(3)(udw(0)) | (bit(3)(udw(1)) << 1) | (bit(3)(udw(2)) << 2);
    const frames = ((udw(0) >> 4) & 0xf) + 10 * ((udw(2) >> 4) & 3);
    const seconds = ((udw(4) >> 4) & 0xf) + 10 * ((udw(6) >> 4) & 7);
    const minutes = ((udw(8) >> 4) & 0xf) + 10 * ((udw(10) >> 4) & 7);
    const hours = ((udw(12) >> 4) & 0xf) + 10 * ((udw(14) >> 4) & 3);
    const tc_validity = !!bit(3)(udw(14));
    const process_bit = !!bit(3)(udw(15));
    const binary_groups = [
      udw(1) >> 4,
      udw(3) >> 4,
      udw(5) >> 4,
      udw(7) >> 4,
      udw(9) >> 4,
      udw(11) >> 4,
      udw(13) >> 4,
      udw(15) >> 4,
    ];
    switch (dbb1) {
      case 0:
        return {
          type: "Timecode",
          hours,
          minutes,
          seconds,
          frames,
          raw: pkt_291,
          tc_type: "LTC",
          tc_validity,
          process_bit,
          binary_groups,
        };
      case 1 | 2: {
        let line_select_raw = 0;
        for (let i = 0; i < 5; ++i) {
          line_select_raw |= bit(3)(udw(8 + i)) << i;
        }
        const line_select = line_select_raw === 0 ? null : line_select_raw;
        const line_duplication_flag = !!bit(3)(udw(13));
        return {
          type: "Timecode",
          tc_type: dbb1 === 1 ? "VITC1" : "VITC2",
          hours,
          minutes,
          seconds,
          frames,
          raw: pkt_291,
          tc_validity,
          process_bit,
          line_select,
          line_duplication_flag,
          binary_groups,
        };
      }
      default:
        break;
    }
  }
  return { type: "other", raw: pkt_291 };
}
