import { ru16be, ru32be, ru32le } from "./binary_io.js";
import { Pipe } from "./pipes.js";
import { enforce, enforce_nonnull } from "vscript";
import { Buffer } from "buffer";
const PCAP_HEADER_SIZE = 24;
function parse_pcap_header(buf) {
  if (buf.length !== PCAP_HEADER_SIZE)
    throw new Error(`Illegal argument: pcap file header should have a length of 24 bytes`);
  const magic_number = buf.readUInt32LE(0);
  let endianness = "LE";
  const nsec_multiplier = (() => {
    switch (magic_number) {
      case 0xa1b23c4d:
        return 1;
      case 0x4d3cb2a1:
        endianness = "BE";
        return 1;
      case 0xa1b2c3d4:
        return 1000;
      case 0xd4c3b2a1:
        endianness = "BE";
        return 1000;
      case 0x0a0d0d0a:
        throw new Error(
          `Unsupported magic number 0x${magic_number
            .toString(16)
            .padStart(
              8,
              "0",
            )} (this is most likely a pcapng file, use a dummy tcpdump-filter such as "udp" to have vsk automatically convert this to plain pcap format)`,
        );
      default:
        throw new Error(`Unsupported magic number 0x${magic_number.toString(16).padStart(8, "0")}`);
    }
  })();
  const version_major = buf.readUInt16LE(4);
  const version_minor = buf.readUInt16LE(6);
  const thiszone = buf.readUInt32LE(8);
  const sigfigs = buf.readUInt32LE(12);
  const snaplen = buf.readUInt32LE(16);
  const network = buf.readUInt32LE(20);
  return [
    nsec_multiplier,
    {
      magic_number,
      version_major,
      version_minor,
      thiszone,
      sigfigs,
      snaplen,
      network,
      endianness,
    },
  ];
}
export async function* iter_pcap(pars) {
  const pipe = new Pipe(pars.instream);
  const pipethrough = (cmd, args) => {
    pipe.add({
      cmd,
      args,
      on_stderr: (data) => {
        enforce(data instanceof Buffer);
        pars.log?.(`[${cmd}/stderr] ${data.toString().trim()}\n`, "Info");
      },
      on_close: (code) => {
        pars.log?.(`[${cmd}] exited with code ${code}\n`, "Debug");
      },
    });
  };
  if (pars.preprocess) {
    const parts = pars.preprocess.split(" ");
    const cmd = enforce_nonnull(parts.shift());
    pipethrough(cmd, parts);
  }
  if (pars["tcpdump-filter"]) {
    pipethrough("tcpdump", [
      pars["tcpdump-filter"].trim(),
      "-r",
      "-",
      "-w",
      "-",
      "--time-stamp-precision=nano",
    ]);
  }
  if (pars["display-filter"]) {
    const y = pars["display-filter"].trim();
    pipethrough("tshark", [
      ...(y.length !== 0 ? ["-Y", y] : []),
      "-r",
      "-",
      "-w",
      "-",
      "-F",
      "nsecpcap",
    ]);
  }
  let nsec_multiplier = null;
  const zerobuf = Buffer.alloc(0);
  let rem = zerobuf;
  let frame_number = 0;
  let endianness = "LE";
  let ru32 = ru32le;
  for await (let chunk of pipe.output) {
    enforce(chunk instanceof Buffer);
    if (nsec_multiplier === null) {
      rem = Buffer.concat([rem, chunk]);
      if (rem.length >= PCAP_HEADER_SIZE) {
        const [mult, header] = parse_pcap_header(chunk.subarray(0, PCAP_HEADER_SIZE));
        nsec_multiplier = mult;
        endianness = header.endianness;
        ru32 = endianness === "LE" ? ru32le : ru32be;
        rem = chunk.subarray(PCAP_HEADER_SIZE);
      }
    } else {
      const peek_u32 = (relative_offset) => {
        const o = relative_offset + consumed;
        if (rem.length >= o + 4) {
          return ru32(rem, o);
        } else if (o >= rem.length) {
          return ru32(chunk, o - rem.length);
        } else {
          const n_tail = 4 - (rem.length - o);
          enforce(n_tail > 0 && n_tail <= 4);
          let result = 0;
          switch (endianness) {
            case "BE":
              for (let i = 0; i < 4 - n_tail; ++i) {
                result = result * 256 + rem[o + i];
              }
              for (let i = 0; i < n_tail; ++i) {
                result = result * 256 + chunk[i];
              }
              break;
            case "LE":
              for (let i = n_tail - 1; i >= 0; --i) {
                result = result * 256 + chunk[i];
              }
              for (let i = 3 - n_tail; i >= 0; --i) {
                result = result * 256 + rem[o + i];
              }
              break;
          }
          return result;
        }
      };
      let consumed = 0;
      while (rem.length + chunk.length > 16 + consumed) {
        const ts_sec = peek_u32(0);
        const ts_nsec = peek_u32(4) * nsec_multiplier;
        const incl_len = peek_u32(8);
        enforce(incl_len < 10000);
        if (rem.length + chunk.length < consumed + 16 + incl_len) {
          break;
        }
        if (rem.length >= consumed + 16 + incl_len) {
          const data = rem.subarray(consumed + 16, consumed + 16 + incl_len);
          consumed += 16 + incl_len;
          yield { frame_number: frame_number++, ts_sec, ts_nsec, data };
        } else if (consumed + 16 >= rem.length) {
          const data = chunk.subarray(
            consumed + 16 - rem.length,
            consumed + 16 - rem.length + incl_len,
          );
          consumed = consumed + 16 - rem.length + incl_len;
          rem = chunk;
          chunk = zerobuf;
          yield { frame_number: frame_number++, ts_sec, ts_nsec, data };
        } else {
          const n_tail = incl_len - (rem.length - consumed - 16);
          enforce(chunk.length >= n_tail);
          const data = Buffer.concat([
            rem.subarray(consumed + 16, rem.length),
            chunk.subarray(0, n_tail),
          ]);
          rem = chunk;
          chunk = zerobuf;
          consumed = n_tail;
          yield { frame_number: frame_number++, ts_sec, ts_nsec, data };
        }
      }
      rem = rem.subarray(consumed);
    }
  }
}
export async function* iter_pcap_ipv4(pars) {
  for await (const { frame_number, ts_sec, ts_nsec, data } of iter_pcap(pars)) {
    if (data.length <= 14) continue;
    const ethtype = ru16be(data, 12);
    const ip_layer = data.subarray(14 + (ethtype === 0x8100 ? 2 : 0));
    switch (ethtype) {
      case 0x0800:
        break;
      case 0x8100:
        if (ru16be(data, 14) !== 0x0800) continue;
        break;
      default:
        continue;
    }
    if (ip_layer[0] >> 4 !== 4) continue;
    const ip_header_len = 4 * (ip_layer[0] & 0xf);
    const proto = ip_layer[9];
    const ip_src = ru32be(ip_layer, 12);
    const ip_dst = ru32be(ip_layer, 16);
    const ip_payload = ip_layer.subarray(ip_header_len);
    yield { frame_number, ts_sec, ts_nsec, ip_src, ip_dst, ip_payload, proto };
  }
}
export async function* iter_pcap_udp(pars) {
  for await (const {
    frame_number,
    ts_sec,
    ts_nsec,
    ip_src,
    ip_dst,
    proto,
    ip_payload,
  } of iter_pcap_ipv4(pars)) {
    if (proto !== 17) continue;
    const udp_src_port = ru16be(ip_payload, 0);
    const udp_dst_port = ru16be(ip_payload, 2);
    const udp_len = ru16be(ip_payload, 4);
    const udp_payload = ip_payload.subarray(8, udp_len);
    yield {
      frame_number,
      ts_sec,
      ts_nsec,
      ip_dst,
      ip_src,
      udp_dst_port,
      udp_src_port,
      udp_payload,
    };
  }
}
export async function* iter_pcap_rtp(pars) {
  for await (const {
    frame_number,
    ts_sec,
    ts_nsec,
    ip_src,
    ip_dst,
    udp_src_port,
    udp_dst_port,
    udp_payload,
  } of iter_pcap_udp(pars)) {
    if (udp_payload[0] >> 6 !== 2) continue;
    const rtp_marker = udp_payload[1] >> 7 === 1;
    const rtp_pt = udp_payload[1] & 0x7f;
    const rtp_seq = ru16be(udp_payload, 2);
    const rtp_timestamp = ru32be(udp_payload, 4);
    const rtp_ssrc = ru32be(udp_payload, 8);
    const bytes_to_skip = 4 * ((!!(udp_payload[0] & 16) ? 1 : 0) + (udp_payload[0] & 0xf));
    const rtp_payload = udp_payload.subarray(12 + bytes_to_skip);
    yield {
      frame_number,
      ts_sec,
      ts_nsec,
      ip_dst,
      ip_src,
      udp_dst_port,
      udp_src_port,
      rtp_marker,
      rtp_pt,
      rtp_payload,
      rtp_seq,
      rtp_ssrc,
      rtp_timestamp,
    };
  }
}
export async function* iter_pcap_rtp_inorder(pars) {
  const max_reorder = pars.max_reorder ?? 128;
  let expected_seqnr = null;
  const reorder_buffer = new Map();
  for await (const pkt of iter_pcap_rtp(pars)) {
    if (!pars.filter(pkt)) continue;
    if (expected_seqnr === null) expected_seqnr = pkt.rtp_seq;
    if (pkt.rtp_seq === expected_seqnr || expected_seqnr === null) {
      yield { missing: false, pkt };
      expected_seqnr = (pkt.rtp_seq + 1) & 0xffff;
    } else {
      reorder_buffer.set(pkt.rtp_seq, pkt);
      while (reorder_buffer.size > max_reorder) {
        const maybe_pkt = reorder_buffer.get(expected_seqnr);
        if (maybe_pkt) {
          yield { missing: false, pkt: maybe_pkt };
          reorder_buffer.delete(expected_seqnr);
        } else yield { missing: true, rtp_seq: expected_seqnr };
        expected_seqnr = (expected_seqnr + 1) & 0xffff;
      }
    }
  }
  while (reorder_buffer.size > 0) {
    enforce(expected_seqnr !== null);
    const maybe_pkt = reorder_buffer.get(expected_seqnr);
    if (maybe_pkt) {
      yield { missing: false, pkt: maybe_pkt };
      reorder_buffer.delete(expected_seqnr);
    } else yield { missing: true, rtp_seq: expected_seqnr };
    expected_seqnr = (expected_seqnr + 1) & 0xffff;
  }
}
export function num_of_ipv4(addr) {
  const parts = addr.split(".").map((s) => parseInt(s));
  if (parts.length !== 4 || parts.some((p) => isNaN(p)))
    throw new Error(`Unable to interpret ${addr} as an IPv4 address`);
  return (
    parts[0] * Math.pow(2, 24) + parts[1] * Math.pow(2, 16) + parts[2] * Math.pow(2, 8) + parts[3]
  );
}
