import * as VAPI from "vapi";
import { closeSync, openSync, writeSync } from "fs";
import { ClockTracker } from "./clock_tracker.js";
import { Duration, Reflection, enforce } from "vscript";
export class Snooper {
  output_path;
  sponges;
  listeners = [];
  fd;
  clockTrackers = new Map();
  constructor(output_path, sponges) {
    this.output_path = output_path;
    this.sponges = sponges;
    this.fd = openSync(this.output_path, "w");
  }
  async addSocket(vsocket) {
    let i = 0;
    enforce(!this.clockTrackers.has(vsocket.ip));
    this.clockTrackers.set(vsocket.ip, await ClockTracker.create(VAPI.VM.adopt(vsocket)));
    if (this.sponges.length !== 0) {
      await Reflection.keyword_iter({
        backing_store: vsocket,
        handler: async (_sock, desc, kwl) => {
          let inner = desc.contents;
          while (inner.container_type !== 0) {
            inner = inner.contents;
          }
          const kw = inner.sys_name;
          for (const sponge of this.sponges) {
            if (`${kwl}.${kw}`.match(sponge)) {
              this.listeners.push(
                vsocket.register_kw_listener(
                  {
                    kwl,
                    kw,
                    listener_type: 0,
                    execution_strategy: 1,
                  },
                  () => {},
                ),
              );
              if (i++ % 10000 === 0) {
                await vsocket.marker({ timeout: new Duration(15, "s") });
              }
            }
          }
        },
        include: "everything",
      });
      await vsocket.marker({ timeout: new Duration(15, "s") });
    }
    vsocket.set_snooper((requests) => {
      this.log_outbound(vsocket.ip, requests);
    });
    vsocket.register_global_listener((msg) => {
      for (const kw in msg.kw) {
        this.log_inbound(vsocket.ip, msg.kwl, kw, msg.kw[kw]);
      }
    });
  }
  log_inbound(ip, kwl, kw, payload) {
    const now = new Date();
    writeSync(
      this.fd,
      JSON.stringify(
        {
          type: "inbound",
          ip,
          t: now.toString(),
          approx_ptp: this.clockTrackers.get(ip).to_ptp(now),
          kwl,
          kw,
          payload,
        },
        null,
        2,
      ) + ",\n",
    );
  }
  log_outbound(ip, requests) {
    const now = new Date();
    for (const request of requests) {
      writeSync(
        this.fd,
        JSON.stringify(
          {
            type: "outbound",
            ip,
            t: now.toString(),
            approx_ptp: this.clockTrackers.get(ip).to_ptp(now),
            request,
          },
          null,
          2,
        ) + ",\n",
      );
    }
  }
  inject(payload) {
    writeSync(
      this.fd,
      JSON.stringify({ type: "injected", t: new Date().toString(), payload }, null, 2) + ",\n",
    );
  }
  end() {
    closeSync(this.fd);
    this.fd = -1;
  }
}
