import { spawn } from "child_process";
import { mkdtempSync, writeFileSync } from "fs";
import { join } from "path";
import { enforce, enforce_nonnull } from "vscript";
export class RTPListener {
  m_procs = [];
  constructor(pars) {
    const socat_invocations = new Set();
    const ports = new Set();
    for (const stream of pars.streams) {
      const raw_dst = stream.dst_ip.replace(/:[0-9]+/, "");
      socat_invocations.add(
        `socat STDIO UDP4-DATAGRAM:${raw_dst}:${stream.dst_port},ip-add-membership=${raw_dst}:${stream.src_ip.replace(/:[0-9]+/, "")}:${pars.ifc}`,
      );
      ports.add(stream.dst_port);
    }
    const tmpdir = mkdtempSync("./");
    writeFileSync(
      join(tmpdir, "decode_as_entries"),
      [...ports].map((port) => `decode_as_entry: udp.port,${port},(none),RTP`).join("\n"),
    );
    const fields = ["frame.time_relative", "ip.src", "ip.dst", "udp.dstport", ...pars.fields];
    let tshark_cmd = `tshark -i ${pars.ifc} -f udp -T json`;
    for (const field of fields) {
      tshark_cmd += ` -e ${field}`;
    }
    for (const arg of pars.extra_tshark_args ?? []) {
      tshark_cmd += ` ${arg.trim()}`;
    }
    let overlap = "";
    let prev_nesting_level = 0;
    const tshark_process = spawn(tshark_cmd, {
      env: { WIRESHARK_CONFIG_DIR: tmpdir },
      shell: true,
    });
    tshark_process.stdout.setEncoding("utf8");
    tshark_process.stdout.on("data", function (data) {
      let buf = overlap;
      let cur_nesting_level = prev_nesting_level;
      let i_start = 0;
      let i = 0;
      for (; i < data.length; ++i) {
        switch (data[i]) {
          case "{":
            if (cur_nesting_level++ === 0) {
              i_start = i;
            }
            break;
          case "}":
            if (--cur_nesting_level === 0) {
              buf += data.substring(i_start, i + 1);
              const dict = {};
              const raw = JSON.parse(buf)["_source"]["layers"];
              for (const k in raw) {
                dict[k] = raw[k][0];
              }
              pars.callback(dict);
              buf = "";
              i++;
              enforce(i === data.length || data[i] === ",");
              i_start = i + 1;
            }
            break;
          default:
            break;
        }
      }
      enforce(
        i_start >= data.length ||
          data[i_start] === "{" ||
          enforce_nonnull(data[i_start]).trim() === "",
      );
      overlap = data.substring(i_start);
      prev_nesting_level = cur_nesting_level;
    });
    this.m_procs.push(tshark_process);
    for (const invocation of socat_invocations) {
      this.m_procs.push(spawn(invocation, { shell: true }));
    }
  }
  close() {
    for (const proc of this.m_procs) {
      proc.kill();
    }
    this.m_procs = [];
  }
}
export class ST2110_40_Listener {
  m_rtp_listener;
  constructor(pars) {
    this.m_rtp_listener = new RTPListener({
      ...pars,
      extra_tshark_args: [
        `-o st_2110_40.dyn_pt:${pars.rtp_payload} `,
        `-X lua_script:data/ST2110_40_Dissector/ST2110-40.lua`,
      ],
      fields: [
        "rtp.seq",
        "rtp.marker",
        "st_2110_40.DID",
        "st_2110_40.SDID",
        "st_2110_40.Line_Number",
        "st_2110_40.Data_Count",
        "st_2110_40.UDW_array",
        "st_2110_40.Data.TimeCode",
        "st_2110_40.Data.TimeCodePT",
        "st_2110_40.F",
      ],
      callback: (response) => {
        const rtp_seq = response["rtp.seq"];
        if (rtp_seq === undefined) return;
        const rtp_seqnr = parseInt(rtp_seq, 10);
        const time_relative = parseFloat(enforce_nonnull(response["frame.time_relative"]));
        const src_ip = enforce_nonnull(response["ip.src"]);
        const dst_ip = enforce_nonnull(response["ip.dst"]);
        const dst_port = parseInt(enforce_nonnull(response["udp.dstport"]), 10);
        switch (response["rtp.marker"]) {
          case "1":
            pars.callback({
              time_relative,
              src_ip,
              dst_ip,
              dst_port,
              rtp_seqnr,
              rtp_marker: true,
            });
            break;
          case "0":
            pars.callback({
              time_relative,
              src_ip,
              dst_ip,
              dst_port,
              rtp_seqnr,
              rtp_marker: false,
              did: parseInt(enforce_nonnull(response["st_2110_40.DID"])),
              sdid: parseInt(enforce_nonnull(response["st_2110_40.SDID"])),
              line_number: parseInt(enforce_nonnull(response["st_2110_40.Line_Number"])),
              dc: parseInt(enforce_nonnull(response["st_2110_40.Data_Count"])),
              udw_array: enforce_nonnull(response["st_2110_40.UDW_array"]),
              time_code: response["st_2110_40.Data.TimeCode"]
                ? {
                    payload_type: parseInt(enforce_nonnull(response["st_2110_40.Data.TimeCodePT"])),
                    as_string: response["st_2110_40.Data.TimeCode"],
                  }
                : null,
              F: parseInt(enforce_nonnull(response["st_2110_40.F"])),
            });
        }
      },
    });
  }
  close() {
    this.m_rtp_listener.close();
  }
}
export class ST2110_40_Listeners {
  m_listeners = [];
  constructor(pars) {
    const payloads = new Set(pars.streams.map((s) => s.rtp_payload));
    for (const rtp_payload of payloads) {
      this.m_listeners.push(
        new ST2110_40_Listener({
          ...pars,
          rtp_payload,
          streams: pars.streams.filter((s) => s.rtp_payload === rtp_payload),
        }),
      );
    }
  }
  close() {
    this.m_listeners.forEach((l) => l.close());
    this.m_listeners = [];
  }
}
